"use strict";
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const lib_2 = require("../lib");
module.exports = {
    'the Permission class is a programming model for iam'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAction('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResource('myQueue');
        p.addResource('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${new cdk_1.Token({ account: 'account' })}name`);
        p.limitToAccount('12221121221');
        test.deepEqual(stack.node.resolve(p), { Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: { AWS: { 'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']] } },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } } });
        test.done();
    },
    'the PolicyDocument class is a dom for iam policy documents'(test) {
        const stack = new cdk_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addAction('sqs:SendMessage');
        p1.addResource('*');
        const p2 = new lib_1.PolicyStatement();
        p2.deny();
        p2.addActions('cloudformation:CreateStack');
        doc.addStatement(p1);
        doc.addStatement(p2);
        test.deepEqual(stack.node.resolve(doc), {
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', Resource: '*' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' }]
        });
        test.done();
    },
    'A PolicyDocument can be initialized with an existing policy, which is merged upon serialization'(test) {
        const stack = new cdk_1.Stack();
        const base = {
            Version: 'Foo',
            Something: 123,
            Statement: [
                { Statement1: 1 },
                { Statement2: 2 }
            ]
        };
        const doc = new lib_1.PolicyDocument(base);
        doc.addStatement(new lib_1.PolicyStatement().addResource('resource').addAction('action'));
        test.deepEqual(stack.node.resolve(doc), { Version: 'Foo',
            Something: 123,
            Statement: [{ Statement1: 1 },
                { Statement2: 2 },
                { Effect: 'Allow', Action: 'action', Resource: 'resource' }] });
        test.done();
    },
    'Permission allows specifying multiple actions upon construction'(test) {
        const stack = new cdk_1.Stack();
        const perm = new lib_1.PolicyStatement().addResource('MyResource').addActions('Action1', 'Action2', 'Action3');
        test.deepEqual(stack.node.resolve(perm), {
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource'
        });
        test.done();
    },
    'PolicyDoc resolves to undefined if there are no permissions'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        test.deepEqual(stack.node.resolve(p), undefined);
        test.done();
    },
    'canonicalUserPrincipal adds a principal to a policy with the passed canonical user id'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = "averysuperduperlongstringfor";
        p.addPrincipal(new lib_1.CanonicalUserPrincipal(canoncialUser));
        test.deepEqual(stack.node.resolve(p), {
            Effect: "Allow",
            Principal: {
                CanonicalUser: canoncialUser
            }
        });
        test.done();
    },
    'addAccountRootPrincipal adds a principal with the current account root'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        test.deepEqual(stack.node.resolve(p), {
            Effect: "Allow",
            Principal: {
                AWS: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            { Ref: "AWS::Partition" },
                            ":iam::",
                            { Ref: "AWS::AccountId" },
                            ":root"
                        ]
                    ]
                }
            }
        });
        test.done();
    },
    'addFederatedPrincipal adds a Federated principal with the passed value'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal("com.amazon.cognito", { StringEquals: { key: 'value' } });
        test.deepEqual(stack.node.resolve(p), {
            Effect: "Allow",
            Principal: {
                Federated: "com.amazon.cognito"
            },
            Condition: {
                StringEquals: { key: 'value' }
            }
        });
        test.done();
    },
    'addAwsAccountPrincipal can be used multiple times'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        test.deepEqual(stack.node.resolve(p), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] }
                ]
            }
        });
        test.done();
    },
    'hasResource': {
        'false if there are no resources'(test) {
            test.equal(new lib_1.PolicyStatement().hasResource, false, 'hasResource should be false for an empty permission');
            test.done();
        },
        'true if there is one resource'(test) {
            test.equal(new lib_1.PolicyStatement().addResource('one-resource').hasResource, true, 'hasResource is true when there is one resource');
            test.done();
        },
        'true for multiple resources'(test) {
            const p = new lib_1.PolicyStatement();
            p.addResource('r1');
            p.addResource('r2');
            test.equal(p.hasResource, true, 'hasResource is true when there are multiple resource');
            test.done();
        },
    },
    'hasPrincipal': {
        'false if there is no principal'(test) {
            test.equal(new lib_1.PolicyStatement().hasPrincipal, false);
            test.done();
        },
        'true if there is a principal'(test) {
            const p = new lib_1.PolicyStatement();
            p.addAwsPrincipal('bla');
            test.equal(p.hasPrincipal, true);
            test.done();
        }
    },
    'statementCount returns the number of statement in the policy document'(test) {
        const p = new lib_1.PolicyDocument();
        test.equal(p.statementCount, 0);
        p.addStatement(new lib_1.PolicyStatement().addAction('action1'));
        test.equal(p.statementCount, 1);
        p.addStatement(new lib_1.PolicyStatement().addAction('action2'));
        test.equal(p.statementCount, 2);
        test.done();
    },
    '{ AWS: "*" } principal': {
        'is represented as `Anyone`'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_1.Anyone()));
            test.deepEqual(stack.node.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        },
        'is represented as `AnyPrincipal`'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_1.AnyPrincipal()));
            test.deepEqual(stack.node.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        },
        'is represented as `addAnyPrincipal`'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatement(new lib_1.PolicyStatement().addAnyPrincipal());
            test.deepEqual(stack.node.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        }
    },
    'addAwsPrincipal/addArnPrincipal are the aliases'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        p.addStatement(new lib_1.PolicyStatement().addAwsPrincipal('111222-A'));
        p.addStatement(new lib_1.PolicyStatement().addArnPrincipal('111222-B'));
        p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_2.ArnPrincipal('111222-C')));
        test.deepEqual(stack.node.resolve(p), {
            Statement: [{
                    Effect: 'Allow', Principal: { AWS: '111222-A' }
                },
                { Effect: 'Allow', Principal: { AWS: '111222-B' } },
                { Effect: 'Allow', Principal: { AWS: '111222-C' } }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addResources() will not break a list-encoded Token'(test) {
        const stack = new cdk_1.Stack();
        const statement = new lib_1.PolicyStatement()
            .addActions(...new cdk_1.Token(() => ['a', 'b', 'c']).toList())
            .addResources(...new cdk_1.Token(() => ['x', 'y', 'z']).toList());
        test.deepEqual(stack.node.resolve(statement), {
            Effect: 'Allow',
            Action: ['a', 'b', 'c'],
            Resource: ['x', 'y', 'z'],
        });
        test.done();
    },
    'addCanonicalUserPrincipal can be used to add cannonical user principals'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        p.addStatement(new lib_1.PolicyStatement().addCanonicalUserPrincipal('cannonical-user-1'));
        p.addStatement(new lib_1.PolicyStatement().addPrincipal(new lib_1.CanonicalUserPrincipal('cannonical-user-2')));
        test.deepEqual(stack.node.resolve(p), {
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addPrincipal correctly merges array in'(test) {
        const stack = new cdk_1.Stack();
        const arrayPrincipal = {
            get grantPrincipal() { return this; },
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: new lib_2.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
            addToPolicy() { return false; }
        };
        const s = new lib_1.PolicyStatement().addAccountRootPrincipal()
            .addPrincipal(arrayPrincipal);
        test.deepEqual(stack.node.resolve(s), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar'
                ]
            }
        });
        test.done();
    },
    // https://github.com/awslabs/aws-cdk/issues/1201
    'policy statements with multiple principal types can be created using multiple addPrincipal calls'(test) {
        const stack = new cdk_1.Stack();
        const s = new lib_1.PolicyStatement()
            .addAwsPrincipal('349494949494')
            .addServicePrincipal('test.service')
            .addResource('resource')
            .addAction('action');
        test.deepEqual(stack.node.resolve(s), {
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource'
        });
        test.done();
    },
    'Service principals': {
        'regional service principals resolve appropriately'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement()
                .addAction('test:Action')
                .addServicePrincipal('codedeploy.amazonaws.com');
            test.deepEqual(stack.node.resolve(s), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'regional service principals resolve appropriately (with user-set region)'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement()
                .addAction('test:Action')
                .addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            test.deepEqual(stack.node.resolve(s), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'obscure service principals resolve to the user-provided value'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement()
                .addAction('test:Action')
                .addServicePrincipal('test.service-principal.dev');
            test.deepEqual(stack.node.resolve(s), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' }
            });
            test.done();
        },
    },
    'CompositePrincipal can be used to represent a principal that has multiple types': {
        'with a single principal'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement().addPrincipal(p);
            test.deepEqual(stack.node.resolve(statement), { Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
            test.done();
        },
        'conditions are not allowed on individual principals of a composite'(test) {
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am'));
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('federated', { condition: 1 })), /Components of a CompositePrincipal must not have conditions/);
            test.done();
        },
        'principals and conditions are a big nice merge'(test) {
            const stack = new cdk_1.Stack();
            // add via ctor
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'), new lib_2.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_2.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement().addPrincipal(p);
            // add via policy statement
            statement.addAwsPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: 123 });
            test.deepEqual(stack.node.resolve(statement), {
                Condition: {
                    cond2: { boom: 123 }
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                }
            });
            test.done();
        },
        'cannot mix types of assumeRoleAction in a single composite'(test) {
            // GIVEN
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('fed', {}, 'sts:Boom')), /Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
            test.done();
        }
    },
    'duplicate statements': {
        'without tokens'(test) {
            // GIVEN
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement = new lib_1.PolicyStatement()
                .addResources('resource1', 'resource2')
                .addActions('action1', 'action2')
                .addServicePrincipal('service')
                .addConditions({
                a: {
                    b: 'c'
                },
                d: {
                    e: 'f'
                }
            });
            // WHEN
            p.addStatement(statement);
            p.addStatement(statement);
            p.addStatement(statement);
            // THEN
            test.equal(stack.node.resolve(p).Statement.length, 1);
            test.done();
        },
        'with tokens'(test) {
            // GIVEN
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement1 = new lib_1.PolicyStatement()
                .addResource(new cdk_1.Token(() => 'resource').toString())
                .addAction(new cdk_1.Token(() => 'action').toString());
            const statement2 = new lib_1.PolicyStatement()
                .addResource(new cdk_1.Token(() => 'resource').toString())
                .addAction(new cdk_1.Token(() => 'action').toString());
            // WHEN
            p.addStatement(statement1);
            p.addStatement(statement2);
            // THEN
            test.equal(stack.node.resolve(p).Statement.length, 1);
            test.done();
        },
        'with base document'(test) {
            // GIVEN
            const stack = new cdk_1.Stack();
            // WHEN
            const p = new lib_1.PolicyDocument({
                Statement: [
                    {
                        Action: 'action',
                        Effect: 'Allow',
                        Resource: 'resource'
                    },
                    {
                        Action: 'action',
                        Effect: 'Allow',
                        Resource: 'resource'
                    }
                ]
            });
            p.addStatement(new lib_1.PolicyStatement()
                .addAction('action')
                .addResource('resource'));
            // THEN
            test.equal(stack.node.resolve(p).Statement.length, 1);
            test.done();
        }
    },
    'autoAssignSids enables auto-assignment of a unique SID for each statement'(test) {
        // GIVEN
        const doc = new lib_1.PolicyDocument();
        doc.addStatement(new lib_1.PolicyStatement().addAction('action1').addResource('resource1'));
        doc.addStatement(new lib_1.PolicyStatement().addAction('action1').addResource('resource1'));
        doc.addStatement(new lib_1.PolicyStatement().addAction('action1').addResource('resource1'));
        doc.addStatement(new lib_1.PolicyStatement().addAction('action1').addResource('resource1'));
        doc.addStatement(new lib_1.PolicyStatement().addAction('action2').addResource('resource2'));
        // WHEN
        doc.autoAssignSids();
        // THEN
        const stack = new cdk_1.Stack();
        test.deepEqual(stack.node.resolve(doc), {
            Version: '2012-10-17',
            Statement: [
                { Action: 'action1', Effect: 'Allow', Resource: 'resource1', Sid: '0' },
                { Action: 'action2', Effect: 'Allow', Resource: 'resource2', Sid: '1' }
            ],
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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