"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
module.exports = {
    'fails when policy is empty'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.run(), /Policy is empty/);
        test.done();
    },
    'policy with statements'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('sqs:SendMessage'));
        policy.addStatement(new lib_1.PolicyStatement().addResource('arn').addAction('sns:Subscribe'));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicyName' } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' } } });
        test.done();
    },
    'policy name can be omitted, in which case the logical id will be used'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('sqs:SendMessage'));
        policy.addStatement(new lib_1.PolicyStatement().addResource('arn').addAction('sns:Subscribe'));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'policy can be attached users, groups and roles and added permissions via props'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement().addResource('*').addAction('dynamodb:PutItem')],
        });
        assert_1.expect(stack).toMatch({ Resources: { User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } },
                MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }] } } } });
        test.done();
    },
    'idempotent if a principal (user/group/role) is attached twice'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatement(new lib_1.PolicyStatement().addAction('*').addResource('*'));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'users, groups, roles and permissions can be added using methods'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('dynamodb:GetItem'));
        assert_1.expect(stack).toMatch({ Resources: { MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }] } },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'policy can be attached to users, groups or role via methods on the principal'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatement(new lib_1.PolicyStatement().addResource('*').addAction('*'));
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'fails if policy name is not unique within a user/group/role'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        test.throws(() => p2.attachToUser(user), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToGroup(group), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToRole(role), /A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
        test.done();
    },
    'fails if policy is not attached to a principal'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.run(), /Policy must be attached to at least one principal: user, group or role/);
        test.done();
    },
    'generated policy name only uses the last 128 characters of the logical id'(test) {
        test.equal(util_1.generatePolicyName('Foo'), 'Foo');
        const logicalId50 = '[' + dup(50 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId50), logicalId50);
        const logicalId128 = '[' + dup(128 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId128), logicalId128);
        const withPrefix = 'PREFIX' + logicalId128;
        test.equal(util_1.generatePolicyName(withPrefix), logicalId128, 'ensure prefix is omitted');
        test.done();
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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