"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'default role'(test) {
        const stack = new cdk_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com')
        });
        assert_1.expect(stack).toMatch({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'a role can grant PassRole permissions'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const role = new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.ServicePrincipal('henk.amazonaws.com') });
        const user = new lib_1.User(stack, 'User');
        // WHEN
        role.grantPassRole(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "iam:PassRole",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Role1ABCC5F0", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'can supply externalId'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalId: 'SomeSecret',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": "SomeSecret" }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'policy is created automatically when permissions are added'(test) {
        // by default we don't expect a role policy
        const before = new cdk_1.Stack();
        new lib_1.Role(before, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        assert_1.expect(before).notTo(assert_1.haveResource('AWS::IAM::Policy'));
        // add a policy to the role
        const after = new cdk_1.Stack();
        const afterRole = new lib_1.Role(after, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        afterRole.addToPolicy(new lib_1.PolicyStatement().addResource('myresource').addAction('myaction'));
        assert_1.expect(after).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "myaction",
                        Effect: "Allow",
                        Resource: "myresource"
                    }
                ],
                Version: "2012-10-17"
            },
            PolicyName: "MyRoleDefaultPolicyA36BE1DD",
            Roles: [
                {
                    Ref: "MyRoleF48FFE04"
                }
            ]
        }));
        test.done();
    },
    'managed policy arns can be supplied upon initialization and also added later'(test) {
        const stack = new cdk_1.Stack();
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
            managedPolicyArns: ['managed1', 'managed2']
        });
        role.attachManagedPolicy('managed3');
        assert_1.expect(stack).toMatch({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: ['managed1', 'managed2', 'managed3'] } } } });
        test.done();
    },
    'federated principal can change AssumeRoleAction'(test) {
        const stack = new cdk_1.Stack();
        const cognitoPrincipal = new lib_1.FederatedPrincipal('foo', { StringEquals: { key: 'value' } }, 'sts:AssumeSomething');
        new lib_1.Role(stack, 'MyRole', { assumedBy: cognitoPrincipal });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Version: "2012-10-17",
                Statement: [
                    {
                        Principal: { Federated: "foo" },
                        Condition: {
                            StringEquals: { key: "value" }
                        },
                        Action: "sts:AssumeSomething",
                        Effect: "Allow",
                    }
                ],
            }
        }));
        test.done();
    },
    'maxSessionDuration': {
        'is not specified by default'(test) {
            const stack = new cdk_1.Stack();
            new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            assert_1.expect(stack).toMatch({
                Resources: {
                    MyRoleF48FFE04: {
                        Type: "AWS::IAM::Role",
                        Properties: {
                            AssumeRolePolicyDocument: {
                                Statement: [
                                    {
                                        Action: "sts:AssumeRole",
                                        Effect: "Allow",
                                        Principal: {
                                            Service: "sns.amazonaws.com"
                                        }
                                    }
                                ],
                                Version: "2012-10-17"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'can be used to specify the maximum session duration for assuming the role'(test) {
            const stack = new cdk_1.Stack();
            new lib_1.Role(stack, 'MyRole', { maxSessionDurationSec: 3700, assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                MaxSessionDuration: 3700
            }));
            test.done();
        },
        'must be between 3600 and 43200'(test) {
            const stack = new cdk_1.Stack();
            const assumedBy = new lib_1.ServicePrincipal('bla');
            new lib_1.Role(stack, 'MyRole1', { assumedBy, maxSessionDurationSec: 3600 });
            new lib_1.Role(stack, 'MyRole2', { assumedBy, maxSessionDurationSec: 43200 });
            const expected = (val) => `maxSessionDuration is set to ${val}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`;
            test.throws(() => new lib_1.Role(stack, 'MyRole3', { assumedBy, maxSessionDurationSec: 60 }), expected(60));
            test.throws(() => new lib_1.Role(stack, 'MyRole4', { assumedBy, maxSessionDurationSec: 3599 }), expected(3599));
            test.throws(() => new lib_1.Role(stack, 'MyRole5', { assumedBy, maxSessionDurationSec: 43201 }), expected(43201));
            test.done();
        }
    },
    'allow role with multiple principals'(test) {
        const stack = new cdk_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.CompositePrincipal(new lib_1.ServicePrincipal('boom.amazonaws.test'), new lib_1.ArnPrincipal('1111111'))
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "boom.amazonaws.test",
                            AWS: "1111111"
                        }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'fromRoleArn'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const importedRole = lib_1.Role.fromRoleArn(stack, 'ImportedRole', 'arn:aws:iam::123456789012:role/S3Access');
        // THEN
        test.deepEqual(importedRole.roleArn, 'arn:aws:iam::123456789012:role/S3Access');
        test.deepEqual(importedRole.roleName, 'S3Access');
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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