import logging
import os
from contextlib import contextmanager
from posixpath import join as url_join
from typing import Iterator

import boto3
import smart_open as smart
from botocore.credentials import RefreshableCredentials
from botocore.session import get_session
from typing_extensions import Self

from emporium.base import Entry, NoSuchFile, Store
from emporium.utils import is_write_mode, select_keys

log = logging.getLogger(__name__)


class RemoteStoreS3(Store):
    """Store that puts data on S3."""

    DEFAULT_DOMAIN = "{bucket}.s3-eu-west-1.amazonaws.com"

    def __init__(self, bucket, prefix=None, domain=None, **extra):
        """Instantiate a store.

        :param bucket: The bucket to store data in.
        :param prefix: The prefix within that bucket, under which to store data.
        :param extra: Additional configuration passed to the boto3 session
            constructor.  This might include, ``aws_access_key_id``, ``aws_secret_access_key``
            and ``role_arn``.

        :returns: Instance of a subclass of :class:`~emporium.base.Store`.
        """

        self._bucket = bucket
        self._prefix = None if not prefix else prefix.lstrip("/")
        self._domain = domain or self.DEFAULT_DOMAIN.format(bucket=bucket)
        self._extra = self._extract_config(extra)

    @classmethod
    def from_config(cls, config):
        return cls(**config)

    @contextmanager
    def open(self, path, mode, *args, **kwargs):
        uri = self._uri(path)
        transport_params = self._create_transport_params()
        transport_params.update(kwargs.get("transport_params", {}))
        if transport_params:
            kwargs["transport_params"] = transport_params
        acl = kwargs.pop("ACL", None)
        try:
            with smart.open(uri, mode, *args, **kwargs) as handle:
                yield handle
        except ValueError as exc:
            raise NoSuchFile(path, exc)
        if is_write_mode(mode) and acl is not None:
            client = boto3.client("s3")
            key = self._key(path)
            client.put_object_acl(Bucket=self._bucket, Key=key, ACL=acl)

    def substore(self, path) -> Self:
        path = path.lstrip("/")
        subpath = url_join(self._prefix, path) if self._prefix is not None else path
        return self.__class__(self._bucket, subpath, self._domain, **self._extra)

    def location(self, path=None) -> str:
        segments = [s.lstrip("/") for s in [self._prefix, path] if s]
        if segments:
            return "https://{}/{}".format(self._domain, url_join(*segments))
        return "https://{}".format(self._domain)

    def list(self, path=None) -> Iterator[Entry]:
        client = boto3.client("s3")
        segments = [s for s in [self._prefix, path] if s is not None]
        if segments:
            prefix = url_join(*segments)
        else:
            prefix = None
        entries = self._get_entries_list(client, prefix)
        seen = set()
        for entry in entries:
            relative_entry = os.path.relpath(entry, prefix)
            if len(relative_entry.split(os.sep)) > 1 or entry.endswith(os.sep):
                entry_tuple = Entry(relative_entry.split(os.sep)[0], "directory")
            else:
                entry_tuple = Entry(relative_entry.split(os.sep)[0], "file")
            if entry_tuple not in seen:
                yield entry_tuple
                seen.add(entry_tuple)

    def _get_entries_list(self, client, prefix=None):
        paginator = client.get_paginator("list_objects_v2")
        page_iterator = paginator.paginate(
            Bucket=self._bucket,
            Prefix=prefix or "",
            PaginationConfig={"PageSize": 1000},
        )
        for page in page_iterator:
            for entry in page["Contents"]:
                yield entry["Key"]

    @classmethod
    def _extract_config(cls, extra):
        if not extra:
            return {}
        restriction = ["aws_access_key_id", "aws_secret_access_key", "role_arn"]
        return select_keys(extra, *restriction)

    def _create_transport_params(self):
        if self._extra:
            if "role_arn" in self._extra:
                return {"session": SessionWithAssumedRole(**self._extra).get_session()}
            return {"session": boto3.Session(**self._extra)}
        return {}

    def _key(self, path):
        segments = [s.lstrip("/") for s in [self._prefix, path] if s]
        return url_join(*segments)

    def _uri(self, path=None):
        segments = [s.lstrip("/") for s in [self._bucket, self._prefix, path] if s]
        return "s3://{}".format(url_join(*segments))

    def __repr__(self):
        if self._domain is None:
            return "<{}({})>".format(self.__class__.__name__, self._uri())
        return "<{}({})>".format(self.__class__.__name__, self.location())

    def remove(self, path: str) -> None:
        raise NotImplementedError("Remove is not supported for S3 stores")


class SessionWithAssumedRole:
    def __init__(self, role_arn, **extra):
        self._role_arn = role_arn
        self._session_name = role_arn.split("/")[-1]
        self._extra = self._extract_config(extra)
        self._sts_client = None

    @classmethod
    def _extract_config(cls, extra):
        if not extra:
            return {}
        restriction = ["aws_access_key_id", "aws_secret_access_key"]
        return select_keys(extra, *restriction)

    def get_session(self):
        self._sts_client = boto3.client("sts", **self._extra)
        session_credentials = RefreshableCredentials.create_from_metadata(
            metadata=self._refresh(),
            refresh_using=self._refresh,
            method="sts-assume-role",
        )
        session = get_session()
        # pylint: disable=protected-access
        session._credentials = session_credentials
        autorefresh_session = boto3.Session(botocore_session=session)
        return autorefresh_session

    def _refresh(self):
        log.debug("Refreshing temporary session credentials")
        params = {
            "RoleArn": self._role_arn,
            "RoleSessionName": self._session_name,
            "DurationSeconds": 900,
        }
        response = self._sts_client.assume_role(**params).get("Credentials")  # type: ignore
        credentials = {
            "access_key": response.get("AccessKeyId"),
            "secret_key": response.get("SecretAccessKey"),
            "token": response.get("SessionToken"),
            "expiry_time": response.get("Expiration").isoformat(),
        }
        return credentials

    def __repr__(self):
        return "<{}>".format(self.__class__.__name__)
