# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""
Version 2 of the L-Land model |lland_v2| is a slight modification of
|lland_v1|.  |lland_v1| implements a specific equation for the calculation
of reference evaporation (|ET0|) for each hydrological response unit (HRU).
In contrast, |lland_v2| expects subbasin wide potential evaporation values
(|PET|) to be calculated externally and adjusts them to the different HRUs
of the subbasin.

|lland_v1| should be applied on daily step sized only due to the restrictions
of the Turc-Wendling equation for calculating reference evaporation.
Instead, |lland_v2| can be applied on arbitrary simulation step sizes.

Integration tests:

    The following integration tests are mostly recalculations of the ones
    performed for |lland_v1| in order to show that both models function
    in an equal manner.  Hence, most configurations are identical.
    One exception is that |lland_v2| requires no global radiation input
    (|Glob|).  Instead, potential evaporation needs (|PET|) to be defined,
    which is taken from the integration tests results of model |lland_v1|
    to achieve comparability.  Another exception is that |lland_v1| allows
    to smooth calculated |ET0| values over time, which is discussed below.

    The following general setup is identical with the one of |lland_v1|:

    >>> from hydpy import pub
    >>> pub.timegrids = '01.01.2000', '05.01.2000', '1h'
    >>> from hydpy.models.lland_v2 import *
    >>> parameterstep('1h')
    >>> from hydpy import Node, Element
    >>> outlet = Node('outlet')
    >>> land = Element('land', outlets=outlet)
    >>> land.connect(model)
    >>> nhru(1)
    >>> ft(1.0)
    >>> fhru(1.0)
    >>> from hydpy import IntegrationTest
    >>> IntegrationTest.plotting_options.height = 800
    >>> IntegrationTest.plotting_options.activated=(
    ...     inputs.nied, inputs.teml, fluxes.q)
    >>> test = IntegrationTest(land)
    >>> test.dateformat = '%d.%m.'

    .. _lland_v2_ex1:

    :ref:`Recalculation of example 1 <lland_v1_ex1>`

    The following parameter and initial condition values are identical with
    the ones of |lland_v1|, except that the Turc-Wendling related parameter
    |KF| is missing:

    >>> lnk(ACKER)
    >>> kg(1.2)
    >>> kt(0.8)
    >>> ke(0.4)
    >>> wfet0(1.0)
    >>> fln(0.5)
    >>> hinz(0.2)
    >>> lai(4.0)
    >>> treft(0.0)
    >>> trefn(0.0)
    >>> tgr(1.0)
    >>> tsp(2.0)
    >>> gtf(0.5)
    >>> rschmelz(334.0)
    >>> cpwasser(4.1868)
    >>> pwmax(1.4)
    >>> grasref_r(5.0)
    >>> nfk(200.0)
    >>> relwz(0.5)
    >>> relwb(0.05)
    >>> beta(0.005)
    >>> fbeta(1.0)
    >>> dmax(1.0)
    >>> dmin(0.1)
    >>> bsf(0.4)
    >>> a1(1.0)
    >>> a2(1.0)
    >>> tind(1.0)
    >>> eqb(100.0)
    >>> eqi1(20.0)
    >>> eqi2(10.0)
    >>> eqd1(5.0)
    >>> eqd2(2.0)
    >>> negq(False)
    >>> test.inits = ((states.inzp, 0.0),
    ...               (states.wats, 0.0),
    ...               (states.waes, 0.0),
    ...               (states.bowa, 100.0),
    ...               (states.qdgz1, 0.0),
    ...               (states.qdgz2, 0.0),
    ...               (states.qigz1, 0.0),
    ...               (states.qigz2, 0.0),
    ...               (states.qbgz, 0.5),
    ...               (states.qdga1, 0.0),
    ...               (states.qdga2, 0.0),
    ...               (states.qiga1, 0.0),
    ...               (states.qiga2, 0.0),
    ...               (states.qbga, 0.5),
    ...               (logs.wet0, 0.0))

    The values of the input sequences of |Nied| (precipitation) and |TemL|
    (air temperature) are also taken from the input data of the example on
    |lland_v1|.  But the values of |PET| (potential evaporation) are taken
    from the output data of |lland_v1| (divided by 0.4 to account for the
    set value of the evaporation adjustment factor |KE|):

    >>> inputs.nied.series = (
    ...     0.0, 0.0,  0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    ...     0.0, 0.0, 0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0,
    ...     2.1, 10.4, 3.5, 3.4, 1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1,
    ...     20.7, 37.9, 8.2, 3.6, 7.5, 18.5, 15.4, 6.3, 1.9, 4.9, 2.7, 0.5,
    ...     0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6, 0.7, 0.3, 0.3, 0.0,
    ...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.3, 0.0,
    ...     0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    ...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
    >>> inputs.teml.series = (
    ...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2,
    ...     35.2, 37.1, 31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8,
    ...     19.6, 19.2, 19.2, 19.2, 18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8,
    ...     19.0, 19.2, 19.3, 19.0, 18.8, 18.7, 17.8, 17.4, 17.3, 16.8, 16.5,
    ...     16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3, 14.9, 15.7, 16.0,
    ...     16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3, 18.1,
    ...     16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1,
    ...     11.9, 12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8,
    ...     22.2, 20.1, 17.8, 15.2, 14.5, 12.4, 11.7, 11.9)
    >>> inputs.pet.series = (
    ...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761,
    ...     0.291908, 1.932875, 4.369536, 7.317556, 8.264362, 9.369867,
    ...     5.126178, 6.62503, 7.397619, 2.39151, 1.829834, 1.136569,
    ...     0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
    ...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599,
    ...     0.187298, 1.264612, 3.045538, 1.930758, 2.461001, 6.215945,
    ...     3.374783, 8.821555, 4.046025, 2.110757, 2.239257, 2.877848,
    ...     1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
    ...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593,
    ...     0.154861, 0.470369, 1.173726, 4.202296, 4.359715, 5.305753,
    ...     5.376027, 4.658915, 7.789594, 4.851567, 5.30692, 3.286036,
    ...     1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
    ...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229,
    ...     1.391958, 3.195876, 5.191651, 7.155036, 8.391432, 8.391286,
    ...     10.715238, 9.383394, 7.861915, 6.298329, 2.948416, 1.309232,
    ...     0.32955, 0.089508, 0.085771, 0.0845, 0.084864)


    The following calculation shows, that the outflow values of the
    integration test for arable land (|ACKER|) are reproduced exactly:

    >>> test('lland_v2_ex1')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo |      nbes | sbes |      evi |      evb |      wgtf | schm |      wada |       qdb |     qib1 |     qib2 |      qbb |      qdgz |        q |     inzp | wats | waes |       bowa |    qdgz1 |     qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.040283 | 0.020141 |       0.0 |  0.0 |      0.0 | 0.017301 | 11.275777 |  0.0 |       0.0 |       0.0 |     0.05 |      0.0 |     0.45 |       0.0 |  0.50098 |      0.0 |  0.0 |  0.0 |  99.482699 |      0.0 |       0.0 |     0.05 |      0.0 |     0.45 |      0.0 |      0.0 | 0.001229 |      0.0 | 0.499751 | 0.139161 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.03912 |  0.01956 |       0.0 |  0.0 |      0.0 | 0.016766 | 10.353214 |  0.0 |       0.0 |       0.0 | 0.049741 |      0.0 | 0.447413 |       0.0 | 0.502845 |      0.0 |  0.0 |  0.0 |  98.968779 |      0.0 |       0.0 | 0.049741 |      0.0 | 0.447413 |      0.0 |      0.0 | 0.003602 |      0.0 | 0.499243 | 0.139679 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |       0.0 |  0.0 |      0.0 | 0.016589 | 10.096946 |  0.0 |       0.0 |       0.0 | 0.049484 |      0.0 | 0.444844 |       0.0 |  0.50456 |      0.0 |  0.0 |  0.0 |  98.457861 |      0.0 |       0.0 | 0.049484 |      0.0 | 0.444844 |      0.0 |      0.0 | 0.005846 |      0.0 | 0.498714 | 0.140156 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.038396 | 0.019198 |       0.0 |  0.0 |      0.0 | 0.016383 |  9.789425 |  0.0 |       0.0 |       0.0 | 0.049229 |      0.0 | 0.442289 |       0.0 | 0.506133 |      0.0 |  0.0 |  0.0 |   97.94996 |      0.0 |       0.0 | 0.049229 |      0.0 | 0.442289 |      0.0 |      0.0 | 0.007968 |      0.0 | 0.498166 | 0.140593 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |       0.0 |  0.0 |      0.0 | 0.016516 | 10.096946 |  0.0 |       0.0 |       0.0 | 0.048975 |      0.0 |  0.43975 |       0.0 | 0.507571 |      0.0 |  0.0 |  0.0 |  97.444719 |      0.0 |       0.0 | 0.048975 |      0.0 |  0.43975 |      0.0 |      0.0 | 0.009974 |      0.0 | 0.497597 | 0.140992 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.041104 | 0.020552 |       0.0 |  0.0 |      0.0 | 0.017461 | 11.942073 |  0.0 |       0.0 |       0.0 | 0.048722 |      0.0 | 0.437224 |       0.0 | 0.508878 |      0.0 |  0.0 |  0.0 |  96.941312 |      0.0 |       0.0 | 0.048722 |      0.0 | 0.437224 |      0.0 |      0.0 |  0.01187 |      0.0 | 0.497009 | 0.141355 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.116763 | 0.058382 |       0.0 |  0.0 |      0.0 |  0.04949 | 13.274665 |  0.0 |       0.0 |       0.0 | 0.048471 |      0.0 | 0.434707 |       0.0 | 0.510062 |      0.0 |  0.0 |  0.0 |  96.408645 |      0.0 |       0.0 | 0.048471 |      0.0 | 0.434707 |      0.0 |      0.0 | 0.013661 |      0.0 | 0.496401 | 0.141684 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 |  0.77315 | 0.386575 |       0.0 |  0.0 |      0.0 | 0.326912 | 14.914778 |  0.0 |       0.0 |       0.0 | 0.048204 |      0.0 | 0.432043 |       0.0 | 0.511126 |      0.0 |  0.0 |  0.0 |  95.601485 |      0.0 |       0.0 | 0.048204 |      0.0 | 0.432043 |      0.0 |      0.0 | 0.015352 |      0.0 | 0.495774 | 0.141979 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 1.747814 | 0.873907 |       0.0 |  0.0 |      0.0 | 0.736293 |  14.65851 |  0.0 |       0.0 |       0.0 | 0.047801 |      0.0 | 0.428007 |       0.0 | 0.512064 |      0.0 |  0.0 |  0.0 |  94.389384 |      0.0 |       0.0 | 0.047801 |      0.0 | 0.428007 |      0.0 |      0.0 | 0.016944 |      0.0 |  0.49512 |  0.14224 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 2.927022 | 1.463511 |       0.0 |  0.0 |      0.0 | 1.226019 | 16.503638 |  0.0 |       0.0 |       0.0 | 0.047195 |      0.0 | 0.421947 |       0.0 | 0.512856 |      0.0 |  0.0 |  0.0 |  92.694223 |      0.0 |       0.0 | 0.047195 |      0.0 | 0.421947 |      0.0 |      0.0 | 0.018434 |      0.0 | 0.494422 |  0.14246 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 3.305745 | 1.652872 |       0.0 |  0.0 |      0.0 | 1.373207 | 16.913666 |  0.0 |       0.0 |       0.0 | 0.046347 |      0.0 | 0.413471 |       0.0 | 0.513475 |      0.0 |  0.0 |  0.0 |  90.861198 |      0.0 |       0.0 | 0.046347 |      0.0 | 0.413471 |      0.0 |      0.0 | 0.019816 |      0.0 | 0.493659 | 0.142632 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 3.747947 | 1.873973 |       0.0 |  0.0 |      0.0 |  1.54235 | 18.451272 |  0.0 |       0.0 |       0.0 | 0.045431 |      0.0 | 0.404306 |       0.0 | 0.513903 |      0.0 |  0.0 |  0.0 |  88.869112 |      0.0 |       0.0 | 0.045431 |      0.0 | 0.404306 |      0.0 |      0.0 | 0.021087 |      0.0 | 0.492815 | 0.142751 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 2.050471 | 1.025236 |       0.0 |  0.0 |      0.0 | 0.834816 | 19.425089 |  0.0 |       0.0 |       0.0 | 0.044435 |      0.0 | 0.394346 |       0.0 | 0.514135 |      0.0 |  0.0 |  0.0 |  87.595516 |      0.0 |       0.0 | 0.044435 |      0.0 | 0.394346 |      0.0 |      0.0 |  0.02225 |      0.0 | 0.491885 | 0.142815 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 2.650012 | 1.325006 |       0.0 |  0.0 |      0.0 | 1.071233 | 16.401131 |  0.0 |       0.0 |       0.0 | 0.043798 |      0.0 | 0.387978 |       0.0 | 0.514199 |      0.0 |  0.0 |  0.0 |  86.092508 |      0.0 |       0.0 | 0.043798 |      0.0 | 0.387978 |      0.0 |      0.0 | 0.023316 |      0.0 | 0.490883 | 0.142833 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 2.959048 | 1.479524 |       0.0 |  0.0 |      0.0 | 1.185757 | 12.864637 |  0.0 |       0.0 |       0.0 | 0.043046 |      0.0 | 0.380463 |       0.0 | 0.514118 |      0.0 |  0.0 |  0.0 |  84.483242 |      0.0 |       0.0 | 0.043046 |      0.0 | 0.380463 |      0.0 |      0.0 | 0.024297 |      0.0 | 0.489821 | 0.142811 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 0.956604 | 0.478302 |       0.0 |  0.0 |     0.24 | 0.189137 | 13.428426 |  0.0 |       0.0 |       0.0 | 0.042242 |      0.0 | 0.372416 |       0.0 | 0.513884 |      0.0 |  0.0 |  0.0 |  83.879448 |      0.0 |       0.0 | 0.042242 |      0.0 | 0.372416 |      0.0 |      0.0 | 0.025191 |      0.0 | 0.488693 | 0.142746 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 0.731934 | 0.365967 |       0.0 |  0.0 |      0.0 | 0.289374 | 13.684693 |  0.0 |       0.0 |       0.0 |  0.04194 |      0.0 | 0.369397 |       0.0 | 0.513537 |      0.0 |  0.0 |  0.0 |  83.178737 |      0.0 |       0.0 |  0.04194 |      0.0 | 0.369397 |      0.0 |      0.0 | 0.026016 |      0.0 | 0.487521 | 0.142649 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 0.454628 | 0.227314 |       0.0 |  0.0 |      0.0 | 0.178944 | 12.557116 |  0.0 |       0.0 |       0.0 | 0.041589 |      0.0 | 0.365894 |       0.0 | 0.513112 |      0.0 |  0.0 |  0.0 |   82.59231 |      0.0 |       0.0 | 0.041589 |      0.0 | 0.365894 |      0.0 |      0.0 | 0.026784 |      0.0 | 0.486328 | 0.142531 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 0.300394 | 0.150197 |      0.76 |  0.0 | 0.150197 |      0.0 | 11.480791 |  0.0 |      0.76 |  0.107812 | 0.041296 |      0.0 | 0.362962 |  0.107812 | 0.522711 | 0.649803 |  0.0 |  0.0 |   82.84024 | 0.107812 |       0.0 | 0.041296 |      0.0 | 0.362962 | 0.010097 |      0.0 | 0.027498 |      0.0 | 0.485115 | 0.145197 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 0.089558 | 0.044779 |  6.569803 |  0.0 | 0.044779 |      0.0 | 11.275777 |  0.0 |  6.569803 |   0.96994 |  0.04142 |      0.0 | 0.364201 |   0.96994 | 0.620632 | 0.755221 |  0.0 |  0.0 |  88.034482 |  0.96994 |       0.0 |  0.04142 |      0.0 | 0.364201 | 0.108551 |      0.0 | 0.028174 |      0.0 | 0.483906 | 0.172398 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 |  0.03977 | 0.019885 |  3.435221 |  0.0 | 0.019885 |      0.0 | 10.865749 |  0.0 |  3.435221 |  0.535552 | 0.044017 |      0.0 | 0.390172 |  0.535552 | 0.735741 | 0.780115 |  0.0 |  0.0 |  90.499961 | 0.535552 |       0.0 | 0.044017 |      0.0 | 0.390172 | 0.224012 |      0.0 | 0.028884 |      0.0 | 0.482844 | 0.204372 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 | 0.039382 | 0.019691 |  5.860115 |  0.0 | 0.019691 |      0.0 | 10.558228 |  0.0 |  5.860115 |  0.958535 |  0.04525 |      0.0 |   0.4025 |  0.958535 | 0.831735 | 0.780309 |  0.0 |  0.0 |  94.953791 | 0.958535 |       0.0 |  0.04525 |      0.0 |   0.4025 | 0.320099 |      0.0 | 0.029653 |      0.0 | 0.481984 | 0.231038 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 0.039251 | 0.019626 | 12.700309 |  0.0 | 0.019626 |      0.0 | 10.455721 |  0.0 | 12.700309 |  2.288291 | 0.047477 |      0.0 | 0.424769 |  2.288291 | 1.158742 | 0.780374 |  0.0 |  0.0 | 104.893563 | 1.562993 |  0.725298 | 0.047477 |      0.0 | 0.424769 | 0.492438 | 0.154533 | 0.030468 |      0.0 | 0.481304 | 0.321873 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 |  0.03899 | 0.019495 |  0.100374 |  0.0 | 0.019495 |      0.0 | 10.250707 |  0.0 |  0.100374 |  0.019216 | 0.052447 | 0.009743 | 0.474468 |  0.019216 | 1.279374 | 0.780505 |  0.0 |  0.0 | 104.438065 | 0.019216 |       0.0 | 0.052447 | 0.009743 | 0.474468 | 0.541916 | 0.224579 |  0.03142 | 0.000471 | 0.480989 | 0.355382 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.03899 | 0.019495 |  0.820505 |  0.0 | 0.019495 |      0.0 | 10.250707 |  0.0 |  0.820505 |   0.15675 | 0.052219 | 0.008415 |  0.47219 |   0.15675 | 1.110902 | 0.780505 |  0.0 |  0.0 | 104.568996 |  0.15675 |       0.0 | 0.052219 | 0.008415 |  0.47219 | 0.460047 | 0.136214 |  0.03244 | 0.001289 | 0.480913 | 0.308584 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 |  0.03899 | 0.019495 |  3.580505 |  0.0 | 0.019495 |      0.0 | 10.250707 |  0.0 |  3.580505 |  0.694905 | 0.052284 |  0.00879 | 0.472845 |  0.694905 | 1.054307 | 0.780505 |  0.0 |  0.0 | 106.920677 | 0.694905 |       0.0 | 0.052284 |  0.00879 | 0.472845 | 0.455469 | 0.082618 | 0.033406 | 0.001986 | 0.480829 | 0.292863 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.038792 | 0.019396 |  2.500505 |  0.0 | 0.019396 |      0.0 | 10.096946 |  0.0 |  2.500505 |  0.497074 |  0.05346 | 0.016386 | 0.484603 |  0.497074 | 1.048618 | 0.780604 |  0.0 |  0.0 | 108.369658 | 0.497074 |       0.0 |  0.05346 | 0.016386 | 0.484603 | 0.480344 |  0.05011 | 0.034356 |    0.003 | 0.480808 | 0.291283 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 | 0.038661 |  0.01933 | 12.460604 |  0.0 |  0.01933 |      0.0 |  9.994439 |  0.0 | 12.460604 |  2.648297 | 0.054185 | 0.021792 | 0.491848 |  2.648297 | 1.358463 |  0.78067 |  0.0 |  0.0 | 117.614139 | 1.622399 |  1.025898 | 0.054185 | 0.021792 | 0.491848 | 0.588767 | 0.248973 | 0.035305 | 0.004536 | 0.480882 | 0.377351 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.038528 | 0.019264 |   4.18067 |  0.0 | 0.019264 |      0.0 |  9.891932 |  0.0 |   4.18067 |  0.954177 | 0.058807 | 0.066533 | 0.538071 |  0.954177 | 1.575545 | 0.780736 |  0.0 |  0.0 | 120.177222 | 0.954177 |       0.0 | 0.058807 | 0.066533 | 0.538071 | 0.713551 |  0.33609 |  0.03634 | 0.008342 | 0.481222 | 0.437651 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.038396 | 0.019198 |  4.060736 |  0.0 | 0.019198 |      0.0 |  9.789425 |  0.0 |  4.060736 |   0.95486 | 0.060089 | 0.081571 | 0.550886 |   0.95486 | 1.495007 | 0.780802 |  0.0 |  0.0 | 122.590552 |  0.95486 |       0.0 | 0.060089 | 0.081571 | 0.550886 | 0.757233 | 0.203849 | 0.037467 | 0.014607 | 0.481851 |  0.41528 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.074919 |  0.03746 |  1.420802 |  0.0 |  0.03746 |      0.0 |  9.891932 |  0.0 |  1.420802 |  0.339664 | 0.061295 | 0.096635 | 0.562953 |  0.339664 | 1.401988 |  0.76254 |  0.0 |  0.0 | 122.950807 | 0.339664 |       0.0 | 0.061295 | 0.096635 | 0.562953 | 0.735442 | 0.123641 |   0.0386 | 0.021708 | 0.482598 | 0.389441 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.505845 | 0.252922 |   0.08254 |  0.0 | 0.252922 |      0.0 | 10.045692 |  0.0 |   0.08254 |  0.019698 | 0.061475 | 0.098955 | 0.564754 |  0.019698 | 1.260794 | 0.547078 |  0.0 |  0.0 | 122.288465 | 0.019698 |       0.0 | 0.061475 | 0.098955 | 0.564754 | 0.633733 | 0.074992 | 0.039711 | 0.028951 | 0.483407 |  0.35022 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 | 1.218215 | 0.609108 |       0.0 |  0.0 | 0.547078 | 0.057195 | 10.045692 |  0.0 |       0.0 |       0.0 | 0.061144 | 0.094703 | 0.561442 |       0.0 | 1.126438 |      0.0 |  0.0 |  0.0 | 121.513981 |      0.0 |       0.0 | 0.061144 | 0.094703 | 0.561442 | 0.520583 | 0.045485 | 0.040764 | 0.035407 |   0.4842 |   0.3129 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.772303 | 0.386152 |       0.0 |  0.0 |      0.0 | 0.355393 |   10.1482 |  0.0 |       0.0 |       0.0 | 0.060757 |  0.08981 |  0.55757 |       0.0 | 1.021316 |      0.0 |  0.0 |  0.0 | 120.450451 |      0.0 |       0.0 | 0.060757 |  0.08981 |  0.55757 | 0.426217 | 0.027588 | 0.041749 | 0.040813 | 0.484949 | 0.283699 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 |   0.9844 |   0.4922 |       0.0 |  0.0 |     0.48 | 0.011199 | 10.250707 |  0.0 |       0.0 |       0.0 | 0.060225 | 0.083233 | 0.552252 |       0.0 | 0.939155 |      0.0 |  0.0 |  0.0 | 119.743542 |      0.0 |       0.0 | 0.060225 | 0.083233 | 0.552252 | 0.348957 | 0.016733 | 0.042663 | 0.045158 | 0.485645 | 0.260876 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 2.486378 | 1.243189 |       0.0 |  0.0 |     0.12 | 1.029208 |  10.30196 |  0.0 |       0.0 |       0.0 | 0.059872 | 0.078955 | 0.548718 |       0.0 | 0.874225 |      0.0 |  0.0 |  0.0 | 118.026789 |      0.0 |       0.0 | 0.059872 | 0.078955 | 0.548718 | 0.285702 | 0.010149 | 0.043511 | 0.048574 |  0.48629 |  0.24284 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 | 1.349913 | 0.674957 |      3.52 |  0.0 | 0.674957 |      0.0 |   10.1482 |  0.0 |      3.52 |  0.804879 | 0.059013 | 0.068884 | 0.540134 |  0.804879 | 0.897582 | 0.125043 |  0.0 |  0.0 | 120.073879 | 0.804879 |       0.0 | 0.059013 | 0.068884 | 0.540134 | 0.309293 | 0.006156 | 0.044287 | 0.050978 | 0.486868 | 0.249328 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 3.528622 | 1.764311 |  6.405043 |  0.0 |      0.8 | 0.884359 | 10.045692 |  0.0 |  6.405043 |  1.520398 | 0.060037 | 0.080945 | 0.550369 |  1.520398 | 1.076887 |      0.0 |  0.0 |  0.0 | 123.382815 | 1.342277 |   0.17812 | 0.060037 | 0.080945 | 0.550369 | 0.449457 | 0.041684 | 0.045031 | 0.053265 | 0.487449 | 0.299135 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 |  1.61841 | 0.809205 |      0.52 |  0.0 |      0.8 | 0.008509 |  9.994439 |  0.0 |      0.52 |  0.124975 | 0.061691 | 0.101762 | 0.566914 |  0.124975 | 1.145578 |      0.0 |  0.0 |  0.0 | 123.038963 | 0.124975 |       0.0 | 0.061691 | 0.101762 | 0.566914 | 0.497293 | 0.057417 | 0.045803 | 0.056906 | 0.488158 | 0.318216 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.844303 | 0.422151 |     24.04 |  0.0 | 0.422151 |      0.0 |  9.533157 |  0.0 |     24.04 |  6.385076 | 0.061519 | 0.099526 | 0.565195 |  6.385076 | 2.189796 | 0.377849 |  0.0 |  0.0 | 139.967647 | 1.843385 |  4.541691 | 0.061519 | 0.099526 | 0.565195 | 0.590738 | 1.002484 | 0.046574 | 0.061067 | 0.488933 | 0.608277 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.895703 | 0.447851 | 45.057849 |  0.0 | 0.447851 |      0.0 |  9.328143 |  0.0 | 45.057849 | 15.758715 | 0.069984 | 0.227408 | 0.649838 | 15.758715 | 5.807432 | 0.352149 |  0.0 |  0.0 |  168.31955 | 1.936543 | 13.822172 | 0.069984 | 0.227408 | 0.649838 | 0.826529 | 4.372365 | 0.047511 | 0.070913 | 0.490114 | 1.613176 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 1.151139 |  0.57557 |  9.392149 |  0.0 |  0.57557 |      0.0 |  9.276889 |  0.0 |  9.392149 |   3.98674 |  0.08416 |  0.50823 | 0.791598 |   3.98674 | 7.273291 |  0.22443 |  0.0 |  0.0 | 172.340972 | 1.749168 |  2.237571 |  0.08416 |  0.50823 | 0.791598 | 1.010192 | 5.622344 | 0.048956 |  0.09939 | 0.492409 | 2.020359 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.636581 |  0.31829 |   3.74443 |  0.0 |  0.31829 |      0.0 |  9.020622 |  0.0 |   3.74443 |  1.640415 |  0.08617 | 0.553757 | 0.811705 |  1.640415 | 5.664322 |  0.48171 |  0.0 |  0.0 | 172.993355 | 1.390398 |  0.250017 |  0.08617 | 0.553757 | 0.811705 | 1.110546 | 3.867069 | 0.050722 | 0.140498 | 0.495487 | 1.573423 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.116642 | 0.058321 |   8.68171 |  0.0 | 0.058321 |      0.0 |  8.866861 |  0.0 |   8.68171 |  3.912578 | 0.086497 | 0.561264 | 0.814967 |  3.912578 | 4.778279 | 0.741679 |  0.0 |  0.0 | 176.299758 | 1.744414 |  2.168164 | 0.086497 | 0.561264 | 0.814967 | 1.194429 | 2.852553 | 0.052459 | 0.180188 | 0.498649 |   1.3273 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.037049 | 0.018524 | 22.141679 |  0.0 | 0.018524 |      0.0 |  8.764354 |  0.0 | 22.141679 | 11.051854 |  0.08815 | 0.599829 | 0.831499 | 11.051854 | 6.153137 | 0.781476 |  0.0 |  0.0 | 185.870106 | 1.909517 |  9.142336 |  0.08815 | 0.599829 | 0.831499 | 1.309587 | 4.069193 |  0.05416 | 0.218318 | 0.501879 | 1.709205 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 |  0.03698 |  0.01849 | 18.461476 |  0.0 |  0.01849 |      0.0 |  8.713101 |  0.0 | 18.461476 | 10.642573 | 0.092935 | 0.716152 | 0.879351 | 10.642573 | 8.218452 |  0.78151 |  0.0 |  0.0 |  192.00057 | 1.906038 |  8.736535 | 0.092935 | 0.716152 | 0.879351 |  1.41801 | 5.978859 | 0.055935 | 0.260251 | 0.505397 | 2.282903 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.036499 |  0.01825 |   7.54151 |  0.0 |  0.01825 |      0.0 |  8.354326 |  0.0 |   7.54151 |  4.706328 |    0.096 | 0.794197 | 0.910003 |  4.706328 | 8.194259 |  0.78175 |  0.0 |  0.0 | 193.035552 |  1.78752 |  2.918807 |    0.096 | 0.794197 | 0.910003 | 1.495375 | 5.824387 | 0.057815 | 0.307411 | 0.509271 | 2.276183 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.035873 | 0.017937 |   2.26175 |  0.0 | 0.017937 |      0.0 |  7.893044 |  0.0 |   2.26175 |  1.410813 | 0.096518 | 0.807636 | 0.915178 |  1.410813 |  6.51394 | 0.782063 |  0.0 |  0.0 | 192.067158 | 1.291189 |  0.119624 | 0.096518 | 0.807636 | 0.915178 | 1.501849 | 4.084733 |  0.05969 | 0.354385 | 0.513284 | 1.809428 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.035943 | 0.017972 |  5.862063 |  0.0 | 0.017972 |      0.0 |  7.944298 |  0.0 |  5.862063 |   3.63316 | 0.096034 | 0.795059 | 0.910336 |   3.63316 | 5.385613 | 0.782028 |  0.0 |  0.0 | 192.494632 | 1.724758 |  1.908403 | 0.096034 | 0.795059 | 0.910336 | 1.504268 | 2.905704 | 0.061474 | 0.396909 | 0.517259 | 1.496004 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.035873 | 0.017937 |  3.222028 |  0.0 | 0.017937 |      0.0 |  7.893044 |  0.0 |  3.222028 |  1.991828 | 0.096247 | 0.800603 | 0.912473 |  1.991828 | 4.754322 | 0.782063 |  0.0 |  0.0 | 191.915509 | 1.497949 |   0.49388 | 0.096247 | 0.800603 | 0.912473 | 1.522994 | 2.211916 | 0.063165 | 0.435066 | 0.521181 | 1.320645 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.035522 | 0.017761 |  0.582063 |  0.0 | 0.017761 |      0.0 |  7.636776 |  0.0 |  0.582063 |  0.350289 | 0.095958 | 0.793096 | 0.909578 |  0.350289 | 3.900986 | 0.782239 |  0.0 |  0.0 | 190.348652 | 0.350289 |       0.0 | 0.095958 | 0.793096 | 0.909578 | 1.410972 | 1.430695 | 0.064771 | 0.469488 |  0.52506 | 1.083607 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.035663 | 0.017831 |  0.222239 |  0.0 | 0.017831 |      0.0 |  7.739283 |  0.0 |  0.222239 |  0.128894 | 0.095174 | 0.772903 | 0.901743 |  0.128894 | 3.160155 | 0.782169 |  0.0 |  0.0 | 188.672176 | 0.128894 |       0.0 | 0.095174 | 0.772903 | 0.901743 | 1.197968 |  0.86776 | 0.066273 | 0.499307 | 0.528847 | 0.877821 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.036083 | 0.018041 |  0.582169 |  0.0 | 0.018041 |      0.0 |  8.046805 |  0.0 |  0.582169 |  0.326672 | 0.094336 | 0.751491 | 0.893361 |  0.326672 | 2.673508 | 0.781959 |  0.0 |  0.0 | 187.188486 | 0.326672 |       0.0 | 0.094336 | 0.751491 | 0.893361 |   1.0227 | 0.526323 | 0.067662 | 0.524307 | 0.532515 | 0.742641 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.036637 | 0.018319 |  2.861959 |  0.0 | 0.018319 |      0.0 |  8.456833 |  0.0 |  2.861959 |   1.57602 | 0.093594 | 0.732708 | 0.885942 |   1.57602 | 2.507939 | 0.781681 |  0.0 |  0.0 | 186.762179 |  1.36549 |   0.21053 | 0.093594 | 0.732708 | 0.885942 | 0.993821 | 0.364087 | 0.068944 | 0.545018 | 0.536069 |  0.69665 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.061944 | 0.030972 |  0.461681 |  0.0 | 0.030972 |      0.0 |  8.610594 |  0.0 |  0.461681 |  0.249638 | 0.093381 | 0.727341 | 0.883811 |  0.249638 | 2.387801 | 0.769028 |  0.0 |  0.0 | 185.269689 | 0.249638 |       0.0 | 0.093381 | 0.727341 | 0.883811 |  0.95669 | 0.258811 | 0.070141 |  0.56262 | 0.539539 | 0.663278 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.188148 | 0.094074 |  0.209028 |  0.0 | 0.094074 |      0.0 |  8.969368 |  0.0 |  0.209028 |  0.109916 | 0.092635 | 0.708654 | 0.876348 |  0.109916 |  2.16399 | 0.705926 |  0.0 |  0.0 | 183.691162 | 0.109916 |       0.0 | 0.092635 | 0.708654 | 0.876348 | 0.815437 | 0.156977 | 0.071256 | 0.577391 | 0.542928 | 0.601108 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.46949 | 0.234745 |       0.0 |  0.0 | 0.234745 |      0.0 |  9.174382 |  0.0 |       0.0 |       0.0 | 0.091846 | 0.689068 | 0.868456 |       0.0 | 1.979886 | 0.471181 |  0.0 |  0.0 | 182.041793 |      0.0 |       0.0 | 0.091846 | 0.689068 | 0.868456 | 0.677254 | 0.095211 |  0.07228 | 0.588935 | 0.546206 | 0.549968 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 1.680918 | 0.840459 |       0.0 |  0.0 | 0.471181 |  0.36645 |  8.713101 |  0.0 |       0.0 |       0.0 | 0.091021 | 0.668798 | 0.860209 |       0.0 | 1.832306 |      0.0 |  0.0 |  0.0 | 180.055315 |      0.0 |       0.0 | 0.091021 | 0.668798 | 0.860209 | 0.554489 | 0.057749 | 0.073214 | 0.597483 | 0.549371 | 0.508974 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 1.743886 | 0.871943 |       0.0 |  0.0 |     0.36 | 0.507473 |   8.55934 |  0.0 |       0.0 |       0.0 | 0.090028 | 0.644656 | 0.850277 |       0.0 | 1.718578 |      0.0 |  0.0 |  0.0 | 177.962883 |      0.0 |       0.0 | 0.090028 | 0.644656 | 0.850277 | 0.453977 | 0.035026 | 0.074058 | 0.603102 | 0.552415 | 0.477383 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 2.122301 | 1.061151 |      2.32 |  0.0 |      0.8 |  0.25856 |  8.764354 |  0.0 |      2.32 |  1.094569 | 0.088981 | 0.619547 | 0.839814 |  1.094569 | 1.732394 |      0.0 |  0.0 |  0.0 |  177.38141 | 1.086399 |  0.008171 | 0.088981 | 0.619547 | 0.839814 |  0.47343 | 0.022985 | 0.074811 | 0.605842 | 0.555326 | 0.481221 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 2.150411 | 1.075205 |      0.04 |  0.0 |      0.8 | 0.272381 |  8.764354 |  0.0 |      0.04 |  0.018544 | 0.088691 | 0.612629 | 0.836907 |  0.018544 | 1.740398 |      0.0 |  0.0 |  0.0 | 175.592259 | 0.018544 |       0.0 | 0.088691 | 0.612629 | 0.836907 | 0.484534 | 0.015415 | 0.075495 | 0.606811 | 0.558143 | 0.483444 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 1.863566 | 0.931783 |       0.0 |  0.0 |     0.36 | 0.565298 |  8.815608 |  0.0 |       0.0 |       0.0 | 0.087796 | 0.591505 | 0.827961 |       0.0 | 1.651009 |      0.0 |  0.0 |  0.0 | 173.519699 |      0.0 |       0.0 | 0.087796 | 0.591505 | 0.827961 | 0.398328 |  0.00935 | 0.076116 | 0.606343 | 0.560872 | 0.458614 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 3.115838 | 1.557919 |       0.0 |  0.0 |     0.36 | 1.182759 |  8.866861 |  0.0 |       0.0 |       0.0 |  0.08676 | 0.567346 | 0.817598 |       0.0 | 1.575695 |      0.0 |  0.0 |  0.0 | 170.865235 |      0.0 |       0.0 |  0.08676 | 0.567346 | 0.817598 | 0.326123 | 0.005671 | 0.076661 | 0.603762 | 0.563478 | 0.437693 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 1.940627 | 0.970313 |       0.0 |  0.0 |      0.0 | 0.956307 |  9.840678 |  0.0 |       0.0 |       0.0 | 0.085433 | 0.536899 | 0.804326 |       0.0 | 1.512331 |      0.0 |  0.0 |  0.0 | 168.482271 |      0.0 |       0.0 | 0.085433 | 0.536899 | 0.804326 | 0.267007 |  0.00344 |  0.07712 | 0.598824 |  0.56594 | 0.420092 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 2.122768 | 1.061384 |       0.0 |  0.0 |      0.0 | 1.044259 |  9.789425 |  0.0 |       0.0 |       0.0 | 0.084241 | 0.510046 | 0.792411 |       0.0 | 1.458074 |      0.0 |  0.0 |  0.0 | 166.051312 |      0.0 |       0.0 | 0.084241 | 0.510046 | 0.792411 | 0.218607 | 0.002086 | 0.077497 | 0.591632 | 0.568253 | 0.405021 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 1.314414 | 0.657207 |       0.0 |  0.0 |      0.0 | 0.645396 |  9.686918 |  0.0 |       0.0 |       0.0 | 0.083026 | 0.483131 | 0.780257 |       0.0 |  1.41103 |      0.0 |  0.0 |  0.0 | 164.059503 |      0.0 |       0.0 | 0.083026 | 0.483131 | 0.780257 |  0.17898 | 0.001265 | 0.077796 | 0.582566 | 0.570423 | 0.391953 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.602486 | 0.301243 |       0.0 |  0.0 |      0.0 | 0.295351 |  8.969368 |  0.0 |       0.0 |       0.0 |  0.08203 | 0.461443 | 0.770298 |       0.0 | 1.369845 |      0.0 |  0.0 |  0.0 | 162.450382 |      0.0 |       0.0 |  0.08203 | 0.461443 | 0.770298 | 0.146537 | 0.000767 | 0.078026 | 0.572054 | 0.572461 | 0.380513 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.109905 | 0.054952 |       0.0 |  0.0 |      0.0 | 0.053804 |  8.200565 |  0.0 |       0.0 |       0.0 | 0.081225 | 0.444166 | 0.762252 |       0.0 | 1.333723 |      0.0 |  0.0 |  0.0 | 161.108936 |      0.0 |       0.0 | 0.081225 | 0.444166 | 0.762252 | 0.119974 | 0.000466 | 0.078202 | 0.560693 | 0.574389 | 0.370479 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.035026 | 0.017513 |       0.0 |  0.0 |      0.0 | 0.017127 |  7.278002 |  0.0 |       0.0 |       0.0 | 0.080554 | 0.429932 | 0.755545 |       0.0 | 1.301981 |      0.0 |  0.0 |  0.0 | 159.825778 |      0.0 |       0.0 | 0.080554 | 0.429932 | 0.755545 | 0.098226 | 0.000282 | 0.078333 | 0.548915 | 0.576225 | 0.361662 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |       0.0 |  0.0 |      0.0 | 0.016756 |  6.765466 |  0.0 |       0.0 |       0.0 | 0.079913 | 0.416462 | 0.749129 |       0.0 | 1.273935 |      0.0 |  0.0 |  0.0 | 158.563518 |      0.0 |       0.0 | 0.079913 | 0.416462 | 0.749129 | 0.080421 | 0.000171 | 0.078425 | 0.536941 | 0.577977 | 0.353871 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.033727 | 0.016863 |       0.0 |  0.0 |      0.0 | 0.016453 |  6.355438 |  0.0 |       0.0 |       0.0 | 0.079282 | 0.403351 | 0.742818 |       0.0 |  1.24892 |      0.0 |  0.0 |  0.0 | 157.321615 |      0.0 |       0.0 | 0.079282 | 0.403351 | 0.742818 | 0.065843 | 0.000104 | 0.078482 | 0.524841 | 0.579649 | 0.346922 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.033286 | 0.016643 |       0.0 |  0.0 |      0.0 | 0.016218 |  6.047917 |  0.0 |       0.0 |       0.0 | 0.078661 | 0.390589 | 0.736608 |       0.0 | 1.226381 |      0.0 |  0.0 |  0.0 | 156.099539 |      0.0 |       0.0 | 0.078661 | 0.390589 | 0.736608 | 0.053908 | 0.000063 | 0.078506 | 0.512663 | 0.581242 | 0.340661 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.032916 | 0.016458 |       0.0 |  0.0 |      0.0 | 0.016018 |  5.791649 |  0.0 |       0.0 |       0.0 |  0.07805 | 0.378165 | 0.730498 |       0.0 | 1.205875 |      0.0 |  0.0 |  0.0 | 154.896809 |      0.0 |       0.0 |  0.07805 | 0.378165 | 0.730498 | 0.044136 | 0.000038 | 0.078498 | 0.500445 | 0.582757 | 0.334965 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |       0.0 |  0.0 |      0.0 | 0.016428 |  6.406692 |  0.0 |       0.0 |       0.0 | 0.077448 | 0.366069 | 0.724484 |       0.0 | 1.187041 |      0.0 |  0.0 |  0.0 | 153.712379 |      0.0 |       0.0 | 0.077448 | 0.366069 | 0.724484 | 0.036135 | 0.000023 | 0.078462 | 0.488223 | 0.584197 | 0.329734 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |       0.0 |  0.0 |      0.0 | 0.016478 |  6.509199 |  0.0 |       0.0 |       0.0 | 0.076856 | 0.354286 | 0.718562 |       0.0 | 1.169589 |      0.0 |  0.0 |  0.0 | 152.546197 |      0.0 |       0.0 | 0.076856 | 0.354286 | 0.718562 | 0.029585 | 0.000014 | 0.078398 | 0.476029 | 0.585563 | 0.324886 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.033434 | 0.016717 |      0.76 |  0.0 | 0.016717 |      0.0 |  6.150424 |  0.0 |      0.76 |  0.256904 | 0.076273 | 0.342811 | 0.712731 |  0.256904 | 1.177345 | 0.783283 |  0.0 |  0.0 | 151.917478 | 0.256904 |       0.0 | 0.076273 | 0.342811 | 0.712731 | 0.048282 | 0.000009 | 0.078308 | 0.463888 | 0.586858 |  0.32704 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 |  0.03336 |  0.01668 |       0.0 |  0.0 |  0.01668 |      0.0 |   6.09917 |  0.0 |       0.0 |       0.0 | 0.075959 | 0.336677 | 0.709587 |       0.0 |  1.18041 | 0.766603 |  0.0 |  0.0 | 150.795255 |      0.0 |       0.0 | 0.075959 | 0.336677 | 0.709587 | 0.062039 | 0.000005 | 0.078201 |  0.45207 | 0.588095 | 0.327892 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |       0.0 |  0.0 | 0.016973 |      0.0 |  6.509199 |  0.0 |       0.0 |       0.0 | 0.075398 |  0.32582 | 0.703976 |       0.0 | 1.158713 |  0.74963 |  0.0 |  0.0 | 149.690062 |      0.0 |       0.0 | 0.075398 |  0.32582 | 0.703976 | 0.050793 | 0.000003 | 0.078078 | 0.440563 | 0.589275 | 0.321865 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.124092 | 0.062046 |       0.0 |  0.0 | 0.062046 |      0.0 |  6.662959 |  0.0 |       0.0 |       0.0 | 0.074845 | 0.315244 |  0.69845 |       0.0 | 1.139043 | 0.687585 |  0.0 |  0.0 | 148.601523 |      0.0 |       0.0 | 0.074845 | 0.315244 |  0.69845 | 0.041586 | 0.000002 | 0.077934 | 0.429132 | 0.590389 | 0.316401 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.556783 | 0.278392 |  0.727585 |  0.0 | 0.278392 |      0.0 |  6.457945 |  0.0 |  0.727585 |   0.23476 | 0.074301 | 0.304942 | 0.693008 |   0.23476 | 1.143038 | 0.521608 |  0.0 |  0.0 | 148.022097 |  0.23476 |       0.0 | 0.074301 | 0.304942 | 0.693008 | 0.056034 | 0.000001 |  0.07777 | 0.417796 | 0.591437 | 0.317511 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 |  1.27835 | 0.639175 |  0.201608 |  0.0 | 0.639175 |      0.0 |  6.252931 |  0.0 |  0.201608 |  0.064476 | 0.074011 | 0.299505 |  0.69011 |  0.064476 | 1.149305 | 0.160825 |  0.0 |  0.0 | 147.095603 | 0.064476 |       0.0 | 0.074011 | 0.299505 |  0.69011 | 0.072484 | 0.000001 | 0.077594 | 0.406794 | 0.592434 | 0.319251 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 |  2.07666 |  1.03833 |       0.0 |  0.0 | 0.280825 | 0.729729 |  6.355438 |  0.0 |       0.0 |       0.0 | 0.073548 | 0.290879 | 0.685478 |       0.0 |  1.13195 |      0.0 |  0.0 |  0.0 | 145.315969 |      0.0 |       0.0 | 0.073548 | 0.290879 | 0.685478 | 0.064994 |      0.0 | 0.077407 | 0.396166 | 0.593382 | 0.314431 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 | 2.862014 | 1.431007 |       0.0 |  0.0 |     0.48 | 0.914014 |  7.072988 |  0.0 |       0.0 |       0.0 | 0.072658 | 0.274549 |  0.67658 |       0.0 | 1.110021 |      0.0 |  0.0 |  0.0 | 143.378168 |      0.0 |       0.0 | 0.072658 | 0.274549 |  0.67658 | 0.053212 |      0.0 | 0.077197 | 0.385357 | 0.594254 | 0.308339 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 3.356573 | 1.678286 |       0.0 |  0.0 |      0.0 | 1.608745 |  9.174382 |  0.0 |       0.0 |       0.0 | 0.071689 | 0.257128 | 0.666891 |       0.0 | 1.089514 |      0.0 |  0.0 |  0.0 | 140.773716 |      0.0 |       0.0 | 0.071689 | 0.257128 | 0.666891 | 0.043567 |      0.0 | 0.076952 |  0.37397 | 0.595025 | 0.302643 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 3.356514 | 1.678257 |       0.0 |  0.0 |      0.0 | 1.602677 |  9.738171 |  0.0 |       0.0 |       0.0 | 0.070387 | 0.234322 | 0.653869 |       0.0 | 1.069756 |      0.0 |  0.0 |  0.0 | 138.212461 |      0.0 |       0.0 | 0.070387 | 0.234322 | 0.653869 | 0.035669 |      0.0 | 0.076663 | 0.361747 | 0.595675 | 0.297154 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 4.286095 | 2.143048 |       0.0 |  0.0 |      0.0 | 2.038476 |  11.89082 |  0.0 |       0.0 |       0.0 | 0.069106 | 0.212594 | 0.641062 |       0.0 |  1.05029 |      0.0 |  0.0 |  0.0 | 135.251223 |      0.0 |       0.0 | 0.069106 | 0.212594 | 0.641062 | 0.029204 |      0.0 | 0.076326 |  0.34857 | 0.596191 | 0.291747 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 3.753358 | 1.876679 |       0.0 |  0.0 |      0.0 | 1.776399 | 11.378284 |  0.0 |       0.0 |       0.0 | 0.067626 | 0.188367 | 0.626256 |       0.0 | 1.030869 |      0.0 |  0.0 |  0.0 | 132.592576 |      0.0 |       0.0 | 0.067626 | 0.188367 | 0.626256 |  0.02391 |      0.0 | 0.075937 | 0.334458 | 0.596563 | 0.286352 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 3.144766 | 1.572383 |       0.0 |  0.0 |      0.0 |  1.48137 | 11.583298 |  0.0 |       0.0 |       0.0 | 0.066296 | 0.167464 | 0.612963 |       0.0 | 1.011412 |      0.0 |  0.0 |  0.0 | 130.264484 |      0.0 |       0.0 | 0.066296 | 0.167464 | 0.612963 | 0.019576 |      0.0 | 0.075499 | 0.319545 | 0.596793 | 0.280948 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 2.519332 | 1.259666 |       0.0 |  0.0 |      0.0 | 1.181556 | 11.788313 |  0.0 |       0.0 |       0.0 | 0.065132 | 0.149845 | 0.601322 |       0.0 | 0.992164 |      0.0 |  0.0 |  0.0 | 128.266628 |      0.0 |       0.0 | 0.065132 | 0.149845 | 0.601322 | 0.016027 |      0.0 | 0.075022 |  0.30422 | 0.596895 | 0.275601 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.179366 | 0.589683 |       0.0 |  0.0 |      0.0 | 0.550923 | 10.711988 |  0.0 |       0.0 |       0.0 | 0.064133 | 0.135255 | 0.591333 |       0.0 |  0.97335 |      0.0 |  0.0 |  0.0 | 126.924984 |      0.0 |       0.0 | 0.064133 | 0.135255 | 0.591333 | 0.013122 |      0.0 | 0.074515 | 0.288823 |  0.59689 | 0.270375 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 0.523693 | 0.261846 |       0.0 |  0.0 |      0.0 | 0.243955 |  9.533157 |  0.0 |       0.0 |       0.0 | 0.063462 | 0.125741 | 0.584625 |       0.0 | 0.955286 |      0.0 |  0.0 |  0.0 | 125.907201 |      0.0 |       0.0 | 0.063462 | 0.125741 | 0.584625 | 0.010743 |      0.0 | 0.073992 | 0.273749 | 0.596801 | 0.265357 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 |  0.13182 |  0.06591 |       0.0 |  0.0 |      0.0 | 0.061273 |  8.200565 |  0.0 |       0.0 |       0.0 | 0.062954 | 0.118679 | 0.579536 |       0.0 | 0.938239 |      0.0 |  0.0 |  0.0 |  125.08476 |      0.0 |       0.0 | 0.062954 | 0.118679 | 0.579536 | 0.008796 |      0.0 | 0.073466 | 0.259323 | 0.596654 | 0.260622 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 0.035803 | 0.017902 |       0.0 |  0.0 |      0.0 | 0.016612 |  7.841791 |  0.0 |       0.0 |       0.0 | 0.062542 | 0.113073 | 0.575424 |       0.0 | 0.922276 |      0.0 |  0.0 |  0.0 | 124.317109 |      0.0 |       0.0 | 0.062542 | 0.113073 | 0.575424 | 0.007202 |      0.0 | 0.072943 | 0.245668 | 0.596464 | 0.256188 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |       0.0 |  0.0 |      0.0 | 0.015891 |  6.765466 |  0.0 |       0.0 |       0.0 | 0.062159 | 0.107922 | 0.571586 |       0.0 | 0.907358 |      0.0 |  0.0 |  0.0 | 123.559551 |      0.0 |       0.0 | 0.062159 | 0.107922 | 0.571586 | 0.005896 |      0.0 | 0.072427 |   0.2328 | 0.596235 | 0.252044 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |       0.0 |  0.0 |      0.0 | 0.015629 |  6.406692 |  0.0 |       0.0 |       0.0 |  0.06178 | 0.102918 | 0.567798 |       0.0 | 0.893389 |      0.0 |  0.0 |  0.0 | 122.811426 |      0.0 |       0.0 |  0.06178 | 0.102918 | 0.567798 | 0.004827 |      0.0 | 0.071916 | 0.220675 | 0.595971 | 0.248164 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |       0.0 |  0.0 |      0.0 | 0.015669 |  6.509199 |  0.0 |       0.0 |       0.0 | 0.061406 | 0.098055 | 0.564057 |       0.0 |  0.88027 |      0.0 |  0.0 |  0.0 | 122.072239 |      0.0 |       0.0 | 0.061406 | 0.098055 | 0.564057 | 0.003952 |      0.0 | 0.071413 | 0.209233 | 0.595672 |  0.24452 |

    .. raw:: html

        <iframe
            src="lland_v2_ex1.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    :ref:`Recalculation of example 2.1 <lland_v1_ex2_1>`

    The following calculation shows, that the outflow values of the
    integration test for water areas of type |WASSER| are reproduced
    exactly (when parameter |NegQ| is set to `False`):

    >>> lnk(WASSER)
    >>> test('lland_v2_ex2_1')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo | nbes | sbes |      evi | evb | wgtf | schm | wada | qdb | qib1 | qib2 | qbb | qdgz |         q | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 |     qbga |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.040283 | 0.020141 |  0.0 |  0.0 | 0.020141 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.477367 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.497508 |  0.132602 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.03912 |  0.01956 |  0.0 |  0.0 |  0.01956 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.472998 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.492558 |  0.131388 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.468261 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.487657 |  0.130072 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 | 0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.463607 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.482805 |   0.12878 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.458605 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.478001 |   0.12739 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.041104 | 0.020552 |  0.0 |  0.0 | 0.020552 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.452692 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.473245 |  0.125748 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.116763 | 0.058382 |  0.0 |  0.0 | 0.058382 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.410154 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.468536 |  0.113932 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 |  0.77315 | 0.386575 |  0.0 |  0.0 | 0.386575 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.077299 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.463874 |  0.021472 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 1.747814 | 0.873907 |  0.0 |  0.0 | 0.459258 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.459258 |       0.0 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 2.927022 | 1.463511 |  0.0 |  0.0 | 0.454688 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.454688 |       0.0 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 3.305745 | 1.652872 |  0.0 |  0.0 | 0.450164 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.450164 |       0.0 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 3.747947 | 1.873973 |  0.0 |  0.0 | 0.445685 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.445685 |       0.0 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 2.050471 | 1.025236 |  0.0 |  0.0 |  0.44125 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.44125 |       0.0 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 2.650012 | 1.325006 |  0.0 |  0.0 |  0.43686 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.43686 |       0.0 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 2.959048 | 1.479524 |  0.0 |  0.0 | 0.432513 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.432513 |       0.0 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 0.956604 | 0.478302 |  0.0 |  0.0 | 0.478302 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.189907 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.428209 |  0.052752 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 0.731934 | 0.365967 |  0.0 |  0.0 | 0.365967 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.057982 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.423949 |  0.016106 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 0.454628 | 0.227314 |  0.0 |  0.0 | 0.227314 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.192416 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.41973 |  0.053449 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 0.300394 | 0.150197 |  0.0 |  0.0 | 0.150197 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.825357 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.415554 |  0.507044 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 0.089558 | 0.044779 |  0.0 |  0.0 | 0.044779 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   7.08664 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.411419 |  1.968511 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 |  0.03977 | 0.019885 |  0.0 |  0.0 | 0.019885 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   3.86744 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.407325 |  1.074289 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 | 0.039382 | 0.019691 |  0.0 |  0.0 | 0.019691 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.263582 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.403272 |  1.739884 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 0.039251 | 0.019626 |  0.0 |  0.0 | 0.019626 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 13.099634 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.39926 |  3.638787 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.495792 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.395287 |   0.13772 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.211859 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.391354 |  0.336628 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.967965 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.38746 |  1.102213 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.884208 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.383605 |  0.801169 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 | 0.038661 |  0.01933 |  0.0 |  0.0 |  0.01933 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 12.840457 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.379788 |  3.566794 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.038528 | 0.019264 |  0.0 |  0.0 | 0.019264 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.556744 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.376009 |  1.265762 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 | 0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.433069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.372267 |  1.231408 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.074919 |  0.03746 |  0.0 |  0.0 |  0.03746 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.771104 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.368563 |  0.491973 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.505845 | 0.252922 |  0.0 |  0.0 | 0.252922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.231974 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.364896 |  0.064437 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 | 1.218215 | 0.609108 |  0.0 |  0.0 | 0.361265 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.361265 |       0.0 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.772303 | 0.386152 |  0.0 |  0.0 | 0.357671 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.357671 |       0.0 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 |   0.9844 |   0.4922 |  0.0 |  0.0 |   0.4922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.341911 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.354112 |  0.094975 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 2.486378 | 1.243189 |  0.0 |  0.0 | 0.470588 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.350588 |       0.0 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 | 1.349913 | 0.674957 |  0.0 |  0.0 | 0.674957 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.992143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   0.3471 |  1.108929 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 3.528622 | 1.764311 |  0.0 |  0.0 | 1.764311 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.659335 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.343646 |  1.572038 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 |  1.61841 | 0.809205 |  0.0 |  0.0 | 0.809205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.851022 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.340227 |  0.236395 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.844303 | 0.422151 |  0.0 |  0.0 | 0.422151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  24.75469 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.336841 |  6.876303 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.895703 | 0.447851 |  0.0 |  0.0 | 0.447851 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 45.365638 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.33349 | 12.601566 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 1.151139 |  0.57557 |  0.0 |  0.0 |  0.57557 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  9.594602 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.330172 |  2.665167 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.636581 |  0.31829 |  0.0 |  0.0 |  0.31829 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.328596 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.326886 |  1.202388 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.116642 | 0.058321 |  0.0 |  0.0 | 0.058321 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  9.265313 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.323634 |  2.573698 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.037049 | 0.018524 |  0.0 |  0.0 | 0.018524 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 22.501889 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.320413 |  6.250525 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 |  0.03698 |  0.01849 |  0.0 |  0.0 |  0.01849 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 18.778735 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.317225 |  5.216315 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.036499 |  0.01825 |  0.0 |  0.0 |  0.01825 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  7.855819 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.314069 |  2.182172 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 | 0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.573007 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.310944 |  0.714724 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.035943 | 0.017972 |  0.0 |  0.0 | 0.017972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.169878 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.30785 |  1.713855 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 | 0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   3.52685 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.304787 |  0.979681 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.035522 | 0.017761 |  0.0 |  0.0 | 0.017761 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.883993 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.301754 |  0.245554 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.035663 | 0.017831 |  0.0 |  0.0 | 0.017831 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.52092 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.298752 |    0.1447 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.036083 | 0.018041 |  0.0 |  0.0 | 0.018041 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.877738 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.295779 |  0.243816 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.036637 | 0.018319 |  0.0 |  0.0 | 0.018319 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.154517 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.292836 |  0.876255 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.061944 | 0.030972 |  0.0 |  0.0 | 0.030972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.73895 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.289922 |  0.205264 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.188148 | 0.094074 |  0.0 |  0.0 | 0.094074 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.432964 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.287037 |  0.120268 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.46949 | 0.234745 |  0.0 |  0.0 | 0.234745 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.049436 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.284181 |  0.013732 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 1.680918 | 0.840459 |  0.0 |  0.0 | 0.281354 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.281354 |       0.0 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 1.743886 | 0.871943 |  0.0 |  0.0 | 0.638554 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.278554 |       0.0 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 2.122301 | 1.061151 |  0.0 |  0.0 | 1.061151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.334632 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.275782 |  0.648509 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 2.150411 | 1.075205 |  0.0 |  0.0 | 1.075205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.037833 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.273038 |  0.010509 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 1.863566 | 0.931783 |  0.0 |  0.0 | 0.630322 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.270322 |       0.0 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 3.115838 | 1.557919 |  0.0 |  0.0 | 0.627632 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.267632 |       0.0 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 1.940627 | 0.970313 |  0.0 |  0.0 | 0.264969 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.264969 |       0.0 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 2.122768 | 1.061384 |  0.0 |  0.0 | 0.262332 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.262332 |       0.0 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 1.314414 | 0.657207 |  0.0 |  0.0 | 0.259722 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.259722 |       0.0 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.602486 | 0.301243 |  0.0 |  0.0 | 0.257138 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.257138 |       0.0 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.109905 | 0.054952 |  0.0 |  0.0 | 0.054952 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.199627 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.254579 |  0.055452 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.035026 | 0.017513 |  0.0 |  0.0 | 0.017513 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.234533 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.252046 |  0.065148 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 | 0.017154 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.232384 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.249538 |  0.064551 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.033727 | 0.016863 |  0.0 |  0.0 | 0.016863 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.230192 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.247055 |  0.063942 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.033286 | 0.016643 |  0.0 |  0.0 | 0.016643 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.227954 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.244597 |  0.063321 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.032916 | 0.016458 |  0.0 |  0.0 | 0.016458 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.225705 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.242163 |  0.062696 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |   0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.222854 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.239754 |  0.061904 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.220395 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.237368 |  0.061221 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.033434 | 0.016717 |  0.0 |  0.0 | 0.016717 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.778289 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.235006 |  0.493969 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 |  0.03336 |  0.01668 |  0.0 |  0.0 |  0.01668 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.215988 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.232668 |  0.059997 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.21338 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.230353 |  0.059272 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.124092 | 0.062046 |  0.0 |  0.0 | 0.062046 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.166015 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.228061 |  0.046115 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.556783 | 0.278392 |  0.0 |  0.0 | 0.278392 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.7874 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.225792 |  0.218722 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 |  1.27835 | 0.639175 |  0.0 |  0.0 | 0.639175 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.06437 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.223545 |   0.01788 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 |  2.07666 |  1.03833 |  0.0 |  0.0 | 0.341321 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.221321 |       0.0 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 | 2.862014 | 1.431007 |  0.0 |  0.0 | 0.699118 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.219118 |       0.0 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 3.356573 | 1.678286 |  0.0 |  0.0 | 0.216938 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.216938 |       0.0 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 3.356514 | 1.678257 |  0.0 |  0.0 |  0.21478 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.21478 |       0.0 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 4.286095 | 2.143048 |  0.0 |  0.0 | 0.212642 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.212642 |       0.0 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 3.753358 | 1.876679 |  0.0 |  0.0 | 0.210527 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.210527 |       0.0 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 3.144766 | 1.572383 |  0.0 |  0.0 | 0.208432 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.208432 |       0.0 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 2.519332 | 1.259666 |  0.0 |  0.0 | 0.206358 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.206358 |       0.0 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.179366 | 0.589683 |  0.0 |  0.0 | 0.204305 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.204305 |       0.0 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 0.523693 | 0.261846 |  0.0 |  0.0 | 0.202272 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.202272 |       0.0 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 |  0.13182 |  0.06591 |  0.0 |  0.0 |  0.06591 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.134349 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.200259 |  0.037319 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 0.035803 | 0.017902 |  0.0 |  0.0 | 0.017902 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.180365 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.198267 |  0.050101 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 | 0.017154 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.17914 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.196294 |  0.049761 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |   0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.177441 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.194341 |  0.049289 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.175434 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.192407 |  0.048732 |

    .. raw:: html

        <iframe
            src="lland_v2_ex2_1.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    :ref:`Modification of example 2.1 <lland_v1_ex2_1>`

    As discussed in the documentation of |lland_v1|, the handling of
    evaporation from water surfaces might be problematic.  |lland_v1|
    offers a smoothung option for the calculation of |ET0| (see method
    |calc_et0_wet0_v1|. In principle, the "delay weighing factor" |WfET0|
    can be applied on all land use classes.  However, its original
    intention is to allow for reflecting the temporal persistence of
    (large) water bodies.  This is demonstrated by setting the weighting
    parameter |WfET0| to a value smaller than one and defining a suitable
    "old" evaporation value (|WET0|):

    >>> wfet0(0.01)
    >>> test.inits.wet0 = 1.0

    Now |ET0| and (at least in low flow situations) consequently
    |lland_fluxes.Q| are strongly dampened:

    >>> test('lland_v2_ex2_1a')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo | nbes | sbes |      evi | evb | wgtf | schm | wada | qdb | qib1 | qib2 | qbb | qdgz |         q | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 |     qbga |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.990403 | 0.495201 |  0.0 |  0.0 | 0.495201 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.002307 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.497508 |  0.000641 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.98089 | 0.490445 |  0.0 |  0.0 | 0.490445 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.002113 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.492558 |  0.000587 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.971469 | 0.485735 |  0.0 |  0.0 | 0.485735 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.001922 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.487657 |  0.000534 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.962138 | 0.481069 |  0.0 |  0.0 | 0.481069 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.001736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.482805 |  0.000482 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.952905 | 0.476452 |  0.0 |  0.0 | 0.476452 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.001548 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.478001 |   0.00043 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.943787 | 0.471893 |  0.0 |  0.0 | 0.471893 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.001351 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.473245 |  0.000375 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.935517 | 0.467758 |  0.0 |  0.0 | 0.467758 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.000777 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.468536 |  0.000216 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 | 0.933893 | 0.466946 |  0.0 |  0.0 | 0.463874 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.463874 |       0.0 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 0.942032 | 0.471016 |  0.0 |  0.0 | 0.459258 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.459258 |       0.0 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 0.961882 | 0.480941 |  0.0 |  0.0 | 0.454688 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.454688 |       0.0 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 0.985321 |  0.49266 |  0.0 |  0.0 | 0.450164 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.450164 |       0.0 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 1.012947 | 0.506473 |  0.0 |  0.0 | 0.445685 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.445685 |       0.0 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 1.023322 | 0.511661 |  0.0 |  0.0 |  0.44125 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.44125 |       0.0 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 1.039589 | 0.519795 |  0.0 |  0.0 |  0.43686 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.43686 |       0.0 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 1.058784 | 0.529392 |  0.0 |  0.0 | 0.432513 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.432513 |       0.0 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 1.057762 | 0.528881 |  0.0 |  0.0 | 0.528881 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.139328 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.428209 |  0.038702 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 1.054504 | 0.527252 |  0.0 |  0.0 | 0.423949 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.423949 |       0.0 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 1.048505 | 0.524252 |  0.0 |  0.0 |  0.41973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.41973 |       0.0 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 1.041024 | 0.520512 |  0.0 |  0.0 | 0.520512 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.455042 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.415554 |  0.404178 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 1.031509 | 0.515755 |  0.0 |  0.0 | 0.515755 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.615665 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.411419 |  1.837685 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 | 1.021592 | 0.510796 |  0.0 |  0.0 | 0.510796 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   3.37653 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.407325 |  0.937925 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 |  1.01177 | 0.505885 |  0.0 |  0.0 | 0.505885 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.777388 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.403272 |   1.60483 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 1.002044 | 0.501022 |  0.0 |  0.0 | 0.501022 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 12.618238 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.39926 |  3.505066 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 | 0.992414 | 0.496207 |  0.0 |  0.0 | 0.496207 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.01908 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.395287 |    0.0053 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.98288 |  0.49144 |  0.0 |  0.0 |  0.49144 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.739914 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.391354 |  0.205532 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 | 0.973441 |  0.48672 |  0.0 |  0.0 |  0.48672 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   3.50074 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.38746 |  0.972428 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.964094 | 0.482047 |  0.0 |  0.0 | 0.482047 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.421557 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.383605 |  0.672655 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 |  0.95484 |  0.47742 |  0.0 |  0.0 |  0.47742 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 12.382368 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.379788 |  3.439547 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.945677 | 0.472838 |  0.0 |  0.0 | 0.472838 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   4.10317 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.376009 |   1.13977 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.936604 | 0.468302 |  0.0 |  0.0 | 0.468302 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.983965 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.372267 |  1.106657 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.927987 | 0.463994 |  0.0 |  0.0 | 0.463994 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   1.34457 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.368563 |  0.373492 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.923766 | 0.461883 |  0.0 |  0.0 | 0.461883 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.023013 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.364896 |  0.006393 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 |  0.92671 | 0.463355 |  0.0 |  0.0 | 0.361265 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.361265 |       0.0 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.925166 | 0.462583 |  0.0 |  0.0 | 0.357671 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.357671 |       0.0 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 | 0.925758 | 0.462879 |  0.0 |  0.0 | 0.462879 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.371232 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.354112 |   0.10312 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 0.941365 | 0.470682 |  0.0 |  0.0 | 0.470588 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.350588 |       0.0 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 |  0.94545 | 0.472725 |  0.0 |  0.0 | 0.472725 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.194375 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   0.3471 |  1.165104 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 0.971282 | 0.485641 |  0.0 |  0.0 | 0.485641 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.938005 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.343646 |  1.927224 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 | 0.977753 | 0.488877 |  0.0 |  0.0 | 0.488877 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   1.17135 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.340227 |  0.325375 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.976419 | 0.488209 |  0.0 |  0.0 | 0.488209 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 24.688632 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.336841 |  6.857953 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.975611 | 0.487806 |  0.0 |  0.0 | 0.487806 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 45.325684 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.33349 | 12.590468 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 0.977367 | 0.488683 |  0.0 |  0.0 | 0.488683 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  9.681488 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.330172 |  2.689302 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.973959 | 0.486979 |  0.0 |  0.0 | 0.486979 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.159907 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.326886 |   1.15553 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.965386 | 0.482693 |  0.0 |  0.0 | 0.482693 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.840941 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.323634 |  2.455817 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.956102 | 0.478051 |  0.0 |  0.0 | 0.478051 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 22.042362 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.320413 |  6.122878 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 | 0.946911 | 0.473456 |  0.0 |  0.0 | 0.473456 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  18.32377 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.317225 |  5.089936 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.937807 | 0.468904 |  0.0 |  0.0 | 0.468904 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  7.405165 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.314069 |   2.05699 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.928788 | 0.464394 |  0.0 |  0.0 | 0.464394 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   2.12655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.310944 |  0.590708 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.919859 |  0.45993 |  0.0 |  0.0 |  0.45993 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   5.72792 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.30785 |  1.591089 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.911019 |  0.45551 |  0.0 |  0.0 |  0.45551 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.089277 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.304787 |  0.858133 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.902264 | 0.451132 |  0.0 |  0.0 | 0.451132 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.450622 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.301754 |  0.125173 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.893598 | 0.446799 |  0.0 |  0.0 | 0.446799 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.091952 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.298752 |  0.025542 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.885023 | 0.442512 |  0.0 |  0.0 | 0.442512 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.453267 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.295779 |  0.125908 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.876539 |  0.43827 |  0.0 |  0.0 |  0.43827 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.734566 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.292836 |  0.759602 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.868393 | 0.434197 |  0.0 |  0.0 | 0.434197 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.335725 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.289922 |  0.093257 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.861591 | 0.430795 |  0.0 |  0.0 | 0.430795 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.096242 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.287037 |  0.026734 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.85767 | 0.428835 |  0.0 |  0.0 | 0.284181 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.284181 |       0.0 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 0.865902 | 0.432951 |  0.0 |  0.0 | 0.281354 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.281354 |       0.0 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 0.874682 | 0.437341 |  0.0 |  0.0 | 0.437341 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.201213 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.278554 |  0.055892 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 0.887158 | 0.443579 |  0.0 |  0.0 | 0.443579 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.952203 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.275782 |  0.820056 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 0.899791 | 0.449895 |  0.0 |  0.0 | 0.449895 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.663143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.273038 |  0.184206 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 0.909429 | 0.454714 |  0.0 |  0.0 | 0.454714 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.175607 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.270322 |   0.04878 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 0.931493 | 0.465746 |  0.0 |  0.0 | 0.465746 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.161885 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.267632 |  0.044968 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 0.941584 | 0.470792 |  0.0 |  0.0 | 0.264969 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.264969 |       0.0 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 0.953396 | 0.476698 |  0.0 |  0.0 | 0.262332 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.262332 |       0.0 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 0.957006 | 0.478503 |  0.0 |  0.0 | 0.259722 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.259722 |       0.0 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.953461 |  0.47673 |  0.0 |  0.0 | 0.257138 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.257138 |       0.0 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.945025 | 0.472513 |  0.0 |  0.0 | 0.254579 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.254579 |       0.0 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.935925 | 0.467963 |  0.0 |  0.0 | 0.252046 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.252046 |       0.0 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.926909 | 0.463455 |  0.0 |  0.0 | 0.249538 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.249538 |       0.0 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.917977 | 0.458989 |  0.0 |  0.0 | 0.247055 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.247055 |       0.0 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.909131 | 0.454565 |  0.0 |  0.0 | 0.244597 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.244597 |       0.0 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.900368 | 0.450184 |  0.0 |  0.0 | 0.242163 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.242163 |       0.0 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 | 0.891703 | 0.445851 |  0.0 |  0.0 | 0.239754 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.239754 |       0.0 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.883125 | 0.441563 |  0.0 |  0.0 | 0.237368 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.237368 |       0.0 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.874628 | 0.437314 |  0.0 |  0.0 | 0.437314 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.357692 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.235006 |  0.377137 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 | 0.866216 | 0.433108 |  0.0 |  0.0 | 0.232668 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.232668 |       0.0 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.857893 | 0.428946 |  0.0 |  0.0 | 0.230353 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.230353 |       0.0 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.850555 | 0.425277 |  0.0 |  0.0 | 0.228061 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.228061 |       0.0 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.847617 | 0.423809 |  0.0 |  0.0 | 0.423809 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.641983 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.225792 |  0.178329 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 | 0.851924 | 0.425962 |  0.0 |  0.0 | 0.425962 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.277583 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.223545 |  0.077106 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 | 0.864172 | 0.432086 |  0.0 |  0.0 | 0.341321 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.221321 |       0.0 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 |  0.88415 | 0.442075 |  0.0 |  0.0 | 0.442075 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.257043 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.219118 |  0.071401 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 0.908874 | 0.454437 |  0.0 |  0.0 | 0.216938 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.216938 |       0.0 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 0.933351 | 0.466675 |  0.0 |  0.0 |  0.21478 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.21478 |       0.0 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 0.966878 | 0.483439 |  0.0 |  0.0 | 0.212642 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.212642 |       0.0 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 0.994743 | 0.497372 |  0.0 |  0.0 | 0.210527 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.210527 |       0.0 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 1.016243 | 0.508122 |  0.0 |  0.0 | 0.208432 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.208432 |       0.0 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 1.031274 | 0.515637 |  0.0 |  0.0 | 0.206358 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.206358 |       0.0 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.032755 | 0.516378 |  0.0 |  0.0 | 0.204305 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.204305 |       0.0 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 1.027664 | 0.513832 |  0.0 |  0.0 | 0.202272 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.202272 |       0.0 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 | 1.018706 | 0.509353 |  0.0 |  0.0 | 0.200259 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.200259 |       0.0 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 1.008877 | 0.504439 |  0.0 |  0.0 | 0.198267 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.198267 |       0.0 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.999131 | 0.499566 |  0.0 |  0.0 | 0.196294 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.196294 |       0.0 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 | 0.989478 | 0.494739 |  0.0 |  0.0 | 0.194341 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.194341 |       0.0 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.979923 | 0.489961 |  0.0 |  0.0 | 0.192407 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.192407 |       0.0 |

    .. raw:: html

        <iframe
            src="lland_v2_ex2_1a.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    >>> wfet0(1.0)
    >>> test.inits.wet0 = 0.0

    :ref:`Recalculation of example 2.2 <lland_v1_ex2_2>`

    The following calculation shows, that the outflow values of the
    integration test for water areas of type |WASSER| are reproduced
    exactly (when parameter |NegQ| is set to `True`):

    >>> negq(True)
    >>> test('lland_v2_ex2_2')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo | nbes | sbes |      evi | evb | wgtf | schm | wada | qdb | qib1 | qib2 | qbb | qdgz |         q | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 |     qbga |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.040283 | 0.020141 |  0.0 |  0.0 | 0.020141 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.477367 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.497508 |  0.132602 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.03912 |  0.01956 |  0.0 |  0.0 |  0.01956 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.472998 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.492558 |  0.131388 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.468261 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.487657 |  0.130072 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 | 0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.463607 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.482805 |   0.12878 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.458605 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.478001 |   0.12739 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.041104 | 0.020552 |  0.0 |  0.0 | 0.020552 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.452692 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.473245 |  0.125748 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.116763 | 0.058382 |  0.0 |  0.0 | 0.058382 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.410154 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.468536 |  0.113932 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 |  0.77315 | 0.386575 |  0.0 |  0.0 | 0.386575 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.077299 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.463874 |  0.021472 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 1.747814 | 0.873907 |  0.0 |  0.0 | 0.873907 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.414649 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.459258 |  -0.11518 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 2.927022 | 1.463511 |  0.0 |  0.0 | 1.463511 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.008823 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.454688 | -0.280229 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 3.305745 | 1.652872 |  0.0 |  0.0 | 1.652872 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.202708 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.450164 | -0.334086 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 3.747947 | 1.873973 |  0.0 |  0.0 | 1.873973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.428288 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.445685 | -0.396747 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 2.050471 | 1.025236 |  0.0 |  0.0 | 1.025236 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.583985 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.44125 | -0.162218 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 2.650012 | 1.325006 |  0.0 |  0.0 | 1.325006 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.888146 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.43686 | -0.246707 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 2.959048 | 1.479524 |  0.0 |  0.0 | 1.479524 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.047011 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.432513 | -0.290836 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 0.956604 | 0.478302 |  0.0 |  0.0 | 0.478302 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.189907 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.428209 |  0.052752 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 0.731934 | 0.365967 |  0.0 |  0.0 | 0.365967 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.057982 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.423949 |  0.016106 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 0.454628 | 0.227314 |  0.0 |  0.0 | 0.227314 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.192416 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.41973 |  0.053449 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 0.300394 | 0.150197 |  0.0 |  0.0 | 0.150197 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.825357 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.415554 |  0.507044 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 0.089558 | 0.044779 |  0.0 |  0.0 | 0.044779 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   7.08664 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.411419 |  1.968511 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 |  0.03977 | 0.019885 |  0.0 |  0.0 | 0.019885 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   3.86744 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.407325 |  1.074289 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 | 0.039382 | 0.019691 |  0.0 |  0.0 | 0.019691 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.263582 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.403272 |  1.739884 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 0.039251 | 0.019626 |  0.0 |  0.0 | 0.019626 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 13.099634 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.39926 |  3.638787 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.495792 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.395287 |   0.13772 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.211859 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.391354 |  0.336628 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.967965 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.38746 |  1.102213 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.884208 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.383605 |  0.801169 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 | 0.038661 |  0.01933 |  0.0 |  0.0 |  0.01933 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 12.840457 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.379788 |  3.566794 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.038528 | 0.019264 |  0.0 |  0.0 | 0.019264 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.556744 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.376009 |  1.265762 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 | 0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.433069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.372267 |  1.231408 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.074919 |  0.03746 |  0.0 |  0.0 |  0.03746 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.771104 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.368563 |  0.491973 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.505845 | 0.252922 |  0.0 |  0.0 | 0.252922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.231974 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.364896 |  0.064437 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 | 1.218215 | 0.609108 |  0.0 |  0.0 | 0.609108 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.247842 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.361265 | -0.068845 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.772303 | 0.386152 |  0.0 |  0.0 | 0.386152 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.028481 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.357671 | -0.007911 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 |   0.9844 |   0.4922 |  0.0 |  0.0 |   0.4922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.341911 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.354112 |  0.094975 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 2.486378 | 1.243189 |  0.0 |  0.0 | 1.243189 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.772601 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.350588 | -0.214611 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 | 1.349913 | 0.674957 |  0.0 |  0.0 | 0.674957 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.992143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   0.3471 |  1.108929 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 3.528622 | 1.764311 |  0.0 |  0.0 | 1.764311 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.659335 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.343646 |  1.572038 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 |  1.61841 | 0.809205 |  0.0 |  0.0 | 0.809205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.851022 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.340227 |  0.236395 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.844303 | 0.422151 |  0.0 |  0.0 | 0.422151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  24.75469 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.336841 |  6.876303 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.895703 | 0.447851 |  0.0 |  0.0 | 0.447851 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 45.365638 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.33349 | 12.601566 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 1.151139 |  0.57557 |  0.0 |  0.0 |  0.57557 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  9.594602 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.330172 |  2.665167 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.636581 |  0.31829 |  0.0 |  0.0 |  0.31829 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.328596 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.326886 |  1.202388 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.116642 | 0.058321 |  0.0 |  0.0 | 0.058321 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  9.265313 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.323634 |  2.573698 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.037049 | 0.018524 |  0.0 |  0.0 | 0.018524 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 22.501889 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.320413 |  6.250525 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 |  0.03698 |  0.01849 |  0.0 |  0.0 |  0.01849 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 18.778735 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.317225 |  5.216315 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.036499 |  0.01825 |  0.0 |  0.0 |  0.01825 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  7.855819 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.314069 |  2.182172 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 | 0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.573007 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.310944 |  0.714724 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.035943 | 0.017972 |  0.0 |  0.0 | 0.017972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.169878 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.30785 |  1.713855 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 | 0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   3.52685 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.304787 |  0.979681 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.035522 | 0.017761 |  0.0 |  0.0 | 0.017761 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.883993 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.301754 |  0.245554 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.035663 | 0.017831 |  0.0 |  0.0 | 0.017831 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.52092 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.298752 |    0.1447 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.036083 | 0.018041 |  0.0 |  0.0 | 0.018041 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.877738 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.295779 |  0.243816 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.036637 | 0.018319 |  0.0 |  0.0 | 0.018319 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.154517 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.292836 |  0.876255 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.061944 | 0.030972 |  0.0 |  0.0 | 0.030972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.73895 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.289922 |  0.205264 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.188148 | 0.094074 |  0.0 |  0.0 | 0.094074 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.432964 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.287037 |  0.120268 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.46949 | 0.234745 |  0.0 |  0.0 | 0.234745 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.049436 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.284181 |  0.013732 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 1.680918 | 0.840459 |  0.0 |  0.0 | 0.840459 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.559106 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.281354 | -0.155307 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 1.743886 | 0.871943 |  0.0 |  0.0 | 0.871943 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.233389 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.278554 |  -0.06483 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 2.122301 | 1.061151 |  0.0 |  0.0 | 1.061151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.334632 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.275782 |  0.648509 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 2.150411 | 1.075205 |  0.0 |  0.0 | 1.075205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.037833 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.273038 |  0.010509 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 1.863566 | 0.931783 |  0.0 |  0.0 | 0.931783 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.301461 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.270322 | -0.083739 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 3.115838 | 1.557919 |  0.0 |  0.0 | 1.557919 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.930287 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.267632 | -0.258413 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 1.940627 | 0.970313 |  0.0 |  0.0 | 0.970313 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.705345 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.264969 | -0.195929 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 2.122768 | 1.061384 |  0.0 |  0.0 | 1.061384 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.799052 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.262332 | -0.221959 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 1.314414 | 0.657207 |  0.0 |  0.0 | 0.657207 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.397485 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.259722 | -0.110413 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.602486 | 0.301243 |  0.0 |  0.0 | 0.301243 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.044105 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.257138 | -0.012251 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.109905 | 0.054952 |  0.0 |  0.0 | 0.054952 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.199627 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.254579 |  0.055452 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.035026 | 0.017513 |  0.0 |  0.0 | 0.017513 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.234533 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.252046 |  0.065148 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 | 0.017154 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.232384 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.249538 |  0.064551 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.033727 | 0.016863 |  0.0 |  0.0 | 0.016863 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.230192 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.247055 |  0.063942 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.033286 | 0.016643 |  0.0 |  0.0 | 0.016643 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.227954 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.244597 |  0.063321 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.032916 | 0.016458 |  0.0 |  0.0 | 0.016458 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.225705 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.242163 |  0.062696 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |   0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.222854 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.239754 |  0.061904 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.220395 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.237368 |  0.061221 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.033434 | 0.016717 |  0.0 |  0.0 | 0.016717 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.778289 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.235006 |  0.493969 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 |  0.03336 |  0.01668 |  0.0 |  0.0 |  0.01668 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.215988 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.232668 |  0.059997 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.21338 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.230353 |  0.059272 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.124092 | 0.062046 |  0.0 |  0.0 | 0.062046 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.166015 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.228061 |  0.046115 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.556783 | 0.278392 |  0.0 |  0.0 | 0.278392 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.7874 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.225792 |  0.218722 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 |  1.27835 | 0.639175 |  0.0 |  0.0 | 0.639175 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.06437 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.223545 |   0.01788 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 |  2.07666 |  1.03833 |  0.0 |  0.0 |  1.03833 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.69701 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.221321 | -0.193614 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 | 2.862014 | 1.431007 |  0.0 |  0.0 | 1.431007 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.731889 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.219118 | -0.203302 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 3.356573 | 1.678286 |  0.0 |  0.0 | 1.678286 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.461348 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.216938 |  -0.40593 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 3.356514 | 1.678257 |  0.0 |  0.0 | 1.678257 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.463478 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.21478 | -0.406522 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 4.286095 | 2.143048 |  0.0 |  0.0 | 2.143048 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.930405 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.212642 | -0.536224 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 3.753358 | 1.876679 |  0.0 |  0.0 | 1.876679 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.666152 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.210527 |  -0.46282 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 3.144766 | 1.572383 |  0.0 |  0.0 | 1.572383 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.363951 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.208432 | -0.378875 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 2.519332 | 1.259666 |  0.0 |  0.0 | 1.259666 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.053308 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.206358 | -0.292586 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.179366 | 0.589683 |  0.0 |  0.0 | 0.589683 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.385379 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.204305 |  -0.10705 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 0.523693 | 0.261846 |  0.0 |  0.0 | 0.261846 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.059575 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.202272 | -0.016549 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 |  0.13182 |  0.06591 |  0.0 |  0.0 |  0.06591 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.134349 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.200259 |  0.037319 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 0.035803 | 0.017902 |  0.0 |  0.0 | 0.017902 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.180365 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.198267 |  0.050101 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 | 0.017154 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.17914 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.196294 |  0.049761 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |   0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.177441 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.194341 |  0.049289 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.175434 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 0.192407 |  0.048732 |

    .. raw:: html

        <iframe
            src="lland_v2_ex2_2.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    >>> negq(False)

    :ref:`Recalculation of example 3 <lland_v1_ex3>`

    The following calculation shows, that the outflow values of the
    integration test for water areas of type |SEE| are reproduced
    exactly:

    >>> lnk(SEE)
    >>> test('lland_v2_ex3')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo | nbes | sbes |      evi | evb | wgtf | schm | wada | qdb | qib1 | qib2 | qbb | qdgz |        q | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |     qbga |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.040283 | 0.020141 |  0.0 |  0.0 | 0.020141 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.497408 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.020141 |   0.0 |   0.0 |   0.0 |   0.0 | 0.497408 | 0.138169 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.03912 |  0.01956 |  0.0 |  0.0 |  0.01956 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.492261 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.01956 |   0.0 |   0.0 |   0.0 |   0.0 | 0.492261 | 0.136739 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.487169 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019396 |   0.0 |   0.0 |   0.0 |   0.0 | 0.487169 | 0.135325 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 | 0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.48213 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019198 |   0.0 |   0.0 |   0.0 |   0.0 |  0.48213 | 0.133925 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.477141 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019396 |   0.0 |   0.0 |   0.0 |   0.0 | 0.477141 | 0.132539 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.041104 | 0.020552 |  0.0 |  0.0 | 0.020552 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.472194 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.020552 |   0.0 |   0.0 |   0.0 |   0.0 | 0.472194 | 0.131165 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.116763 | 0.058382 |  0.0 |  0.0 | 0.058382 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.467103 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058382 |   0.0 |   0.0 |   0.0 |   0.0 | 0.467103 | 0.129751 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 |  0.77315 | 0.386575 |  0.0 |  0.0 | 0.386575 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.460239 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.386575 |   0.0 |   0.0 |   0.0 |   0.0 | 0.460239 | 0.127844 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 1.747814 | 0.873907 |  0.0 |  0.0 | 0.873907 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.449384 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.873907 |   0.0 |   0.0 |   0.0 |   0.0 | 0.449384 | 0.124829 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 2.927022 | 1.463511 |  0.0 |  0.0 | 1.463511 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.433279 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.463511 |   0.0 |   0.0 |   0.0 |   0.0 | 0.433279 | 0.120355 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 3.305745 | 1.652872 |  0.0 |  0.0 | 1.652872 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.413462 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.652872 |   0.0 |   0.0 |   0.0 |   0.0 | 0.413462 | 0.114851 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 3.747947 | 1.873973 |  0.0 |  0.0 | 1.873973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   0.3918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.873973 |   0.0 |   0.0 |   0.0 |   0.0 |   0.3918 | 0.108833 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 2.050471 | 1.025236 |  0.0 |  0.0 | 1.025236 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.373484 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.025236 |   0.0 |   0.0 |   0.0 |   0.0 | 0.373484 | 0.103746 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 2.650012 | 1.325006 |  0.0 |  0.0 | 1.325006 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.358073 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.325006 |   0.0 |   0.0 |   0.0 |   0.0 | 0.358073 | 0.099465 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 2.959048 | 1.479524 |  0.0 |  0.0 | 1.479524 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.340556 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.479524 |   0.0 |   0.0 |   0.0 |   0.0 | 0.340556 | 0.094599 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 0.956604 | 0.478302 |  0.0 |  0.0 | 0.478302 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.328631 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.238302 |   0.0 |   0.0 |   0.0 |   0.0 | 0.328631 | 0.091287 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 0.731934 | 0.365967 |  0.0 |  0.0 | 0.365967 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.322354 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.365967 |   0.0 |   0.0 |   0.0 |   0.0 | 0.322354 | 0.089543 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 0.454628 | 0.227314 |  0.0 |  0.0 | 0.227314 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.316196 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.227314 |   0.0 |   0.0 |   0.0 |   0.0 | 0.316196 | 0.087832 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 0.300394 | 0.150197 |  0.0 |  0.0 | 0.150197 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.318947 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.409803 |   0.0 |   0.0 |   0.0 |   0.0 | 0.318947 | 0.088596 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 0.089558 | 0.044779 |  0.0 |  0.0 | 0.044779 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.35604 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  6.675221 |   0.0 |   0.0 |   0.0 |   0.0 |  0.35604 |   0.0989 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 |  0.03977 | 0.019885 |  0.0 |  0.0 | 0.019885 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.402895 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.460115 |   0.0 |   0.0 |   0.0 |   0.0 | 0.402895 | 0.111915 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 | 0.039382 | 0.019691 |  0.0 |  0.0 | 0.019691 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.445276 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.860309 |   0.0 |   0.0 |   0.0 |   0.0 | 0.445276 | 0.123688 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 0.039251 | 0.019626 |  0.0 |  0.0 | 0.019626 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.533243 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 12.700374 |   0.0 |   0.0 |   0.0 |   0.0 | 0.533243 | 0.148123 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.591518 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.100505 |   0.0 |   0.0 |   0.0 |   0.0 | 0.591518 | 0.164311 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.590221 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.820505 |   0.0 |   0.0 |   0.0 |   0.0 | 0.590221 |  0.16395 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 | 0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.606266 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.580505 |   0.0 |   0.0 |   0.0 |   0.0 | 0.606266 | 0.168407 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 | 0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.630479 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.500604 |   0.0 |   0.0 |   0.0 |   0.0 | 0.630479 | 0.175133 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 | 0.038661 |  0.01933 |  0.0 |  0.0 |  0.01933 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.698722 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  12.46067 |   0.0 |   0.0 |   0.0 |   0.0 | 0.698722 | 0.194089 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.038528 | 0.019264 |  0.0 |  0.0 | 0.019264 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.774493 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  4.180736 |   0.0 |   0.0 |   0.0 |   0.0 | 0.774493 | 0.215137 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 | 0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.807788 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  4.060802 |   0.0 |   0.0 |   0.0 |   0.0 | 0.807788 | 0.224386 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.074919 |  0.03746 |  0.0 |  0.0 |  0.03746 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.826909 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   1.40254 |   0.0 |   0.0 |   0.0 |   0.0 | 0.826909 | 0.229697 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.505845 | 0.252922 |  0.0 |  0.0 | 0.252922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.824985 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.132922 |   0.0 |   0.0 |   0.0 |   0.0 | 0.824985 | 0.229162 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 | 1.218215 | 0.609108 |  0.0 |  0.0 | 0.609108 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.81308 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.609108 |   0.0 |   0.0 |   0.0 |   0.0 |  0.81308 | 0.225856 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.772303 | 0.386152 |  0.0 |  0.0 | 0.386152 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.80004 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.386152 |   0.0 |   0.0 |   0.0 |   0.0 |  0.80004 | 0.222233 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 |   0.9844 |   0.4922 |  0.0 |  0.0 |   0.4922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.790101 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0122 |   0.0 |   0.0 |   0.0 |   0.0 | 0.790101 | 0.219473 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 2.486378 | 1.243189 |  0.0 |  0.0 | 1.243189 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.776582 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.123189 |   0.0 |   0.0 |   0.0 |   0.0 | 0.776582 | 0.215717 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 | 1.349913 | 0.674957 |  0.0 |  0.0 | 0.674957 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.78144 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.645043 |   0.0 |   0.0 |   0.0 |   0.0 |  0.78144 | 0.217067 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 3.528622 | 1.764311 |  0.0 |  0.0 | 1.764311 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.818259 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.315689 |   0.0 |   0.0 |   0.0 |   0.0 | 0.818259 | 0.227294 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 |  1.61841 | 0.809205 |  0.0 |  0.0 | 0.809205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.839065 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.510795 |   0.0 |   0.0 |   0.0 |   0.0 | 0.839065 | 0.233074 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.844303 | 0.422151 |  0.0 |  0.0 | 0.422151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 0.954936 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 24.417849 |   0.0 |   0.0 |   0.0 |   0.0 | 0.954936 |  0.26526 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.895703 | 0.447851 |  0.0 |  0.0 | 0.447851 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.291125 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 45.032149 |   0.0 |   0.0 |   0.0 |   0.0 | 1.291125 | 0.358646 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 1.151139 |  0.57557 |  0.0 |  0.0 |  0.57557 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.548111 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   9.26443 |   0.0 |   0.0 |   0.0 |   0.0 | 1.548111 | 0.430031 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.636581 |  0.31829 |  0.0 |  0.0 |  0.31829 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.598664 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   4.00171 |   0.0 |   0.0 |   0.0 |   0.0 | 1.598664 | 0.444073 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.116642 | 0.058321 |  0.0 |  0.0 | 0.058321 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.647192 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.941679 |   0.0 |   0.0 |   0.0 |   0.0 | 1.647192 | 0.457553 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.037049 | 0.018524 |  0.0 |  0.0 | 0.018524 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.785753 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 22.181476 |   0.0 |   0.0 |   0.0 |   0.0 | 1.785753 | 0.496042 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 |  0.03698 |  0.01849 |  0.0 |  0.0 |  0.01849 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.970156 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  18.46151 |   0.0 |   0.0 |   0.0 |   0.0 | 1.970156 | 0.547265 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.036499 |  0.01825 |  0.0 |  0.0 |  0.01825 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.07983 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   7.54175 |   0.0 |   0.0 |   0.0 |   0.0 |  2.07983 | 0.577731 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 | 0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.107866 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.262063 |   0.0 |   0.0 |   0.0 |   0.0 | 2.107866 | 0.585518 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.035943 | 0.017972 |  0.0 |  0.0 | 0.017972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.127341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.862028 |   0.0 |   0.0 |   0.0 |   0.0 | 2.127341 | 0.590928 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 | 0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.151345 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.222063 |   0.0 |   0.0 |   0.0 |   0.0 | 2.151345 | 0.597596 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.035522 | 0.017761 |  0.0 |  0.0 | 0.017761 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.148844 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.582239 |   0.0 |   0.0 |   0.0 |   0.0 | 2.148844 | 0.596901 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.035663 | 0.017831 |  0.0 |  0.0 | 0.017831 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.131462 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.222169 |   0.0 |   0.0 |   0.0 |   0.0 | 2.131462 | 0.592073 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.036083 | 0.018041 |  0.0 |  0.0 | 0.018041 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.114257 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.581959 |   0.0 |   0.0 |   0.0 |   0.0 | 2.114257 | 0.587294 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.036637 | 0.018319 |  0.0 |  0.0 | 0.018319 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.110371 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.861681 |   0.0 |   0.0 |   0.0 |   0.0 | 2.110371 | 0.586214 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.061944 | 0.030972 |  0.0 |  0.0 | 0.030972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.105823 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.449028 |   0.0 |   0.0 |   0.0 |   0.0 | 2.105823 | 0.584951 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.188148 | 0.094074 |  0.0 |  0.0 | 0.094074 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.087828 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.145926 |   0.0 |   0.0 |   0.0 |   0.0 | 2.087828 | 0.579952 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.46949 | 0.234745 |  0.0 |  0.0 | 0.234745 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.066608 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.234745 |   0.0 |   0.0 |   0.0 |   0.0 | 2.066608 | 0.574058 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 1.680918 | 0.840459 |  0.0 |  0.0 | 0.840459 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.040691 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.840459 |   0.0 |   0.0 |   0.0 |   0.0 | 2.040691 | 0.566859 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 1.743886 | 0.871943 |  0.0 |  0.0 | 0.871943 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.01366 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.511943 |   0.0 |   0.0 |   0.0 |   0.0 |  2.01366 |  0.55935 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 2.122301 | 1.061151 |  0.0 |  0.0 | 1.061151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 2.001341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.058849 |   0.0 |   0.0 |   0.0 |   0.0 | 2.001341 | 0.555928 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 2.150411 | 1.075205 |  0.0 |  0.0 | 1.075205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.990481 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.235205 |   0.0 |   0.0 |   0.0 |   0.0 | 1.990481 | 0.552911 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 1.863566 | 0.931783 |  0.0 |  0.0 | 0.931783 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.966658 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.571783 |   0.0 |   0.0 |   0.0 |   0.0 | 1.966658 | 0.546294 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 3.115838 | 1.557919 |  0.0 |  0.0 | 1.557919 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.93828 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.197919 |   0.0 |   0.0 |   0.0 |   0.0 |  1.93828 | 0.538411 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 1.940627 | 0.970313 |  0.0 |  0.0 | 0.970313 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.908208 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.970313 |   0.0 |   0.0 |   0.0 |   0.0 | 1.908208 | 0.530058 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 2.122768 | 1.061384 |  0.0 |  0.0 | 1.061384 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.879113 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.061384 |   0.0 |   0.0 |   0.0 |   0.0 | 1.879113 | 0.521976 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 1.314414 | 0.657207 |  0.0 |  0.0 | 0.657207 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.851869 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.657207 |   0.0 |   0.0 |   0.0 |   0.0 | 1.851869 | 0.514408 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.602486 | 0.301243 |  0.0 |  0.0 | 0.301243 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.828677 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.301243 |   0.0 |   0.0 |   0.0 |   0.0 | 1.828677 | 0.507966 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.109905 | 0.054952 |  0.0 |  0.0 | 0.054952 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.808711 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.054952 |   0.0 |   0.0 |   0.0 |   0.0 | 1.808711 |  0.50242 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.035026 | 0.017513 |  0.0 |  0.0 | 0.017513 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.790354 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017513 |   0.0 |   0.0 |   0.0 |   0.0 | 1.790354 | 0.497321 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 | 0.017154 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.772367 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017154 |   0.0 |   0.0 |   0.0 |   0.0 | 1.772367 | 0.492324 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.033727 | 0.016863 |  0.0 |  0.0 | 0.016863 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.754562 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016863 |   0.0 |   0.0 |   0.0 |   0.0 | 1.754562 | 0.487378 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.033286 | 0.016643 |  0.0 |  0.0 | 0.016643 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.736938 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016643 |   0.0 |   0.0 |   0.0 |   0.0 | 1.736938 | 0.482483 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.032916 | 0.016458 |  0.0 |  0.0 | 0.016458 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.71949 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016458 |   0.0 |   0.0 |   0.0 |   0.0 |  1.71949 | 0.477636 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |   0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.702215 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0169 |   0.0 |   0.0 |   0.0 |   0.0 | 1.702215 | 0.472837 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.685109 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 | 1.685109 | 0.468086 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.033434 | 0.016717 |  0.0 |  0.0 | 0.016717 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.675948 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.543283 |   0.0 |   0.0 |   0.0 |   0.0 | 1.675948 | 0.465541 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 |  0.03336 |  0.01668 |  0.0 |  0.0 |  0.01668 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.666854 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.01668 |   0.0 |   0.0 |   0.0 |   0.0 | 1.666854 | 0.463015 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.650102 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 | 1.650102 | 0.458362 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.124092 | 0.062046 |  0.0 |  0.0 | 0.062046 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.633289 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.062046 |   0.0 |   0.0 |   0.0 |   0.0 | 1.633289 | 0.453691 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.556783 | 0.278392 |  0.0 |  0.0 | 0.278392 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.619528 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.561608 |   0.0 |   0.0 |   0.0 |   0.0 | 1.619528 | 0.449869 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 |  1.27835 | 0.639175 |  0.0 |  0.0 | 0.639175 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.60541 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.159175 |   0.0 |   0.0 |   0.0 |   0.0 |  1.60541 | 0.445947 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 |  2.07666 |  1.03833 |  0.0 |  0.0 |  1.03833 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.584069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.91833 |   0.0 |   0.0 |   0.0 |   0.0 | 1.584069 | 0.440019 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 | 2.862014 | 1.431007 |  0.0 |  0.0 | 1.431007 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.559007 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.951007 |   0.0 |   0.0 |   0.0 |   0.0 | 1.559007 | 0.433057 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 3.356573 | 1.678286 |  0.0 |  0.0 | 1.678286 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.530407 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.678286 |   0.0 |   0.0 |   0.0 |   0.0 | 1.530407 | 0.425113 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 3.356514 | 1.678257 |  0.0 |  0.0 | 1.678257 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.49848 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.678257 |   0.0 |   0.0 |   0.0 |   0.0 |  1.49848 | 0.416245 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 4.286095 | 2.143048 |  0.0 |  0.0 | 2.143048 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.464555 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.143048 |   0.0 |   0.0 |   0.0 |   0.0 | 1.464555 | 0.406821 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 3.753358 | 1.876679 |  0.0 |  0.0 | 1.876679 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.429986 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.876679 |   0.0 |   0.0 |   0.0 |   0.0 | 1.429986 | 0.397218 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 3.144766 | 1.572383 |  0.0 |  0.0 | 1.572383 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.398601 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.572383 |   0.0 |   0.0 |   0.0 |   0.0 | 1.398601 |   0.3885 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 2.519332 | 1.259666 |  0.0 |  0.0 | 1.259666 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.370597 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.259666 |   0.0 |   0.0 |   0.0 |   0.0 | 1.370597 | 0.380722 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.179366 | 0.589683 |  0.0 |  0.0 | 0.589683 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.347765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.589683 |   0.0 |   0.0 |   0.0 |   0.0 | 1.347765 | 0.374379 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 0.523693 | 0.261846 |  0.0 |  0.0 | 0.261846 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.330121 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.261846 |   0.0 |   0.0 |   0.0 |   0.0 | 1.330121 | 0.369478 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 |  0.13182 |  0.06591 |  0.0 |  0.0 |  0.06591 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.315257 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06591 |   0.0 |   0.0 |   0.0 |   0.0 | 1.315257 | 0.365349 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 0.035803 | 0.017902 |  0.0 |  0.0 | 0.017902 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.301753 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017902 |   0.0 |   0.0 |   0.0 |   0.0 | 1.301753 | 0.361598 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 | 0.017154 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.288626 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017154 |   0.0 |   0.0 |   0.0 |   0.0 | 1.288626 | 0.357952 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |   0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.275634 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0169 |   0.0 |   0.0 |   0.0 |   0.0 | 1.275634 | 0.354343 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | 0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 1.262773 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 | 1.262773 |  0.35077 |

    .. raw:: html

        <iframe
            src="lland_v2_ex3.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    :ref:`Recalculation of example 4 <lland_v1_ex4>`

    The following calculation shows, that the outflow values of the
    integration test for water areas of type |FLUSS| are reproduced
    exactly:

    >>> lnk(FLUSS)
    >>> test('lland_v2_ex4')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo | nbes | sbes |       evi | evb | wgtf | schm | wada | qdb | qib1 | qib2 | qbb |      qdgz |         q | inzp | wats | waes | bowa |     qdgz1 |     qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |     qdga2 | qiga1 | qiga2 |     qbga |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.040283 | 0.020141 |  0.0 |  0.0 |  0.020141 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.020141 |  0.495622 |  0.0 |  0.0 |  0.0 |  0.0 | -0.020141 |       0.0 |   0.0 |   0.0 |  0.0 | -0.001886 |       0.0 |   0.0 |   0.0 | 0.497508 | 0.137673 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.03912 |  0.01956 |  0.0 |  0.0 |   0.01956 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  -0.01956 |  0.487417 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.01956 |       0.0 |   0.0 |   0.0 |  0.0 | -0.005141 |       0.0 |   0.0 |   0.0 | 0.492558 | 0.135394 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 |  0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.019396 |  0.479918 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019396 |       0.0 |   0.0 |   0.0 |  0.0 | -0.007739 |       0.0 |   0.0 |   0.0 | 0.487657 |  0.13331 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 |  0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.019198 |  0.472971 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019198 |       0.0 |   0.0 |   0.0 |  0.0 | -0.009834 |       0.0 |   0.0 |   0.0 | 0.482805 | 0.131381 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 |  0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.019396 |  0.466451 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019396 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.01155 |       0.0 |   0.0 |   0.0 | 0.478001 |  0.12957 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.041104 | 0.020552 |  0.0 |  0.0 |  0.020552 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.020552 |  0.460164 |  0.0 |  0.0 |  0.0 |  0.0 | -0.020552 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.01308 |       0.0 |   0.0 |   0.0 | 0.473245 | 0.127823 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.116763 | 0.058382 |  0.0 |  0.0 |  0.058382 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.058382 |  0.450558 |  0.0 |  0.0 |  0.0 |  0.0 | -0.058382 |       0.0 |   0.0 |   0.0 |  0.0 | -0.017978 |       0.0 |   0.0 |   0.0 | 0.468536 | 0.125155 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 |  0.77315 | 0.386575 |  0.0 |  0.0 |  0.386575 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.386575 |  0.407835 |  0.0 |  0.0 |  0.0 |  0.0 | -0.386575 |       0.0 |   0.0 |   0.0 |  0.0 | -0.056038 |       0.0 |   0.0 |   0.0 | 0.463874 | 0.113288 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 1.747814 | 0.873907 |  0.0 |  0.0 |  0.873907 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.873907 |  0.297663 |  0.0 |  0.0 |  0.0 |  0.0 | -0.873907 |       0.0 |   0.0 |   0.0 |  0.0 | -0.161595 |       0.0 |   0.0 |   0.0 | 0.459258 | 0.082684 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 2.927022 | 1.463511 |  0.0 |  0.0 |  1.463511 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.463511 |  0.108755 |  0.0 |  0.0 |  0.0 |  0.0 | -1.463511 |       0.0 |   0.0 |   0.0 |  0.0 | -0.345934 |       0.0 |   0.0 |   0.0 | 0.454688 |  0.03021 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 3.305745 | 1.652872 |  0.0 |  0.0 |  1.536786 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.652872 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.652872 |       0.0 |   0.0 |   0.0 |  0.0 | -0.566251 |       0.0 |   0.0 |   0.0 | 0.450164 |      0.0 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 3.747947 | 1.873973 |  0.0 |  0.0 |   1.53573 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.873973 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.873973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.783929 |       0.0 |   0.0 |   0.0 | 0.445685 |      0.0 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 2.050471 | 1.025236 |  0.0 |  0.0 |  0.564453 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.025236 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.025236 |       0.0 |   0.0 |   0.0 |  0.0 | -0.902033 |       0.0 |   0.0 |   0.0 |  0.44125 |      0.0 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 2.650012 | 1.325006 |  0.0 |  0.0 |  0.809425 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.325006 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.325006 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.95244 |       0.0 |   0.0 |   0.0 |  0.43686 |      0.0 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 2.959048 | 1.479524 |  0.0 |  0.0 |  0.877591 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.479524 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.479524 |       0.0 |   0.0 |   0.0 |  0.0 | -1.034446 |       0.0 |   0.0 |   0.0 | 0.432513 |      0.0 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 0.956604 | 0.478302 |  0.0 |  0.0 | -0.092369 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.238302 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.238302 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.99888 |       0.0 |   0.0 |   0.0 | 0.428209 |      0.0 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 0.731934 | 0.365967 |  0.0 |  0.0 | -0.083051 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.365967 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.365967 |       0.0 |   0.0 |   0.0 |  0.0 | -0.872967 |       0.0 |   0.0 |   0.0 | 0.423949 |      0.0 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 0.454628 | 0.227314 |  0.0 |  0.0 | -0.121034 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.227314 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.227314 |       0.0 |   0.0 |   0.0 |  0.0 | -0.768078 |       0.0 |   0.0 |   0.0 |  0.41973 |      0.0 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 0.300394 | 0.150197 |  0.0 |  0.0 |  0.063243 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  1.409803 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  1.290681 |  0.119122 |   0.0 |   0.0 |  0.0 | -0.527888 |   0.02538 |   0.0 |   0.0 | 0.415554 |      0.0 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 0.089558 | 0.044779 |  0.0 |  0.0 |  0.044779 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  6.675221 |  1.330493 |  0.0 |  0.0 |  0.0 |  0.0 |  1.850192 |  4.825029 |   0.0 |   0.0 |  0.0 | -0.145837 |  1.064911 |   0.0 |   0.0 | 0.411419 | 0.369581 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 |  0.03977 | 0.019885 |  0.0 |  0.0 |  0.019885 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  3.460115 |  2.499316 |  0.0 |  0.0 |  0.0 |  0.0 |  1.710992 |  1.749123 |   0.0 |   0.0 |  0.0 |  0.202945 |  1.889046 |   0.0 |   0.0 | 0.407325 | 0.694254 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 | 0.039382 | 0.019691 |  0.0 |  0.0 |  0.019691 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  5.860309 |  3.210825 |  0.0 |  0.0 |  0.0 |  0.0 |  1.829361 |  4.030949 |   0.0 |   0.0 |  0.0 |  0.487393 |  2.320159 |   0.0 |   0.0 | 0.403272 | 0.891896 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 0.039251 | 0.019626 |  0.0 |  0.0 |  0.019626 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 12.700374 |  5.569592 |  0.0 |  0.0 |  0.0 |  0.0 |  1.921262 | 10.779112 |   0.0 |   0.0 |  0.0 |  0.739258 |  4.431075 |   0.0 |   0.0 |  0.39926 | 1.547109 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 |  0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.100505 |  5.810506 |  0.0 |  0.0 |  0.0 |  0.0 |  0.100505 |       0.0 |   0.0 |   0.0 |  0.0 |  0.782998 |  4.632221 |   0.0 |   0.0 | 0.395287 | 1.614029 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 |  0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.820505 |  3.927652 |  0.0 |  0.0 |  0.0 |  0.0 |  0.820505 |       0.0 |   0.0 |   0.0 |  0.0 |  0.726714 |  2.809584 |   0.0 |   0.0 | 0.391354 | 1.091014 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 |  0.03899 | 0.019495 |  0.0 |  0.0 |  0.019495 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  3.580505 |  3.315832 |  0.0 |  0.0 |  0.0 |  0.0 |   1.72071 |  1.859795 |   0.0 |   0.0 |  0.0 |  0.828023 |  2.100349 |   0.0 |   0.0 |  0.38746 | 0.921064 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.038792 | 0.019396 |  0.0 |  0.0 |  0.019396 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  2.500604 |   3.16346 |  0.0 |  0.0 |  0.0 |  0.0 |  1.600097 |  0.900507 |   0.0 |   0.0 |  0.0 |  0.978544 |  1.801312 |   0.0 |   0.0 | 0.383605 | 0.878739 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 | 0.038661 |  0.01933 |  0.0 |  0.0 |   0.01933 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  12.46067 |  5.001808 |  0.0 |  0.0 |  0.0 |  0.0 |  1.919747 | 10.540922 |   0.0 |   0.0 |  0.0 |  1.121149 |  3.500872 |   0.0 |   0.0 | 0.379788 | 1.389391 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.038528 | 0.019264 |  0.0 |  0.0 |  0.019264 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  4.180736 |   6.16768 |  0.0 |  0.0 |  0.0 |  0.0 |  1.760808 |  2.419928 |   0.0 |   0.0 |  0.0 |  1.251025 |  4.540646 |   0.0 |   0.0 | 0.376009 | 1.713244 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.038396 | 0.019198 |  0.0 |  0.0 |  0.019198 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  4.060802 |  5.397199 |  0.0 |  0.0 |  0.0 |  0.0 |  1.753743 |  2.307059 |   0.0 |   0.0 |  0.0 |  1.342771 |  3.682161 |   0.0 |   0.0 | 0.372267 | 1.499222 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.074919 |  0.03746 |  0.0 |  0.0 |   0.03746 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   1.40254 |   4.41629 |  0.0 |  0.0 |  0.0 |  0.0 |  1.287008 |  0.115532 |   0.0 |   0.0 |  0.0 |  1.373556 |  2.674171 |   0.0 |   0.0 | 0.368563 | 1.226747 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.505845 | 0.252922 |  0.0 |  0.0 |  0.252922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.132922 |  3.232591 |  0.0 |  0.0 |  0.0 |  0.0 | -0.132922 |       0.0 |   0.0 |   0.0 |  0.0 |  1.224886 |  1.642809 |   0.0 |   0.0 | 0.364896 | 0.897942 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 | 1.218215 | 0.609108 |  0.0 |  0.0 |  0.609108 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.609108 |   2.29184 |  0.0 |  0.0 |  0.0 |  0.0 | -0.609108 |       0.0 |   0.0 |   0.0 |  0.0 |   0.93416 |  0.996414 |   0.0 |   0.0 | 0.361265 | 0.636622 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.772303 | 0.386152 |  0.0 |  0.0 |  0.386152 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.386152 |   1.63732 |  0.0 |  0.0 |  0.0 |  0.0 | -0.386152 |       0.0 |   0.0 |   0.0 |  0.0 |  0.675294 |  0.604356 |   0.0 |   0.0 | 0.357671 | 0.454811 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 |   0.9844 |   0.4922 |  0.0 |  0.0 |    0.4922 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   -0.0122 |  1.238581 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0122 |       0.0 |   0.0 |   0.0 |  0.0 |  0.517909 |   0.36656 |   0.0 |   0.0 | 0.354112 |  0.34405 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 2.486378 | 1.243189 |  0.0 |  0.0 |  1.243189 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.123189 |  0.890686 |  0.0 |  0.0 |  0.0 |  0.0 | -1.123189 |       0.0 |   0.0 |   0.0 |  0.0 |  0.317768 |   0.22233 |   0.0 |   0.0 | 0.350588 | 0.247413 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 | 1.349913 | 0.674957 |  0.0 |  0.0 |  0.674957 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  3.645043 |  1.214269 |  0.0 |  0.0 |  0.0 |  0.0 |  1.725655 |  1.919389 |   0.0 |   0.0 |  0.0 |  0.323372 |  0.543797 |   0.0 |   0.0 |   0.3471 | 0.337297 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 3.528622 | 1.764311 |  0.0 |  0.0 |  1.764311 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  5.315689 |  2.351913 |  0.0 |  0.0 |  0.0 |  0.0 |  1.811878 |  3.503811 |   0.0 |   0.0 |  0.0 |  0.585638 |   1.42263 |   0.0 |   0.0 | 0.343646 | 0.653309 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 |  1.61841 | 0.809205 |  0.0 |  0.0 |  0.809205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.510795 |  2.521277 |  0.0 |  0.0 |  0.0 |  0.0 |  0.510795 |       0.0 |   0.0 |   0.0 |  0.0 |  0.686066 |  1.494984 |   0.0 |   0.0 | 0.340227 | 0.700355 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.844303 | 0.422151 |  0.0 |  0.0 |  0.422151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 24.417849 |  6.818626 |  0.0 |  0.0 |  0.0 |  0.0 |  1.959046 | 22.458802 |   0.0 |   0.0 |  0.0 |  0.789929 |  5.691856 |   0.0 |   0.0 | 0.336841 | 1.894063 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.895703 | 0.447851 |  0.0 |  0.0 |  0.447851 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 45.032149 |  18.01435 |  0.0 |  0.0 |  0.0 |  0.0 |  1.977794 | 43.054355 |   0.0 |   0.0 |  0.0 |   1.00361 | 16.677251 |   0.0 |   0.0 |  0.33349 | 5.003986 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 1.151139 |  0.57557 |  0.0 |  0.0 |   0.57557 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   9.26443 | 20.955723 |  0.0 |  0.0 |  0.0 |  0.0 |   1.89206 |   7.37237 |   0.0 |   0.0 |  0.0 |   1.17217 | 19.453382 |   0.0 |   0.0 | 0.330172 | 5.821034 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.636581 |  0.31829 |  0.0 |  0.0 |   0.31829 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   4.00171 | 15.225092 |  0.0 |  0.0 |  0.0 |  0.0 |  1.750107 |  2.251603 |   0.0 |   0.0 |  0.0 |  1.289369 | 13.608837 |   0.0 |   0.0 | 0.326886 | 4.229192 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.116642 | 0.058321 |  0.0 |  0.0 |  0.058321 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  8.941679 | 11.872665 |  0.0 |  0.0 |  0.0 |  0.0 |  1.888164 |  7.053515 |   0.0 |   0.0 |  0.0 |  1.385817 | 10.163215 |   0.0 |   0.0 | 0.323634 | 3.297963 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.037049 | 0.018524 |  0.0 |  0.0 |  0.018524 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 22.181476 | 13.549851 |  0.0 |  0.0 |  0.0 |  0.0 |  1.954917 | 20.226558 |   0.0 |   0.0 |  0.0 |  1.483128 | 11.746309 |   0.0 |   0.0 | 0.320413 | 3.763848 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 |  0.03698 |  0.01849 |  0.0 |  0.0 |   0.01849 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  18.46151 | 16.177406 |  0.0 |  0.0 |  0.0 |  0.0 |  1.945833 | 16.515677 |   0.0 |   0.0 |  0.0 |  1.567799 | 14.292382 |   0.0 |   0.0 | 0.317225 | 4.493724 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.036499 |  0.01825 |  0.0 |  0.0 |   0.01825 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   7.54175 |  14.80036 |  0.0 |  0.0 |  0.0 |  0.0 |  1.867405 |  5.674346 |   0.0 |   0.0 |  0.0 |  1.628979 | 12.857312 |   0.0 |   0.0 | 0.314069 | 4.111211 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 |  0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  2.262063 | 10.926234 |  0.0 |  0.0 |  0.0 |  0.0 |  1.557926 |  0.704138 |   0.0 |   0.0 |  0.0 |  1.643215 |  8.972076 |   0.0 |   0.0 | 0.310944 | 3.035065 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.035943 | 0.017972 |  0.0 |  0.0 |  0.017972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  5.862028 |  8.389096 |  0.0 |  0.0 |  0.0 |  0.0 |  1.829411 |  4.032618 |   0.0 |   0.0 |  0.0 |   1.65318 |  6.428066 |   0.0 |   0.0 |  0.30785 | 2.330304 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.035873 | 0.017937 |  0.0 |  0.0 |  0.017937 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  3.222063 |  6.929658 |  0.0 |  0.0 |  0.0 |  0.0 |   1.68964 |  1.532424 |   0.0 |   0.0 |  0.0 |  1.672035 |  4.952836 |   0.0 |   0.0 | 0.304787 | 1.924905 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.035522 | 0.017761 |  0.0 |  0.0 |  0.017761 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.582239 |  5.153776 |  0.0 |  0.0 |  0.0 |  0.0 |  0.582239 |       0.0 |   0.0 |   0.0 |  0.0 |  1.571514 |  3.280508 |   0.0 |   0.0 | 0.301754 | 1.431605 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.035663 | 0.017831 |  0.0 |  0.0 |  0.017831 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.222169 |  3.646947 |  0.0 |  0.0 |  0.0 |  0.0 |  0.222169 |       0.0 |   0.0 |   0.0 |  0.0 |  1.358467 |  1.989729 |   0.0 |   0.0 | 0.298752 | 1.013041 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.036083 | 0.018041 |  0.0 |  0.0 |  0.018041 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.581959 |  2.688797 |  0.0 |  0.0 |  0.0 |  0.0 |  0.581959 |       0.0 |   0.0 |   0.0 |  0.0 |  1.186187 |  1.206832 |   0.0 |   0.0 | 0.295779 | 0.746888 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.036637 | 0.018319 |  0.0 |  0.0 |  0.018319 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  2.861681 |  2.459597 |  0.0 |  0.0 |  0.0 |  0.0 |  1.650555 |  1.211126 |   0.0 |   0.0 |  0.0 |  1.176737 |  0.990025 |   0.0 |   0.0 | 0.292836 | 0.683221 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.061944 | 0.030972 |  0.0 |  0.0 |  0.030972 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.449028 |  2.258997 |  0.0 |  0.0 |  0.0 |  0.0 |  0.449028 |       0.0 |   0.0 |   0.0 |  0.0 |  1.150098 |  0.818977 |   0.0 |   0.0 | 0.289922 | 0.627499 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.188148 | 0.094074 |  0.0 |  0.0 |  0.094074 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.145926 |  1.778401 |  0.0 |  0.0 |  0.0 |  0.0 |  0.145926 |       0.0 |   0.0 |   0.0 |  0.0 |  0.994629 |  0.496735 |   0.0 |   0.0 | 0.287037 |    0.494 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.46949 | 0.234745 |  0.0 |  0.0 |  0.234745 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.234745 |    1.3906 |  0.0 |  0.0 |  0.0 |  0.0 | -0.234745 |       0.0 |   0.0 |   0.0 |  0.0 |  0.805134 |  0.301285 |   0.0 |   0.0 | 0.284181 | 0.386278 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 1.680918 | 0.840459 |  0.0 |  0.0 |  0.840459 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.840459 |     1.024 |  0.0 |  0.0 |  0.0 |  0.0 | -0.840459 |       0.0 |   0.0 |   0.0 |  0.0 |  0.559908 |  0.182738 |   0.0 |   0.0 | 0.281354 | 0.284445 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 1.743886 | 0.871943 |  0.0 |  0.0 |  0.871943 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.511943 |  0.726222 |  0.0 |  0.0 |  0.0 |  0.0 | -0.511943 |       0.0 |   0.0 |   0.0 |  0.0 |  0.336832 |  0.110837 |   0.0 |   0.0 | 0.278554 | 0.201728 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 2.122301 | 1.061151 |  0.0 |  0.0 |  1.061151 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  2.058849 |  0.831772 |  0.0 |  0.0 |  0.0 |  0.0 |  1.514292 |  0.544558 |   0.0 |   0.0 |  0.0 |  0.372739 |   0.18325 |   0.0 |   0.0 | 0.275782 | 0.231048 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 2.150411 | 1.075205 |  0.0 |  0.0 |  1.075205 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.235205 |  0.898248 |  0.0 |  0.0 |  0.0 |  0.0 | -0.235205 |       0.0 |   0.0 |   0.0 |  0.0 |  0.415821 |  0.209389 |   0.0 |   0.0 | 0.273038 | 0.249513 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 1.863566 | 0.931783 |  0.0 |  0.0 |  0.931783 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.571783 |  0.663611 |  0.0 |  0.0 |  0.0 |  0.0 | -0.571783 |       0.0 |   0.0 |   0.0 |  0.0 |  0.266288 |  0.127001 |   0.0 |   0.0 | 0.270322 | 0.184336 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 3.115838 | 1.557919 |  0.0 |  0.0 |  1.557919 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.197919 |  0.400393 |  0.0 |  0.0 |  0.0 |  0.0 | -1.197919 |       0.0 |   0.0 |   0.0 |  0.0 |  0.055731 |   0.07703 |   0.0 |   0.0 | 0.267632 |  0.11122 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 1.940627 | 0.970313 |  0.0 |  0.0 |  0.970313 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.970313 |  0.161489 |  0.0 |  0.0 |  0.0 |  0.0 | -0.970313 |       0.0 |   0.0 |   0.0 |  0.0 | -0.150201 |  0.046721 |   0.0 |   0.0 | 0.264969 | 0.044858 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 2.122768 | 1.061384 |  0.0 |  0.0 |  1.044663 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.061384 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.061384 |       0.0 |   0.0 |   0.0 |  0.0 | -0.307391 |  0.028338 |   0.0 |   0.0 | 0.262332 |      0.0 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 1.314414 | 0.657207 |  0.0 |  0.0 |  0.527903 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.657207 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.657207 |       0.0 |   0.0 |   0.0 |  0.0 | -0.406214 |  0.017188 |   0.0 |   0.0 | 0.259722 |      0.0 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.602486 | 0.301243 |  0.0 |  0.0 |  0.150432 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.301243 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.301243 |       0.0 |   0.0 |   0.0 |  0.0 | -0.418374 |  0.010425 |   0.0 |   0.0 | 0.257138 |      0.0 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.109905 | 0.054952 |  0.0 |  0.0 | -0.058221 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.054952 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.054952 |       0.0 |   0.0 |   0.0 |  0.0 | -0.374076 |  0.006323 |   0.0 |   0.0 | 0.254579 |      0.0 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.035026 | 0.017513 |  0.0 |  0.0 | -0.039328 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.017513 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017513 |       0.0 |   0.0 |   0.0 |  0.0 | -0.312722 |  0.003835 |   0.0 |   0.0 | 0.252046 |      0.0 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 |  0.009842 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.017154 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017154 |       0.0 |   0.0 |   0.0 |  0.0 | -0.259176 |  0.002326 |   0.0 |   0.0 | 0.249538 |      0.0 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.033727 | 0.016863 |  0.0 |  0.0 |  0.016863 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.016863 |  0.033188 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016863 |       0.0 |   0.0 |   0.0 |  0.0 | -0.215278 |  0.001411 |   0.0 |   0.0 | 0.247055 | 0.009219 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.033286 | 0.016643 |  0.0 |  0.0 |  0.016643 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.016643 |  0.066162 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016643 |       0.0 |   0.0 |   0.0 |  0.0 | -0.179291 |  0.000856 |   0.0 |   0.0 | 0.244597 | 0.018378 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.032916 | 0.016458 |  0.0 |  0.0 |  0.016458 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.016458 |  0.092892 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016458 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.14979 |  0.000519 |   0.0 |   0.0 | 0.242163 | 0.025803 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 |    0.0169 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   -0.0169 |  0.114406 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0169 |       0.0 |   0.0 |   0.0 |  0.0 | -0.125663 |  0.000315 |   0.0 |   0.0 | 0.239754 | 0.031779 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 |  0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.016973 |  0.131605 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.105954 |  0.000191 |   0.0 |   0.0 | 0.237368 | 0.036557 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.033434 | 0.016717 |  0.0 |  0.0 |  0.016717 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  1.543283 |  0.314258 |  0.0 |  0.0 |  0.0 |  0.0 |  1.352031 |  0.191252 |   0.0 |   0.0 |  0.0 |  0.038388 |  0.040864 |   0.0 |   0.0 | 0.235006 | 0.087294 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 |  0.03336 |  0.01668 |  0.0 |  0.0 |   0.01668 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  -0.01668 |  0.440283 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.01668 |       0.0 |   0.0 |   0.0 |  0.0 |  0.148326 |  0.059289 |   0.0 |   0.0 | 0.232668 | 0.122301 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 |  0.016973 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.016973 |  0.384701 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 |  0.118388 |  0.035961 |   0.0 |   0.0 | 0.230353 | 0.106861 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.124092 | 0.062046 |  0.0 |  0.0 |  0.062046 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.062046 |  0.339502 |  0.0 |  0.0 |  0.0 |  0.0 | -0.062046 |       0.0 |   0.0 |   0.0 |  0.0 |   0.08963 |  0.021811 |   0.0 |   0.0 | 0.228061 | 0.094306 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.556783 | 0.278392 |  0.0 |  0.0 |  0.278392 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.561608 |  0.359564 |  0.0 |  0.0 |  0.0 |  0.0 |  0.561608 |       0.0 |   0.0 |   0.0 |  0.0 |  0.120543 |  0.013229 |   0.0 |   0.0 | 0.225792 | 0.099879 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 |  1.27835 | 0.639175 |  0.0 |  0.0 |  0.639175 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.159175 |   0.36456 |  0.0 |  0.0 |  0.0 |  0.0 | -0.159175 |       0.0 |   0.0 |   0.0 |  0.0 |  0.132991 |  0.008024 |   0.0 |   0.0 | 0.223545 | 0.101267 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 |  2.07666 |  1.03833 |  0.0 |  0.0 |   1.03833 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  -0.91833 |   0.23512 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.91833 |       0.0 |   0.0 |   0.0 |  0.0 |  0.008932 |  0.004867 |   0.0 |   0.0 | 0.221321 | 0.065311 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 | 2.862014 | 1.431007 |  0.0 |  0.0 |  1.431007 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.951007 |  0.059858 |  0.0 |  0.0 |  0.0 |  0.0 | -0.951007 |       0.0 |   0.0 |   0.0 |  0.0 | -0.162212 |  0.002952 |   0.0 |   0.0 | 0.219118 | 0.016627 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 3.356573 | 1.678286 |  0.0 |  0.0 |  1.523706 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.678286 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.678286 |       0.0 |   0.0 |   0.0 |  0.0 | -0.373309 |   0.00179 |   0.0 |   0.0 | 0.216938 |      0.0 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 3.356514 | 1.678257 |  0.0 |  0.0 |  1.284264 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.678257 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.678257 |       0.0 |   0.0 |   0.0 |  0.0 | -0.609858 |  0.001086 |   0.0 |   0.0 |  0.21478 |      0.0 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 4.286095 | 2.143048 |  0.0 |  0.0 |  1.509293 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -2.143048 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -2.143048 |       0.0 |   0.0 |   0.0 |  0.0 | -0.847056 |  0.000659 |   0.0 |   0.0 | 0.212642 |      0.0 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 3.753358 | 1.876679 |  0.0 |  0.0 |  1.030572 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.876679 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.876679 |       0.0 |   0.0 |   0.0 |  0.0 | -1.057033 |  0.000399 |   0.0 |   0.0 | 0.210527 |      0.0 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 3.144766 | 1.572383 |  0.0 |  0.0 |  0.603946 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.572383 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.572383 |       0.0 |   0.0 |   0.0 |  0.0 | -1.177111 |  0.000242 |   0.0 |   0.0 | 0.208432 |      0.0 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 2.519332 | 1.259666 |  0.0 |  0.0 |  0.246696 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -1.259666 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.259666 |       0.0 |   0.0 |   0.0 |  0.0 | -1.219474 |  0.000147 |   0.0 |   0.0 | 0.206358 |      0.0 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.179366 | 0.589683 |  0.0 |  0.0 | -0.369936 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.589683 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.589683 |       0.0 |   0.0 |   0.0 |  0.0 | -1.164013 |  0.000089 |   0.0 |   0.0 | 0.204305 |      0.0 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 0.523693 | 0.261846 |  0.0 |  0.0 |  -0.56503 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.261846 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.261846 |       0.0 |   0.0 |   0.0 |  0.0 | -1.029202 |  0.000054 |   0.0 |   0.0 | 0.202272 |      0.0 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 |  0.13182 |  0.06591 |  0.0 |  0.0 | -0.605552 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  -0.06591 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.06591 |       0.0 |   0.0 |   0.0 |  0.0 | -0.871754 |  0.000033 |   0.0 |   0.0 | 0.200259 |      0.0 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 0.035803 | 0.017902 |  0.0 |  0.0 | -0.504995 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.017902 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017902 |       0.0 |   0.0 |   0.0 |  0.0 | -0.721183 |   0.00002 |   0.0 |   0.0 | 0.198267 |      0.0 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |  0.0 |  0.0 |  -0.38017 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.017154 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017154 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.59363 |  0.000012 |   0.0 |   0.0 | 0.196294 |      0.0 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |  0.0 |  0.0 | -0.277861 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   -0.0169 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0169 |       0.0 |   0.0 |   0.0 |  0.0 | -0.489109 |  0.000007 |   0.0 |   0.0 | 0.194341 |      0.0 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |  0.0 |  0.0 | -0.194134 | 0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 | -0.016973 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.403518 |  0.000004 |   0.0 |   0.0 | 0.192407 |      0.0 |

    .. raw:: html

        <iframe
            src="lland_v2_ex4.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    :ref:`Recalculation of example 5 <lland_v1_ex5>`

    The following calculation shows, that the outflow values of the
    integration test for sealed areas (|VERS|) are reproduced exactly:

    >>> lnk(VERS)
    >>> test('lland_v2_ex5')
    |   date | nied | teml |       pet |  nkor | tkor |      et0 |     evpo |      nbes | sbes |      evi | evb |      wgtf | schm |      wada |       qdb | qib1 | qib2 | qbb |      qdgz |         q |     inzp | wats | waes | bowa |    qdgz1 |     qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |     qdga2 | qiga1 | qiga2 |     qbga |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 21.2 |  0.100707 |   0.0 | 22.0 | 0.040283 | 0.020141 |       0.0 |  0.0 |      0.0 | 0.0 | 11.275777 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.497508 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.497508 | 0.138197 |
    | 01.01. |  0.0 | 19.4 |  0.097801 |   0.0 | 20.2 |  0.03912 |  0.01956 |       0.0 |  0.0 |      0.0 | 0.0 | 10.353214 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.492558 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.492558 | 0.136822 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |       0.0 |  0.0 |      0.0 | 0.0 | 10.096946 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.487657 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.487657 |  0.13546 |
    | 01.01. |  0.0 | 18.3 |   0.09599 |   0.0 | 19.1 | 0.038396 | 0.019198 |       0.0 |  0.0 |      0.0 | 0.0 |  9.789425 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.482805 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.482805 | 0.134112 |
    | 01.01. |  0.0 | 18.9 |  0.096981 |   0.0 | 19.7 | 0.038792 | 0.019396 |       0.0 |  0.0 |      0.0 | 0.0 | 10.096946 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.478001 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.478001 | 0.132778 |
    | 01.01. |  0.0 | 22.5 |  0.102761 |   0.0 | 23.3 | 0.041104 | 0.020552 |       0.0 |  0.0 |      0.0 | 0.0 | 11.942073 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.473245 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.473245 | 0.131457 |
    | 01.01. |  0.0 | 25.1 |  0.291908 |   0.0 | 25.9 | 0.116763 | 0.058382 |       0.0 |  0.0 |      0.0 | 0.0 | 13.274665 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.468536 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.468536 | 0.130149 |
    | 01.01. |  0.0 | 28.3 |  1.932875 |   0.0 | 29.1 |  0.77315 | 0.386575 |       0.0 |  0.0 |      0.0 | 0.0 | 14.914778 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.463874 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.463874 | 0.128854 |
    | 01.01. |  0.0 | 27.8 |  4.369536 |   0.0 | 28.6 | 1.747814 | 0.873907 |       0.0 |  0.0 |      0.0 | 0.0 |  14.65851 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.459258 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.459258 | 0.127572 |
    | 01.01. |  0.0 | 31.4 |  7.317556 |   0.0 | 32.2 | 2.927022 | 1.463511 |       0.0 |  0.0 |      0.0 | 0.0 | 16.503638 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.454688 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.454688 | 0.126302 |
    | 01.01. |  0.0 | 32.2 |  8.264362 |   0.0 | 33.0 | 3.305745 | 1.652872 |       0.0 |  0.0 |      0.0 | 0.0 | 16.913666 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.450164 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.450164 | 0.125046 |
    | 01.01. |  0.0 | 35.2 |  9.369867 |   0.0 | 36.0 | 3.747947 | 1.873973 |       0.0 |  0.0 |      0.0 | 0.0 | 18.451272 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.445685 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.445685 | 0.123801 |
    | 01.01. |  0.0 | 37.1 |  5.126178 |   0.0 | 37.9 | 2.050471 | 1.025236 |       0.0 |  0.0 |      0.0 | 0.0 | 19.425089 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.44125 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.44125 |  0.12257 |
    | 01.01. |  0.0 | 31.2 |   6.62503 |   0.0 | 32.0 | 2.650012 | 1.325006 |       0.0 |  0.0 |      0.0 | 0.0 | 16.401131 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.43686 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.43686 |  0.12135 |
    | 01.01. |  0.0 | 24.3 |  7.397619 |   0.0 | 25.1 | 2.959048 | 1.479524 |       0.0 |  0.0 |      0.0 | 0.0 | 12.864637 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.432513 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.432513 | 0.120142 |
    | 01.01. |  0.2 | 25.4 |   2.39151 |  0.24 | 26.2 | 0.956604 | 0.478302 |       0.0 |  0.0 |     0.24 | 0.0 | 13.428426 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.428209 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.428209 | 0.118947 |
    | 01.01. |  0.0 | 25.9 |  1.829834 |   0.0 | 26.7 | 0.731934 | 0.365967 |       0.0 |  0.0 |      0.0 | 0.0 | 13.684693 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.423949 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 | 0.423949 | 0.117764 |
    | 01.01. |  0.0 | 23.7 |  1.136569 |   0.0 | 24.5 | 0.454628 | 0.227314 |       0.0 |  0.0 |      0.0 | 0.0 | 12.557116 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.41973 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.41973 | 0.116592 |
    | 01.01. |  1.3 | 21.6 |  0.750986 |  1.56 | 22.4 | 0.300394 | 0.150197 |      0.76 |  0.0 | 0.150197 | 0.0 | 11.480791 |  0.0 |      0.76 |      0.76 |  0.0 |  0.0 | 0.0 |      0.76 |  0.486731 | 0.649803 |  0.0 |  0.0 |  0.0 |     0.76 |       0.0 |   0.0 |   0.0 |  0.0 | 0.071177 |       0.0 |   0.0 |   0.0 | 0.415554 | 0.135203 |
    | 01.01. |  5.6 | 21.2 |  0.223895 |  6.72 | 22.0 | 0.089558 | 0.044779 |  6.569803 |  0.0 | 0.044779 | 0.0 | 11.275777 |  0.0 |  6.569803 |  6.569803 |  0.0 |  0.0 | 0.0 |  6.569803 |  1.715412 | 0.755221 |  0.0 |  0.0 |  0.0 | 1.847788 |  4.722014 |   0.0 |   0.0 |  0.0 | 0.297915 |  1.006079 |   0.0 |   0.0 | 0.411419 | 0.476503 |
    | 01.01. |  2.9 | 20.4 |  0.099425 |  3.48 | 21.2 |  0.03977 | 0.019885 |  3.435221 |  0.0 | 0.019885 | 0.0 | 10.865749 |  0.0 |  3.435221 |  3.435221 |  0.0 |  0.0 | 0.0 |  3.435221 |  2.803096 | 0.780115 |  0.0 |  0.0 |  0.0 | 1.708898 |  1.726323 |   0.0 |   0.0 |  0.0 | 0.565852 |  1.829919 |   0.0 |   0.0 | 0.407325 | 0.778638 |
    | 01.01. |  4.9 | 19.8 |  0.098454 |  5.88 | 20.6 | 0.039382 | 0.019691 |  5.860115 |  0.0 | 0.019691 | 0.0 | 10.558228 |  0.0 |  5.860115 |  5.860115 |  0.0 |  0.0 | 0.0 |  5.860115 |  3.467748 | 0.780309 |  0.0 |  0.0 |  0.0 | 1.829355 |   4.03076 |   0.0 |   0.0 |  0.0 | 0.784332 |  2.280144 |   0.0 |   0.0 | 0.403272 | 0.963263 |
    | 01.01. | 10.6 | 19.6 |  0.098128 | 12.72 | 20.4 | 0.039251 | 0.019626 | 12.700309 |  0.0 | 0.019626 | 0.0 | 10.455721 |  0.0 | 12.700309 | 12.700309 |  0.0 |  0.0 | 0.0 | 12.700309 |  5.788386 | 0.780374 |  0.0 |  0.0 |  0.0 | 1.921262 | 10.779047 |   0.0 |   0.0 |  0.0 |  0.98237 |  4.406757 |   0.0 |   0.0 |  0.39926 | 1.607885 |
    | 01.01. |  0.1 | 19.2 |  0.097474 |  0.12 | 20.0 |  0.03899 | 0.019495 |  0.100374 |  0.0 | 0.019495 | 0.0 | 10.250707 |  0.0 |  0.100374 |  0.100374 |  0.0 |  0.0 | 0.0 |  0.100374 |  5.994776 | 0.780505 |  0.0 |  0.0 |  0.0 | 0.100374 |       0.0 |   0.0 |   0.0 |  0.0 | 0.982029 |   4.61746 |   0.0 |   0.0 | 0.395287 | 1.665216 |
    | 02.01. |  0.7 | 19.2 |  0.097474 |  0.84 | 20.0 |  0.03899 | 0.019495 |  0.820505 |  0.0 | 0.019495 | 0.0 | 10.250707 |  0.0 |  0.820505 |  0.820505 |  0.0 |  0.0 | 0.0 |  0.820505 |   4.08164 | 0.780505 |  0.0 |  0.0 |  0.0 | 0.820505 |       0.0 |   0.0 |   0.0 |  0.0 | 0.889655 |  2.800631 |   0.0 |   0.0 | 0.391354 | 1.133789 |
    | 02.01. |  3.0 | 19.2 |  0.097474 |   3.6 | 20.0 |  0.03899 | 0.019495 |  3.580505 |  0.0 | 0.019495 | 0.0 | 10.250707 |  0.0 |  3.580505 |  3.580505 |  0.0 |  0.0 | 0.0 |  3.580505 |  3.443807 | 0.780505 |  0.0 |  0.0 |  0.0 |  1.72071 |  1.859795 |   0.0 |   0.0 |  0.0 | 0.961428 |  2.094919 |   0.0 |   0.0 |  0.38746 | 0.956613 |
    | 02.01. |  2.1 | 18.9 |  0.096981 |  2.52 | 19.7 | 0.038792 | 0.019396 |  2.500505 |  0.0 | 0.019396 | 0.0 | 10.096946 |  0.0 |  2.500505 |  2.500505 |  0.0 |  0.0 | 0.0 |  2.500505 |   3.26937 | 0.780604 |  0.0 |  0.0 |  0.0 | 1.600081 |  0.900424 |   0.0 |   0.0 |  0.0 | 1.087765 |     1.798 |   0.0 |   0.0 | 0.383605 | 0.908158 |
    | 02.01. | 10.4 | 18.7 |  0.096652 | 12.48 | 19.5 | 0.038661 |  0.01933 | 12.460604 |  0.0 |  0.01933 | 0.0 |  9.994439 |  0.0 | 12.460604 | 12.460604 |  0.0 |  0.0 | 0.0 | 12.460604 |  5.089193 |  0.78067 |  0.0 |  0.0 |  0.0 | 1.919747 | 10.540857 |   0.0 |   0.0 |  0.0 |  1.21057 |  3.498835 |   0.0 |   0.0 | 0.379788 | 1.413665 |
    | 02.01. |  3.5 | 18.5 |  0.096321 |   4.2 | 19.3 | 0.038528 | 0.019264 |   4.18067 |  0.0 | 0.019264 | 0.0 |  9.891932 |  0.0 |   4.18067 |   4.18067 |  0.0 |  0.0 | 0.0 |   4.18067 |  6.239631 | 0.780736 |  0.0 |  0.0 |  0.0 | 1.760804 |  2.419866 |   0.0 |   0.0 |  0.0 | 1.324236 |  4.539386 |   0.0 |   0.0 | 0.376009 | 1.733231 |
    | 02.01. |  3.4 | 18.3 |   0.09599 |  4.08 | 19.1 | 0.038396 | 0.019198 |  4.060736 |  0.0 | 0.019198 | 0.0 |  9.789425 |  0.0 |  4.060736 |  4.060736 |  0.0 |  0.0 | 0.0 |  4.060736 |   5.45635 | 0.780802 |  0.0 |  0.0 |  0.0 | 1.753739 |  2.306997 |   0.0 |   0.0 |  0.0 | 1.402711 |  3.681371 |   0.0 |   0.0 | 0.372267 | 1.515653 |
    | 02.01. |  1.2 | 18.5 |  0.187298 |  1.44 | 19.3 | 0.074919 |  0.03746 |  1.420802 |  0.0 |  0.03746 | 0.0 |  9.891932 |  0.0 |  1.420802 |  1.420802 |  0.0 |  0.0 | 0.0 |  1.420802 |  4.467671 |  0.76254 |  0.0 |  0.0 |  0.0 | 1.296172 |   0.12463 |   0.0 |   0.0 |  0.0 | 1.423489 |  2.675619 |   0.0 |   0.0 | 0.368563 |  1.24102 |
    | 02.01. |  0.1 | 18.8 |  1.264612 |  0.12 | 19.6 | 0.505845 | 0.252922 |   0.08254 |  0.0 | 0.252922 | 0.0 | 10.045692 |  0.0 |   0.08254 |   0.08254 |  0.0 |  0.0 | 0.0 |   0.08254 |  3.296974 | 0.547078 |  0.0 |  0.0 |  0.0 |  0.08254 |       0.0 |   0.0 |   0.0 |  0.0 | 1.286749 |  1.645329 |   0.0 |   0.0 | 0.364896 | 0.915826 |
    | 02.01. |  0.0 | 18.8 |  3.045538 |   0.0 | 19.6 | 1.218215 | 0.609108 |       0.0 |  0.0 | 0.547078 | 0.0 | 10.045692 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  2.419941 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 1.060733 |  0.997943 |   0.0 |   0.0 | 0.361265 | 0.672206 |
    | 02.01. |  0.0 | 19.0 |  1.930758 |   0.0 | 19.8 | 0.772303 | 0.386152 |       0.0 |  0.0 |      0.0 | 0.0 |   10.1482 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  1.831408 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.868455 |  0.605283 |   0.0 |   0.0 | 0.357671 | 0.508724 |
    | 02.01. |  0.4 | 19.2 |  2.461001 |  0.48 | 20.0 |   0.9844 |   0.4922 |       0.0 |  0.0 |     0.48 | 0.0 | 10.250707 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  1.432265 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.71103 |  0.367123 |   0.0 |   0.0 | 0.354112 | 0.397851 |
    | 02.01. |  0.1 | 19.3 |  6.215945 |  0.12 | 20.1 | 2.486378 | 1.243189 |       0.0 |  0.0 |     0.12 | 0.0 |  10.30196 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  1.155402 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.582143 |  0.222671 |   0.0 |   0.0 | 0.350588 | 0.320945 |
    | 02.01. |  3.6 | 19.0 |  3.374783 |  4.32 | 19.8 | 1.349913 | 0.674957 |      3.52 |  0.0 | 0.674957 | 0.0 |   10.1482 |  0.0 |      3.52 |      3.52 |  0.0 |  0.0 | 0.0 |      3.52 |  1.503858 | 0.125043 |  0.0 |  0.0 |  0.0 | 1.715909 |  1.804091 |   0.0 |   0.0 |  0.0 | 0.637319 |  0.519439 |   0.0 |   0.0 |   0.3471 | 0.417738 |
    | 02.01. |  5.9 | 18.8 |  8.821555 |  7.08 | 19.6 | 3.528622 | 1.764311 |  6.405043 |  0.0 |      0.8 | 0.0 | 10.045692 |  0.0 |  6.405043 |  6.405043 |  0.0 |  0.0 | 0.0 |  6.405043 |  2.800802 |      0.0 |  0.0 |  0.0 |  0.0 | 1.843873 |   4.56117 |   0.0 |   0.0 |  0.0 | 0.844819 |  1.612337 |   0.0 |   0.0 | 0.343646 | 0.778001 |
    | 02.01. |  1.1 | 18.7 |  4.046025 |  1.32 | 19.5 |  1.61841 | 0.809205 |      0.52 |  0.0 |      0.8 | 0.0 |  9.994439 |  0.0 |      0.52 |      0.52 |  0.0 |  0.0 | 0.0 |      0.52 |  3.042961 |      0.0 |  0.0 |  0.0 |  0.0 |     0.52 |       0.0 |   0.0 |   0.0 |  0.0 | 0.901931 |  1.800804 |   0.0 |   0.0 | 0.340227 | 0.845267 |
    | 02.01. | 20.7 | 17.8 |  2.110757 | 24.84 | 18.6 | 0.844303 | 0.422151 |     24.04 |  0.0 | 0.422151 | 0.0 |  9.533157 |  0.0 |     24.04 |     24.04 |  0.0 |  0.0 | 0.0 |     24.04 |  7.101229 | 0.377849 |  0.0 |  0.0 |  0.0 | 1.958403 | 22.081597 |   0.0 |   0.0 |  0.0 |  0.96741 |  5.796977 |   0.0 |   0.0 | 0.336841 | 1.972564 |
    | 02.01. | 37.9 | 17.4 |  2.239257 | 45.48 | 18.2 | 0.895703 | 0.447851 | 45.057849 |  0.0 | 0.447851 | 0.0 |  9.328143 |  0.0 | 45.057849 | 45.057849 |  0.0 |  0.0 | 0.0 | 45.057849 | 18.160786 | 0.352149 |  0.0 |  0.0 |  0.0 | 1.977806 | 43.080042 |   0.0 |   0.0 |  0.0 | 1.148864 | 16.678432 |   0.0 |   0.0 |  0.33349 | 5.044663 |
    | 02.01. |  8.2 | 17.3 |  2.877848 |  9.84 | 18.1 | 1.151139 |  0.57557 |  9.392149 |  0.0 |  0.57557 | 0.0 |  9.276889 |  0.0 |  9.392149 |  9.392149 |  0.0 |  0.0 | 0.0 |  9.392149 | 21.107036 |  0.22443 |  0.0 |  0.0 |  0.0 | 1.893528 |  7.498621 |   0.0 |   0.0 |  0.0 | 1.291233 | 19.485632 |   0.0 |   0.0 | 0.330172 | 5.863066 |
    | 02.01. |  3.6 | 16.8 |  1.591452 |  4.32 | 17.6 | 0.636581 |  0.31829 |   3.74443 |  0.0 |  0.31829 | 0.0 |  9.020622 |  0.0 |   3.74443 |   3.74443 |  0.0 |  0.0 | 0.0 |   3.74443 | 15.312272 |  0.48171 |  0.0 |  0.0 |  0.0 | 1.732937 |  2.011494 |   0.0 |   0.0 |  0.0 |  1.38537 | 13.600016 |   0.0 |   0.0 | 0.326886 | 4.253409 |
    | 02.01. |  7.5 | 16.5 |  0.291604 |   9.0 | 17.3 | 0.116642 | 0.058321 |   8.68171 |  0.0 | 0.058321 | 0.0 |  8.866861 |  0.0 |   8.68171 |   8.68171 |  0.0 |  0.0 | 0.0 |   8.68171 | 11.846103 | 0.741679 |  0.0 |  0.0 |  0.0 | 1.884815 |  6.796894 |   0.0 |   0.0 |  0.0 | 1.462598 | 10.059871 |   0.0 |   0.0 | 0.323634 | 3.290584 |
    | 02.01. | 18.5 | 16.3 |  0.092622 |  22.2 | 17.1 | 0.037049 | 0.018524 | 22.141679 |  0.0 | 0.018524 | 0.0 |  8.764354 |  0.0 | 22.141679 | 22.141679 |  0.0 |  0.0 | 0.0 | 22.141679 | 13.494974 | 0.781476 |  0.0 |  0.0 |  0.0 | 1.954836 | 20.186843 |   0.0 |   0.0 |  0.0 |  1.54569 |  11.62887 |   0.0 |   0.0 | 0.320413 | 3.748604 |
    | 02.01. | 15.4 | 16.2 |  0.092451 | 18.48 | 17.0 |  0.03698 |  0.01849 | 18.461476 |  0.0 |  0.01849 | 0.0 |  8.713101 |  0.0 | 18.461476 | 18.461476 |  0.0 |  0.0 | 0.0 | 18.461476 | 16.150217 |  0.78151 |  0.0 |  0.0 |  0.0 | 1.945833 | 16.515642 |   0.0 |   0.0 |  0.0 | 1.619013 | 14.213979 |   0.0 |   0.0 | 0.317225 | 4.486171 |
    | 02.01. |  6.3 | 15.5 |  0.091248 |  7.56 | 16.3 | 0.036499 |  0.01825 |   7.54151 |  0.0 |  0.01825 | 0.0 |  8.354326 |  0.0 |   7.54151 |   7.54151 |  0.0 |  0.0 | 0.0 |   7.54151 |  14.79468 |  0.78175 |  0.0 |  0.0 |  0.0 | 1.867401 |  5.674109 |   0.0 |   0.0 |  0.0 |  1.67091 | 12.809702 |   0.0 |   0.0 | 0.314069 | 4.109633 |
    | 02.01. |  1.9 | 14.6 |  0.089683 |  2.28 | 15.4 | 0.035873 | 0.017937 |   2.26175 |  0.0 | 0.017937 | 0.0 |  7.893044 |  0.0 |   2.26175 |   2.26175 |  0.0 |  0.0 | 0.0 |   2.26175 | 10.931585 | 0.782063 |  0.0 |  0.0 |  0.0 | 1.557865 |  0.703886 |   0.0 |   0.0 |  0.0 | 1.677538 |  8.943103 |   0.0 |   0.0 | 0.310944 | 3.036551 |
    | 03.01. |  4.9 | 14.7 |  0.089858 |  5.88 | 15.5 | 0.035943 | 0.017972 |  5.862063 |  0.0 | 0.017972 | 0.0 |  7.944298 |  0.0 |  5.862063 |  5.862063 |  0.0 |  0.0 | 0.0 |  5.862063 |  8.399581 | 0.782028 |  0.0 |  0.0 |  0.0 | 1.829412 |  4.032652 |   0.0 |   0.0 |  0.0 | 1.681277 |  6.410455 |   0.0 |   0.0 |  0.30785 | 2.333217 |
    | 03.01. |  2.7 | 14.6 |  0.089683 |  3.24 | 15.4 | 0.035873 | 0.017937 |  3.222028 |  0.0 | 0.017937 | 0.0 |  7.893044 |  0.0 |  3.222028 |  3.222028 |  0.0 |  0.0 | 0.0 |  3.222028 |  6.941979 | 0.782063 |  0.0 |  0.0 |  0.0 | 1.689637 |  1.532392 |   0.0 |   0.0 |  0.0 | 1.695038 |  4.942153 |   0.0 |   0.0 | 0.304787 | 1.928327 |
    | 03.01. |  0.5 | 14.1 |  0.088805 |   0.6 | 14.9 | 0.035522 | 0.017761 |  0.582063 |  0.0 | 0.017761 | 0.0 |  7.636776 |  0.0 |  0.582063 |  0.582063 |  0.0 |  0.0 | 0.0 |  0.582063 |  5.166108 | 0.782239 |  0.0 |  0.0 |  0.0 | 0.582063 |       0.0 |   0.0 |   0.0 |  0.0 | 1.590331 |  3.274023 |   0.0 |   0.0 | 0.301754 |  1.43503 |
    | 03.01. |  0.2 | 14.3 |  0.089157 |  0.24 | 15.1 | 0.035663 | 0.017831 |  0.222239 |  0.0 | 0.017831 | 0.0 |  7.739283 |  0.0 |  0.222239 |  0.222239 |  0.0 |  0.0 | 0.0 |  0.222239 |  3.658411 | 0.782169 |  0.0 |  0.0 |  0.0 | 0.222239 |       0.0 |   0.0 |   0.0 |  0.0 | 1.373864 |  1.985796 |   0.0 |   0.0 | 0.298752 | 1.016225 |
    | 03.01. |  0.5 | 14.9 |  0.090207 |   0.6 | 15.7 | 0.036083 | 0.018041 |  0.582169 |  0.0 | 0.018041 | 0.0 |  8.046805 |  0.0 |  0.582169 |  0.582169 |  0.0 |  0.0 | 0.0 |  0.582169 |  2.699043 | 0.781959 |  0.0 |  0.0 |  0.0 | 0.582169 |       0.0 |   0.0 |   0.0 |  0.0 | 1.198819 |  1.204446 |   0.0 |   0.0 | 0.295779 | 0.749734 |
    | 03.01. |  2.4 | 15.7 |  0.091593 |  2.88 | 16.5 | 0.036637 | 0.018319 |  2.861959 |  0.0 | 0.018319 | 0.0 |  8.456833 |  0.0 |  2.861959 |  2.861959 |  0.0 |  0.0 | 0.0 |  2.861959 |  2.468566 | 0.781681 |  0.0 |  0.0 |  0.0 | 1.650589 |   1.21137 |   0.0 |   0.0 |  0.0 |   1.1871 |  0.988629 |   0.0 |   0.0 | 0.292836 | 0.685713 |
    | 03.01. |  0.4 | 16.0 |  0.154861 |  0.48 | 16.8 | 0.061944 | 0.030972 |  0.461681 |  0.0 | 0.030972 | 0.0 |  8.610594 |  0.0 |  0.461681 |  0.461681 |  0.0 |  0.0 | 0.0 |  0.461681 |  2.267868 | 0.769028 |  0.0 |  0.0 |  0.0 | 0.461681 |       0.0 |   0.0 |   0.0 |  0.0 | 1.159771 |  0.818175 |   0.0 |   0.0 | 0.289922 | 0.629963 |
    | 03.01. |  0.2 | 16.7 |  0.470369 |  0.24 | 17.5 | 0.188148 | 0.094074 |  0.209028 |  0.0 | 0.094074 | 0.0 |  8.969368 |  0.0 |  0.209028 |  0.209028 |  0.0 |  0.0 | 0.0 |  0.209028 |  1.792852 | 0.705926 |  0.0 |  0.0 |  0.0 | 0.209028 |       0.0 |   0.0 |   0.0 |  0.0 | 1.009567 |  0.496248 |   0.0 |   0.0 | 0.287037 | 0.498015 |
    | 03.01. |  0.0 | 17.1 |  1.173726 |   0.0 | 17.9 |  0.46949 | 0.234745 |       0.0 |  0.0 | 0.234745 | 0.0 |  9.174382 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  1.430048 | 0.471181 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.844878 |   0.30099 |   0.0 |   0.0 | 0.284181 | 0.397236 |
    | 03.01. |  0.0 | 16.2 |  4.202296 |   0.0 | 17.0 | 1.680918 | 0.840459 |       0.0 |  0.0 | 0.471181 | 0.0 |  8.713101 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   1.15564 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.691727 |  0.182559 |   0.0 |   0.0 | 0.281354 | 0.321011 |
    | 03.01. |  0.3 | 15.9 |  4.359715 |  0.36 | 16.7 | 1.743886 | 0.871943 |       0.0 |  0.0 |     0.36 | 0.0 |   8.55934 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.95562 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.566338 |  0.110728 |   0.0 |   0.0 | 0.278554 |  0.26545 |
    | 03.01. |  2.6 | 16.3 |  5.305753 |  3.12 | 17.1 | 2.122301 | 1.061151 |      2.32 |  0.0 |      0.8 | 0.0 |  8.764354 |  0.0 |      2.32 |      2.32 |  0.0 |  0.0 | 0.0 |      2.32 |  1.113577 |      0.0 |  0.0 |  0.0 |  0.0 | 1.568966 |  0.751034 |   0.0 |   0.0 |  0.0 | 0.610618 |  0.227176 |   0.0 |   0.0 | 0.275782 | 0.309327 |
    | 03.01. |  0.7 | 16.3 |  5.376027 |  0.84 | 17.1 | 2.150411 | 1.075205 |      0.04 |  0.0 |      0.8 | 0.0 |  8.764354 |  0.0 |      0.04 |      0.04 |  0.0 |  0.0 | 0.0 |      0.04 |  1.187464 |      0.0 |  0.0 |  0.0 |  0.0 |     0.04 |       0.0 |   0.0 |   0.0 |  0.0 | 0.641144 |  0.273282 |   0.0 |   0.0 | 0.273038 | 0.329851 |
    | 03.01. |  0.3 | 16.4 |  4.658915 |  0.36 | 17.2 | 1.863566 | 0.931783 |       0.0 |  0.0 |     0.36 | 0.0 |  8.815608 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.964504 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.528429 |  0.165754 |   0.0 |   0.0 | 0.270322 | 0.267918 |
    | 03.01. |  0.3 | 16.5 |  7.789594 |  0.36 | 17.3 | 3.115838 | 1.557919 |       0.0 |  0.0 |     0.36 | 0.0 |  8.866861 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.800807 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.432641 |  0.100535 |   0.0 |   0.0 | 0.267632 | 0.222447 |
    | 03.01. |  0.0 | 18.4 |  4.851567 |   0.0 | 19.2 | 1.940627 | 0.970313 |       0.0 |  0.0 |      0.0 | 0.0 |  9.840678 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.680163 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.354216 |  0.060977 |   0.0 |   0.0 | 0.264969 | 0.188934 |
    | 03.01. |  0.0 | 18.3 |   5.30692 |   0.0 | 19.1 | 2.122768 | 1.061384 |       0.0 |  0.0 |      0.0 | 0.0 |  9.789425 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.589325 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.290008 |  0.036985 |   0.0 |   0.0 | 0.262332 | 0.163701 |
    | 03.01. |  0.0 | 18.1 |  3.286036 |   0.0 | 18.9 | 1.314414 | 0.657207 |       0.0 |  0.0 |      0.0 | 0.0 |  9.686918 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.519593 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.237438 |  0.022432 |   0.0 |   0.0 | 0.259722 | 0.144331 |
    | 03.01. |  0.0 | 16.7 |  1.506216 |   0.0 | 17.5 | 0.602486 | 0.301243 |       0.0 |  0.0 |      0.0 | 0.0 |  8.969368 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.465142 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.194398 |  0.013606 |   0.0 |   0.0 | 0.257138 | 0.129206 |
    | 03.01. |  0.0 | 15.2 |  0.274762 |   0.0 | 16.0 | 0.109905 | 0.054952 |       0.0 |  0.0 |      0.0 | 0.0 |  8.200565 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.421991 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.15916 |  0.008252 |   0.0 |   0.0 | 0.254579 |  0.11722 |
    | 03.01. |  0.0 | 13.4 |  0.087565 |   0.0 | 14.2 | 0.035026 | 0.017513 |       0.0 |  0.0 |      0.0 | 0.0 |  7.278002 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.38736 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.130309 |  0.005005 |   0.0 |   0.0 | 0.252046 |   0.1076 |
    | 03.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |       0.0 |  0.0 |      0.0 | 0.0 |  6.765466 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.359262 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.106688 |  0.003036 |   0.0 |   0.0 | 0.249538 | 0.099795 |
    | 03.01. |  0.0 | 11.6 |  0.084317 |   0.0 | 12.4 | 0.033727 | 0.016863 |       0.0 |  0.0 |      0.0 | 0.0 |  6.355438 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.336245 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.087349 |  0.001841 |   0.0 |   0.0 | 0.247055 | 0.093401 |
    | 03.01. |  0.0 | 11.0 |  0.083215 |   0.0 | 11.8 | 0.033286 | 0.016643 |       0.0 |  0.0 |      0.0 | 0.0 |  6.047917 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.317229 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.071515 |  0.001117 |   0.0 |   0.0 | 0.244597 | 0.088119 |
    | 04.01. |  0.0 | 10.5 |  0.082289 |   0.0 | 11.3 | 0.032916 | 0.016458 |       0.0 |  0.0 |      0.0 | 0.0 |  5.791649 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.301392 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.058552 |  0.000677 |   0.0 |   0.0 | 0.242163 |  0.08372 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |       0.0 |  0.0 |      0.0 | 0.0 |  6.406692 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.288103 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.047938 |  0.000411 |   0.0 |   0.0 | 0.239754 | 0.080028 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |       0.0 |  0.0 |      0.0 | 0.0 |  6.509199 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.276866 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.039248 |  0.000249 |   0.0 |   0.0 | 0.237368 | 0.076907 |
    | 04.01. |  1.3 | 11.2 |  0.083584 |  1.56 | 12.0 | 0.033434 | 0.016717 |      0.76 |  0.0 | 0.016717 | 0.0 |  6.150424 |  0.0 |      0.76 |      0.76 |  0.0 |  0.0 | 0.0 |      0.76 |  0.338468 | 0.783283 |  0.0 |  0.0 |  0.0 |     0.76 |       0.0 |   0.0 |   0.0 |  0.0 | 0.103311 |  0.000151 |   0.0 |   0.0 | 0.235006 | 0.094019 |
    | 04.01. |  0.0 | 11.1 |    0.0834 |   0.0 | 11.9 |  0.03336 |  0.01668 |       0.0 |  0.0 |  0.01668 | 0.0 |   6.09917 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.383931 | 0.766603 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.151171 |  0.000092 |   0.0 |   0.0 | 0.232668 | 0.106647 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |       0.0 |  0.0 | 0.016973 | 0.0 |  6.509199 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.354177 |  0.74963 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.123769 |  0.000056 |   0.0 |   0.0 | 0.230353 | 0.098383 |
    | 04.01. |  0.0 | 12.2 |  0.310229 |   0.0 | 13.0 | 0.124092 | 0.062046 |       0.0 |  0.0 | 0.062046 | 0.0 |  6.662959 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.329428 | 0.687585 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.101333 |  0.000034 |   0.0 |   0.0 | 0.228061 | 0.091508 |
    | 04.01. |  0.7 | 11.8 |  1.391958 |  0.84 | 12.6 | 0.556783 | 0.278392 |  0.727585 |  0.0 | 0.278392 | 0.0 |  6.457945 |  0.0 |  0.727585 |  0.727585 |  0.0 |  0.0 | 0.0 |  0.727585 |  0.376918 | 0.521608 |  0.0 |  0.0 |  0.0 | 0.727585 |       0.0 |   0.0 |   0.0 |  0.0 | 0.151106 |   0.00002 |   0.0 |   0.0 | 0.225792 | 0.104699 |
    | 04.01. |  0.4 | 11.4 |  3.195876 |  0.48 | 12.2 |  1.27835 | 0.639175 |  0.201608 |  0.0 | 0.639175 | 0.0 |  6.252931 |  0.0 |  0.201608 |  0.201608 |  0.0 |  0.0 | 0.0 |  0.201608 |  0.429901 | 0.160825 |  0.0 |  0.0 |  0.0 | 0.201608 |       0.0 |   0.0 |   0.0 |  0.0 | 0.206344 |  0.000012 |   0.0 |   0.0 | 0.223545 | 0.119417 |
    | 04.01. |  0.1 | 11.6 |  5.191651 |  0.12 | 12.4 |  2.07666 |  1.03833 |       0.0 |  0.0 | 0.280825 | 0.0 |  6.355438 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.407932 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.186604 |  0.000008 |   0.0 |   0.0 | 0.221321 | 0.113315 |
    | 04.01. |  0.4 | 13.0 |  7.155036 |  0.48 | 13.8 | 2.862014 | 1.431007 |       0.0 |  0.0 |     0.48 | 0.0 |  7.072988 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.371901 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.152779 |  0.000005 |   0.0 |   0.0 | 0.219118 | 0.103306 |
    | 04.01. |  0.0 | 17.1 |  8.391432 |   0.0 | 17.9 | 3.356573 | 1.678286 |       0.0 |  0.0 |      0.0 | 0.0 |  9.174382 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.342025 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.125084 |  0.000003 |   0.0 |   0.0 | 0.216938 | 0.095007 |
    | 04.01. |  0.0 | 18.2 |  8.391286 |   0.0 | 19.0 | 3.356514 | 1.678257 |       0.0 |  0.0 |      0.0 | 0.0 |  9.738171 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.317192 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.102411 |  0.000002 |   0.0 |   0.0 |  0.21478 | 0.088109 |
    | 04.01. |  0.0 | 22.4 | 10.715238 |   0.0 | 23.2 | 4.286095 | 2.143048 |       0.0 |  0.0 |      0.0 | 0.0 |  11.89082 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.29649 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.083847 |  0.000001 |   0.0 |   0.0 | 0.212642 | 0.082358 |
    | 04.01. |  0.0 | 21.4 |  9.383394 |   0.0 | 22.2 | 3.753358 | 1.876679 |       0.0 |  0.0 |      0.0 | 0.0 | 11.378284 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.279175 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.068648 |  0.000001 |   0.0 |   0.0 | 0.210527 | 0.077549 |
    | 04.01. |  0.0 | 21.8 |  7.861915 |   0.0 | 22.6 | 3.144766 | 1.572383 |       0.0 |  0.0 |      0.0 | 0.0 | 11.583298 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.264636 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.056204 |       0.0 |   0.0 |   0.0 | 0.208432 |  0.07351 |
    | 04.01. |  0.0 | 22.2 |  6.298329 |   0.0 | 23.0 | 2.519332 | 1.259666 |       0.0 |  0.0 |      0.0 | 0.0 | 11.788313 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.252374 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.046016 |       0.0 |   0.0 |   0.0 | 0.206358 | 0.070104 |
    | 04.01. |  0.0 | 20.1 |  2.948416 |   0.0 | 20.9 | 1.179366 | 0.589683 |       0.0 |  0.0 |      0.0 | 0.0 | 10.711988 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |   0.24198 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.037675 |       0.0 |   0.0 |   0.0 | 0.204305 | 0.067217 |
    | 04.01. |  0.0 | 17.8 |  1.309232 |   0.0 | 18.6 | 0.523693 | 0.261846 |       0.0 |  0.0 |      0.0 | 0.0 |  9.533157 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.233117 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.030845 |       0.0 |   0.0 |   0.0 | 0.202272 | 0.064755 |
    | 04.01. |  0.0 | 15.2 |   0.32955 |   0.0 | 16.0 |  0.13182 |  0.06591 |       0.0 |  0.0 |      0.0 | 0.0 |  8.200565 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.225513 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.025254 |       0.0 |   0.0 |   0.0 | 0.200259 | 0.062643 |
    | 04.01. |  0.0 | 14.5 |  0.089508 |   0.0 | 15.3 | 0.035803 | 0.017902 |       0.0 |  0.0 |      0.0 | 0.0 |  7.841791 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.218943 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.020676 |       0.0 |   0.0 |   0.0 | 0.198267 | 0.060817 |
    | 04.01. |  0.0 | 12.4 |  0.085771 |   0.0 | 13.2 | 0.034308 | 0.017154 |       0.0 |  0.0 |      0.0 | 0.0 |  6.765466 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.213222 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.016928 |       0.0 |   0.0 |   0.0 | 0.196294 | 0.059228 |
    | 04.01. |  0.0 | 11.7 |    0.0845 |   0.0 | 12.5 |   0.0338 |   0.0169 |       0.0 |  0.0 |      0.0 | 0.0 |  6.406692 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |    0.2082 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.01386 |       0.0 |   0.0 |   0.0 | 0.194341 | 0.057833 |
    | 04.01. |  0.0 | 11.9 |  0.084864 |   0.0 | 12.7 | 0.033946 | 0.016973 |       0.0 |  0.0 |      0.0 | 0.0 |  6.509199 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |       0.0 |  0.203754 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.011347 |       0.0 |   0.0 |   0.0 | 0.192407 | 0.056598 |

    .. raw:: html

        <iframe
            src="lland_v2_ex4.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>

    :ref:`Recalculation of example 6 <lland_v1_ex6>`

    The following calculation shows, that the outflow values of the
    integration test for snow events are reproduced exactly (note that
    the |PET| values have to be adapted to the changed |TemL| values):

    >>> lnk(ACKER)
    >>> inputs.teml.series = numpy.linspace(-10.0, 10.0, 96)
    >>> inputs.pet.series = (
    ...     0.03733, 0.037872, 0.038415, 0.038955, 0.039492, 0.040028,
    ...     0.110905, 0.71239, 1.641935, 2.661377, 3.014447, 3.343815,
    ...     1.81419, 2.535277, 3.137525, 1.010042, 0.776177, 0.503072,
    ...     0.346957, 0.105225, 0.047837, 0.048345, 0.048848, 0.04935,
    ...     0.04985, 0.05035, 0.050845, 0.051342, 0.051835, 0.052327,
    ...     0.102702, 0.69626, 1.692095, 1.078725, 1.382545, 3.516795,
    ...     1.935692, 5.120597, 2.372527, 1.267695, 1.365632, 1.772603,
    ...     0.997112, 0.185172, 0.0595, 0.059965, 0.060428, 0.06089, 0.06135,
    ...     0.06181, 0.062267, 0.062723, 0.063177, 0.06363, 0.10774,
    ...     0.325317, 0.811522, 2.973865, 3.12363, 3.798742, 3.874722,
    ...     3.373822, 5.667417, 3.434725, 3.787713, 2.368437, 1.120002,
    ...     0.211415, 0.070242, 0.07067, 0.0711, 0.071525, 0.07195, 0.072375,
    ...     0.072795, 0.073217, 0.073637, 0.074055, 0.2705, 1.23095,
    ...     2.866595, 4.66193, 6.271463, 6.833792, 6.738053, 8.078045,
    ...     7.222285, 6.044787, 4.837587, 2.353282, 1.092032, 0.289745,
    ...     0.080167, 0.080565, 0.080962, 0.081357)
    >>> test('lland_v2_ex6')
    |   date | nied |      teml |      pet |  nkor |      tkor |      et0 |     evpo |      nbes |      sbes |      evi |      evb |     wgtf |     schm |     wada |      qdb |     qib1 |     qib2 |      qbb |     qdgz |        q |     inzp |       wats |       waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 |     -10.0 |  0.03733 |   0.0 |      -9.2 | 0.014932 | 0.007466 |       0.0 |       0.0 |      0.0 | 0.006413 |      0.0 |      0.0 |      0.0 |      0.0 |     0.05 |      0.0 |     0.45 |      0.0 |  0.50098 |      0.0 |        0.0 |        0.0 |  99.493587 |      0.0 |      0.0 |     0.05 |      0.0 |     0.45 |      0.0 |      0.0 | 0.001229 |      0.0 | 0.499751 | 0.139161 |
    | 01.01. |  0.0 | -9.789474 | 0.037872 |   0.0 | -8.989474 | 0.015149 | 0.007574 |       0.0 |       0.0 |      0.0 | 0.006492 |      0.0 |      0.0 |      0.0 |      0.0 | 0.049747 |      0.0 | 0.447468 |      0.0 | 0.502845 |      0.0 |        0.0 |        0.0 |   98.98988 |      0.0 |      0.0 | 0.049747 |      0.0 | 0.447468 |      0.0 |      0.0 | 0.003602 |      0.0 | 0.499243 | 0.139679 |
    | 01.01. |  0.0 | -9.578947 | 0.038415 |   0.0 | -8.778947 | 0.015366 | 0.007683 |       0.0 |       0.0 |      0.0 | 0.006572 |      0.0 |      0.0 |      0.0 |      0.0 | 0.049495 |      0.0 | 0.444949 |      0.0 | 0.504562 |      0.0 |        0.0 |        0.0 |  98.488864 |      0.0 |      0.0 | 0.049495 |      0.0 | 0.444949 |      0.0 |      0.0 | 0.005846 |      0.0 | 0.498715 | 0.140156 |
    | 01.01. |  0.0 | -9.368421 | 0.038955 |   0.0 | -8.568421 | 0.015582 | 0.007791 |       0.0 |       0.0 |      0.0 |  0.00665 |      0.0 |      0.0 |      0.0 |      0.0 | 0.049244 |      0.0 | 0.442444 |      0.0 | 0.506137 |      0.0 |        0.0 |        0.0 |  97.990525 |      0.0 |      0.0 | 0.049244 |      0.0 | 0.442444 |      0.0 |      0.0 | 0.007969 |      0.0 | 0.498168 | 0.140594 |
    | 01.01. |  0.0 | -9.157895 | 0.039492 |   0.0 | -8.357895 | 0.015797 | 0.007898 |       0.0 |       0.0 |      0.0 | 0.006727 |      0.0 |      0.0 |      0.0 |      0.0 | 0.048995 |      0.0 | 0.439953 |      0.0 | 0.507577 |      0.0 |        0.0 |        0.0 |  97.494851 |      0.0 |      0.0 | 0.048995 |      0.0 | 0.439953 |      0.0 |      0.0 | 0.009976 |      0.0 | 0.497601 | 0.140994 |
    | 01.01. |  0.0 | -8.947368 | 0.040028 |   0.0 | -8.147368 | 0.016011 | 0.008006 |       0.0 |       0.0 |      0.0 | 0.006803 |      0.0 |      0.0 |      0.0 |      0.0 | 0.048747 |      0.0 | 0.437474 |      0.0 | 0.508888 |      0.0 |        0.0 |        0.0 |  97.001826 |      0.0 |      0.0 | 0.048747 |      0.0 | 0.437474 |      0.0 |      0.0 | 0.011873 |      0.0 | 0.497015 | 0.141358 |
    | 01.01. |  0.0 | -8.736842 | 0.110905 |   0.0 | -7.936842 | 0.044362 | 0.022181 |       0.0 |       0.0 |      0.0 | 0.018808 |      0.0 |      0.0 |      0.0 |      0.0 | 0.048501 |      0.0 | 0.435009 |      0.0 | 0.510075 |      0.0 |        0.0 |        0.0 |  96.499508 |      0.0 |      0.0 | 0.048501 |      0.0 | 0.435009 |      0.0 |      0.0 | 0.013665 |      0.0 |  0.49641 | 0.141688 |
    | 01.01. |  0.0 | -8.526316 |  0.71239 |   0.0 | -7.726316 | 0.284956 | 0.142478 |       0.0 |       0.0 |      0.0 | 0.120538 |      0.0 |      0.0 |      0.0 |      0.0 |  0.04825 |      0.0 | 0.432498 |      0.0 | 0.511145 |      0.0 |        0.0 |        0.0 |  95.898222 |      0.0 |      0.0 |  0.04825 |      0.0 | 0.432498 |      0.0 |      0.0 | 0.015358 |      0.0 | 0.495787 | 0.141985 |
    | 01.01. |  0.0 | -8.315789 | 1.641935 |   0.0 | -7.515789 | 0.656774 | 0.328387 |       0.0 |       0.0 |      0.0 | 0.277056 |      0.0 |      0.0 |      0.0 |      0.0 | 0.047949 |      0.0 | 0.429491 |      0.0 | 0.512097 |      0.0 |        0.0 |        0.0 |  95.143726 |      0.0 |      0.0 | 0.047949 |      0.0 | 0.429491 |      0.0 |      0.0 | 0.016954 |      0.0 | 0.495142 | 0.142249 |
    | 01.01. |  0.0 | -8.105263 | 2.661377 |   0.0 | -7.305263 | 1.064551 | 0.532275 |       0.0 |       0.0 |      0.0 | 0.447499 |      0.0 |      0.0 |      0.0 |      0.0 | 0.047572 |      0.0 | 0.425719 |      0.0 | 0.512927 |      0.0 |        0.0 |        0.0 |  94.222936 |      0.0 |      0.0 | 0.047572 |      0.0 | 0.425719 |      0.0 |      0.0 | 0.018457 |      0.0 |  0.49447 |  0.14248 |
    | 01.01. |  0.0 | -7.894737 | 3.014447 |   0.0 | -7.094737 | 1.205779 | 0.602889 |       0.0 |       0.0 |      0.0 | 0.504652 |      0.0 |      0.0 |      0.0 |      0.0 | 0.047111 |      0.0 | 0.421115 |      0.0 | 0.513629 |      0.0 |        0.0 |        0.0 |  93.250058 |      0.0 |      0.0 | 0.047111 |      0.0 | 0.421115 |      0.0 |      0.0 | 0.019865 |      0.0 | 0.493763 | 0.142675 |
    | 01.01. |  0.0 | -7.684211 | 3.343815 |   0.0 | -6.884211 | 1.337526 | 0.668763 |       0.0 |       0.0 |      0.0 | 0.557145 |      0.0 |      0.0 |      0.0 |      0.0 | 0.046625 |      0.0 |  0.41625 |      0.0 | 0.514198 |      0.0 |        0.0 |        0.0 |  92.230038 |      0.0 |      0.0 | 0.046625 |      0.0 |  0.41625 |      0.0 |      0.0 | 0.021182 |      0.0 | 0.493016 | 0.142833 |
    | 01.01. |  0.0 | -7.473684 |  1.81419 |   0.0 | -6.673684 | 0.725676 | 0.362838 |       0.0 |       0.0 |      0.0 | 0.300742 |      0.0 |      0.0 |      0.0 |      0.0 | 0.046115 |      0.0 |  0.41115 |      0.0 | 0.514637 |      0.0 |        0.0 |        0.0 |  91.472031 |      0.0 |      0.0 | 0.046115 |      0.0 |  0.41115 |      0.0 |      0.0 | 0.022411 |      0.0 | 0.492227 | 0.142955 |
    | 01.01. |  0.0 | -7.263158 | 2.535277 |   0.0 | -6.463158 | 1.014111 | 0.507055 |       0.0 |       0.0 |      0.0 | 0.418653 |      0.0 |      0.0 |      0.0 |      0.0 | 0.045736 |      0.0 |  0.40736 |      0.0 | 0.514959 |      0.0 |        0.0 |        0.0 |  90.600281 |      0.0 |      0.0 | 0.045736 |      0.0 |  0.40736 |      0.0 |      0.0 | 0.023557 |      0.0 | 0.491401 | 0.143044 |
    | 01.01. |  0.0 | -7.052632 | 3.137525 |   0.0 | -6.252632 |  1.25501 | 0.627505 |       0.0 |       0.0 |      0.0 | 0.515752 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0453 |      0.0 | 0.403001 |      0.0 | 0.515172 |      0.0 |        0.0 |        0.0 |  89.636228 |      0.0 |      0.0 |   0.0453 |      0.0 | 0.403001 |      0.0 |      0.0 | 0.024628 |      0.0 | 0.490543 | 0.143103 |
    | 01.01. |  0.2 | -6.842105 | 1.010042 |  0.24 | -6.042105 | 0.404017 | 0.202008 |       0.0 |       0.0 | 0.202008 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.044818 |      0.0 | 0.398181 |      0.0 | 0.515273 | 0.037992 |        0.0 |        0.0 |  89.193229 |      0.0 |      0.0 | 0.044818 |      0.0 | 0.398181 |      0.0 |      0.0 | 0.025625 |      0.0 | 0.489648 | 0.143131 |
    | 01.01. |  0.0 | -6.631579 | 0.776177 |   0.0 | -5.831579 | 0.310471 | 0.155235 |       0.0 |       0.0 | 0.037992 | 0.095638 |      0.0 |      0.0 |      0.0 |      0.0 | 0.044597 |      0.0 | 0.395966 |      0.0 | 0.515282 |      0.0 |        0.0 |        0.0 |  88.657028 |      0.0 |      0.0 | 0.044597 |      0.0 | 0.395966 |      0.0 |      0.0 | 0.026555 |      0.0 | 0.488727 | 0.143134 |
    | 01.01. |  0.0 | -6.421053 | 0.503072 |   0.0 | -5.621053 | 0.201229 | 0.100614 |       0.0 |       0.0 |      0.0 | 0.081831 |      0.0 |      0.0 |      0.0 |      0.0 | 0.044329 |      0.0 | 0.393285 |      0.0 | 0.515219 |      0.0 |        0.0 |        0.0 |  88.137583 |      0.0 |      0.0 | 0.044329 |      0.0 | 0.393285 |      0.0 |      0.0 | 0.027429 |      0.0 | 0.487791 | 0.143116 |
    | 01.01. |  1.3 | -6.210526 | 0.346957 |  1.56 | -5.410526 | 0.138783 | 0.069391 |      0.76 |      0.76 | 0.069391 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.044069 |      0.0 | 0.390688 |      0.0 | 0.515084 | 0.730609 |       0.76 |       0.76 |  87.702827 |      0.0 |      0.0 | 0.044069 |      0.0 | 0.390688 |      0.0 |      0.0 | 0.028246 |      0.0 | 0.486837 | 0.143079 |
    | 01.01. |  5.6 |      -6.0 | 0.105225 |  6.72 |      -5.2 |  0.04209 | 0.021045 |  6.650609 |  6.650609 | 0.021045 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.043851 |      0.0 | 0.388514 |      0.0 | 0.514883 | 0.778955 |   7.410609 |   7.410609 |  87.270461 |      0.0 |      0.0 | 0.043851 |      0.0 | 0.388514 |      0.0 |      0.0 | 0.029013 |      0.0 |  0.48587 | 0.143023 |
    | 01.01. |  2.9 | -5.789474 | 0.047837 |  3.48 | -4.989474 | 0.019135 | 0.009567 |  3.458955 |  3.458955 | 0.009567 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.043635 |      0.0 | 0.386352 |      0.0 | 0.514621 | 0.790433 |  10.869564 |  10.869564 |  86.840473 |      0.0 |      0.0 | 0.043635 |      0.0 | 0.386352 |      0.0 |      0.0 | 0.029731 |      0.0 |  0.48489 |  0.14295 |
    | 01.01. |  4.9 | -5.578947 | 0.048345 |  5.88 | -4.778947 | 0.019338 | 0.009669 |  5.870433 |  5.870433 | 0.009669 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |  0.04342 |      0.0 | 0.384202 |      0.0 | 0.514303 | 0.790331 |  16.739996 |  16.739996 |  86.412851 |      0.0 |      0.0 |  0.04342 |      0.0 | 0.384202 |      0.0 |      0.0 | 0.030404 |      0.0 | 0.483899 | 0.142862 |
    | 01.01. | 10.6 | -5.368421 | 0.048848 | 12.72 | -4.568421 | 0.019539 |  0.00977 | 12.710331 | 12.710331 |  0.00977 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.043206 |      0.0 | 0.382064 |      0.0 |  0.51393 |  0.79023 |  29.450327 |  29.450327 |   85.98758 |      0.0 |      0.0 | 0.043206 |      0.0 | 0.382064 |      0.0 |      0.0 | 0.031033 |      0.0 | 0.482897 | 0.142758 |
    | 01.01. |  0.1 | -5.157895 |  0.04935 |  0.12 | -4.357895 |  0.01974 |  0.00987 |   0.11023 |   0.11023 |  0.00987 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.042994 |      0.0 | 0.379938 |      0.0 | 0.513505 |  0.79013 |  29.560558 |  29.560558 |  85.564648 |      0.0 |      0.0 | 0.042994 |      0.0 | 0.379938 |      0.0 |      0.0 | 0.031622 |      0.0 | 0.481883 |  0.14264 |
    | 02.01. |  0.7 | -4.947368 |  0.04985 |  0.84 | -4.147368 |  0.01994 |  0.00997 |   0.83013 |   0.83013 |  0.00997 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.042782 |      0.0 | 0.377823 |      0.0 | 0.513029 |  0.79003 |  30.390688 |  30.390688 |  85.144043 |      0.0 |      0.0 | 0.042782 |      0.0 | 0.377823 |      0.0 |      0.0 | 0.032171 |      0.0 | 0.480858 | 0.142508 |
    | 02.01. |  3.0 | -4.736842 |  0.05035 |   3.6 | -3.936842 |  0.02014 |  0.01007 |   3.59003 |   3.59003 |  0.01007 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.042572 |      0.0 |  0.37572 |      0.0 | 0.512506 |  0.78993 |  33.980718 |  33.980718 |  84.725751 |      0.0 |      0.0 | 0.042572 |      0.0 |  0.37572 |      0.0 |      0.0 | 0.032684 |      0.0 | 0.479822 | 0.142363 |
    | 02.01. |  2.1 | -4.526316 | 0.050845 |  2.52 | -3.726316 | 0.020338 | 0.010169 |   2.50993 |   2.50993 | 0.010169 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.042363 |      0.0 | 0.373629 |      0.0 | 0.511937 | 0.789831 |  36.490648 |  36.490648 |  84.309759 |      0.0 |      0.0 | 0.042363 |      0.0 | 0.373629 |      0.0 |      0.0 | 0.033161 |      0.0 | 0.478776 | 0.142205 |
    | 02.01. | 10.4 | -4.315789 | 0.051342 | 12.48 | -3.515789 | 0.020537 | 0.010268 | 12.469831 | 12.469831 | 0.010268 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.042155 |      0.0 | 0.371549 |      0.0 | 0.511324 | 0.789732 |  48.960479 |  48.960479 |  83.896055 |      0.0 |      0.0 | 0.042155 |      0.0 | 0.371549 |      0.0 |      0.0 | 0.033604 |      0.0 | 0.477719 | 0.142034 |
    | 02.01. |  3.5 | -4.105263 | 0.051835 |   4.2 | -3.305263 | 0.020734 | 0.010367 |  4.189732 |  4.189732 | 0.010367 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.041948 |      0.0 |  0.36948 |      0.0 | 0.510669 | 0.789633 |   53.15021 |   53.15021 |  83.484627 |      0.0 |      0.0 | 0.041948 |      0.0 |  0.36948 |      0.0 |      0.0 | 0.034016 |      0.0 | 0.476652 | 0.141852 |
    | 02.01. |  3.4 | -3.894737 | 0.052327 |  4.08 | -3.094737 | 0.020931 | 0.010465 |  4.069633 |  4.069633 | 0.010465 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.041742 |      0.0 | 0.367423 |      0.0 | 0.509974 | 0.789535 |  57.219843 |  57.219843 |  83.075462 |      0.0 |      0.0 | 0.041742 |      0.0 | 0.367423 |      0.0 |      0.0 | 0.034398 |      0.0 | 0.475576 | 0.141659 |
    | 02.01. |  1.2 | -3.684211 | 0.102702 |  1.44 | -2.884211 | 0.041081 |  0.02054 |  1.429535 |  1.429535 |  0.02054 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.041538 |      0.0 | 0.365377 |      0.0 | 0.509241 |  0.77946 |  58.649378 |  58.649378 |  82.668547 |      0.0 |      0.0 | 0.041538 |      0.0 | 0.365377 |      0.0 |      0.0 | 0.034751 |      0.0 |  0.47449 | 0.141456 |
    | 02.01. |  0.1 | -3.473684 |  0.69626 |  0.12 | -2.673684 | 0.278504 | 0.139252 |   0.09946 |   0.09946 | 0.139252 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.041334 |      0.0 | 0.363343 |      0.0 | 0.508471 | 0.660748 |  58.748837 |  58.748837 |   82.26387 |      0.0 |      0.0 | 0.041334 |      0.0 | 0.363343 |      0.0 |      0.0 | 0.035077 |      0.0 | 0.473394 | 0.141242 |
    | 02.01. |  0.0 | -3.263158 | 1.692095 |   0.0 | -2.463158 | 0.676838 | 0.338419 |       0.0 |       0.0 | 0.338419 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.041132 |      0.0 | 0.361319 |      0.0 | 0.507666 | 0.322329 |  58.748837 |  58.748837 |  81.861418 |      0.0 |      0.0 | 0.041132 |      0.0 | 0.361319 |      0.0 |      0.0 | 0.035377 |      0.0 | 0.472289 | 0.141018 |
    | 02.01. |  0.0 | -3.052632 | 1.078725 |   0.0 | -2.252632 |  0.43149 | 0.215745 |       0.0 |       0.0 | 0.215745 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.040931 |      0.0 | 0.359307 |      0.0 | 0.506828 | 0.106584 |  58.748837 |  58.748837 |   81.46118 |      0.0 |      0.0 | 0.040931 |      0.0 | 0.359307 |      0.0 |      0.0 | 0.035653 |      0.0 | 0.471174 | 0.140785 |
    | 02.01. |  0.4 | -2.842105 | 1.382545 |  0.48 | -2.042105 | 0.553018 | 0.276509 |       0.0 |       0.0 | 0.276509 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.040731 |      0.0 | 0.357306 |      0.0 | 0.505957 | 0.310075 |  58.748837 |  58.748837 |  81.063144 |      0.0 |      0.0 | 0.040731 |      0.0 | 0.357306 |      0.0 |      0.0 | 0.035906 |      0.0 | 0.470051 | 0.140544 |
    | 02.01. |  0.1 | -2.631579 | 3.516795 |  0.12 | -1.831579 | 1.406718 | 0.703359 |       0.0 |       0.0 | 0.430075 | 0.212168 |      0.0 |      0.0 |      0.0 |      0.0 | 0.040532 |      0.0 | 0.355316 |      0.0 | 0.505056 |      0.0 |  58.748837 |  58.748837 |  80.455129 |      0.0 |      0.0 | 0.040532 |      0.0 | 0.355316 |      0.0 |      0.0 | 0.036136 |      0.0 |  0.46892 | 0.140293 |
    | 02.01. |  3.6 | -2.421053 | 1.935692 |  4.32 | -1.621053 | 0.774277 | 0.387138 |      3.52 |      3.52 | 0.387138 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.040228 |      0.0 | 0.352276 |      0.0 | 0.504117 | 0.412862 |  62.268837 |  62.268837 |  80.062626 |      0.0 |      0.0 | 0.040228 |      0.0 | 0.352276 |      0.0 |      0.0 | 0.036343 |      0.0 | 0.467774 | 0.140032 |
    | 02.01. |  5.9 | -2.210526 | 5.120597 |  7.08 | -1.410526 | 2.048239 | 1.024119 |  6.692862 |  6.692862 |      0.8 | 0.172813 |      0.0 |      0.0 |      0.0 |      0.0 | 0.040031 |      0.0 | 0.350313 |      0.0 | 0.503143 |      0.0 |  68.961699 |  68.961699 |  79.499468 |      0.0 |      0.0 | 0.040031 |      0.0 | 0.350313 |      0.0 |      0.0 | 0.036528 |      0.0 | 0.466615 | 0.139762 |
    | 02.01. |  1.1 |      -2.0 | 2.372527 |  1.32 |      -1.2 | 0.949011 | 0.474505 |      0.52 |      0.52 | 0.474505 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |  0.03975 |      0.0 | 0.347497 |      0.0 | 0.502135 | 0.325495 |  69.481699 |  69.481699 |  79.112221 |      0.0 |      0.0 |  0.03975 |      0.0 | 0.347497 |      0.0 |      0.0 | 0.036692 |      0.0 | 0.465444 | 0.139482 |
    | 02.01. | 20.7 | -1.789474 | 1.267695 | 24.84 | -0.989474 | 0.507078 | 0.253539 | 24.365495 | 24.365495 | 0.253539 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.039556 |      0.0 | 0.345561 |      0.0 | 0.501096 | 0.546461 |  93.847194 |  93.847194 |  78.727104 |      0.0 |      0.0 | 0.039556 |      0.0 | 0.345561 |      0.0 |      0.0 | 0.036836 |      0.0 | 0.464261 | 0.139193 |
    | 02.01. | 37.9 | -1.578947 | 1.365632 | 45.48 | -0.778947 | 0.546253 | 0.273126 | 45.226461 | 45.226461 | 0.273126 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.039364 |      0.0 | 0.343636 |      0.0 | 0.500034 | 0.526874 | 139.073655 | 139.073655 |  78.344105 |      0.0 |      0.0 | 0.039364 |      0.0 | 0.343636 |      0.0 |      0.0 | 0.036964 |      0.0 |  0.46307 | 0.138898 |
    | 02.01. |  8.2 | -1.368421 | 1.772603 |  9.84 | -0.568421 | 0.709041 | 0.354521 |  9.566874 |  9.566874 | 0.354521 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.039172 |      0.0 | 0.341721 |      0.0 | 0.498948 | 0.445479 | 148.640528 | 148.640528 |  77.963212 |      0.0 |      0.0 | 0.039172 |      0.0 | 0.341721 |      0.0 |      0.0 | 0.037076 |      0.0 | 0.461872 | 0.138597 |
    | 02.01. |  3.6 | -1.157895 | 0.997112 |  4.32 | -0.357895 | 0.398845 | 0.199422 |  3.965479 |  3.965479 | 0.199422 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.038982 |      0.0 | 0.339816 |      0.0 | 0.497841 | 0.600578 | 152.606008 | 152.606008 |  77.584415 |      0.0 |      0.0 | 0.038982 |      0.0 | 0.339816 |      0.0 |      0.0 | 0.037174 |      0.0 | 0.460667 | 0.138289 |
    | 02.01. |  7.5 | -0.947368 | 0.185172 |   9.0 | -0.147368 | 0.074069 | 0.037034 |  8.800578 |  8.800578 | 0.037034 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.038792 |      0.0 | 0.337922 |      0.0 | 0.496712 | 0.762966 | 161.406585 | 161.406585 |    77.2077 |      0.0 |      0.0 | 0.038792 |      0.0 | 0.337922 |      0.0 |      0.0 | 0.037257 |      0.0 | 0.459455 | 0.137976 |
    | 02.01. | 18.5 | -0.736842 |   0.0595 |  22.2 |  0.063158 |   0.0238 |   0.0119 | 22.162966 | 21.463082 |   0.0119 |      0.0 | 0.032371 | 0.032371 |      0.0 |      0.0 | 0.038604 |      0.0 | 0.336039 |      0.0 | 0.495564 |   0.7881 | 182.837297 | 183.569551 |  76.833058 |      0.0 |      0.0 | 0.038604 |      0.0 | 0.336039 |      0.0 |      0.0 | 0.037327 |      0.0 | 0.458236 | 0.137657 |
    | 02.01. | 15.4 | -0.526316 | 0.059965 | 18.48 |  0.273684 | 0.023986 | 0.011993 |   18.4681 | 15.940886 | 0.011993 |      0.0 | 0.140273 | 0.140273 |      0.0 |      0.0 | 0.038417 |      0.0 | 0.334165 |      0.0 | 0.494396 | 0.788007 | 198.637911 | 202.037651 |  76.460476 |      0.0 |      0.0 | 0.038417 |      0.0 | 0.334165 |      0.0 |      0.0 | 0.037385 |      0.0 | 0.457011 | 0.137332 |
    | 02.01. |  6.3 | -0.315789 | 0.060428 |  7.56 |  0.484211 | 0.024171 | 0.012086 |  7.548007 |  5.720595 | 0.012086 |      0.0 | 0.248175 | 0.248175 |      0.0 |      0.0 |  0.03823 |      0.0 | 0.332302 |      0.0 |  0.49321 | 0.787914 |  204.11033 | 209.585658 |  76.089944 |      0.0 |      0.0 |  0.03823 |      0.0 | 0.332302 |      0.0 |      0.0 | 0.037431 |      0.0 | 0.455779 | 0.137003 |
    | 02.01. |  1.9 | -0.105263 |  0.06089 |  2.28 |  0.694737 | 0.024356 | 0.012178 |  2.267914 |  1.480113 | 0.012178 |      0.0 | 0.356077 | 0.356077 |      0.0 |      0.0 | 0.038045 |      0.0 |  0.33045 |      0.0 | 0.492007 | 0.787822 | 205.234366 | 211.853572 |  75.721449 |      0.0 |      0.0 | 0.038045 |      0.0 |  0.33045 |      0.0 |      0.0 | 0.037465 |      0.0 | 0.454542 | 0.136669 |
    | 03.01. |  4.9 |  0.105263 |  0.06135 |  5.88 |  0.905263 |  0.02454 |  0.01227 |  5.867822 |   3.21186 |  0.01227 |      0.0 | 0.463979 | 0.463979 |      0.0 |      0.0 | 0.037861 |      0.0 | 0.328607 |      0.0 | 0.490787 |  0.78773 | 207.982247 | 217.721394 |  75.354981 |      0.0 |      0.0 | 0.037861 |      0.0 | 0.328607 |      0.0 |      0.0 | 0.037489 |      0.0 | 0.453298 |  0.13633 |
    | 03.01. |  2.7 |  0.315789 |  0.06181 |  3.24 |  1.115789 | 0.024724 | 0.012362 |   3.22773 |  1.426996 | 0.012362 |      0.0 | 0.571882 | 0.571882 |      0.0 |      0.0 | 0.037677 |      0.0 | 0.326775 |      0.0 | 0.489551 | 0.787638 | 208.837362 | 220.949124 |  74.990528 |      0.0 |      0.0 | 0.037677 |      0.0 | 0.326775 |      0.0 |      0.0 | 0.037503 |      0.0 | 0.452048 | 0.135986 |
    | 03.01. |  0.5 |  0.526316 | 0.062267 |   0.6 |  1.326316 | 0.024907 | 0.012453 |  0.587638 |  0.197941 | 0.012453 |      0.0 | 0.679784 | 0.679784 |      0.0 |      0.0 | 0.037495 |      0.0 | 0.324953 |      0.0 | 0.488299 | 0.787547 | 208.355519 | 221.536762 |  74.628081 |      0.0 |      0.0 | 0.037495 |      0.0 | 0.324953 |      0.0 |      0.0 | 0.037507 |      0.0 | 0.450792 | 0.135639 |
    | 03.01. |  0.2 |  0.736842 | 0.062723 |  0.24 |  1.536842 | 0.025089 | 0.012545 |  0.227547 |  0.052695 | 0.012545 |      0.0 | 0.787686 | 0.787686 |      0.0 |      0.0 | 0.037314 |      0.0 |  0.32314 |      0.0 | 0.487033 | 0.787455 | 207.620528 | 221.764309 |  74.267626 |      0.0 |      0.0 | 0.037314 |      0.0 |  0.32314 |      0.0 |      0.0 | 0.037502 |      0.0 | 0.449531 | 0.135287 |
    | 03.01. |  0.5 |  0.947368 | 0.063177 |   0.6 |  1.747368 | 0.025271 | 0.012635 |  0.587455 |  0.074205 | 0.012635 |      0.0 | 0.895588 | 0.895588 |      0.0 |      0.0 | 0.037134 |      0.0 | 0.321338 |      0.0 | 0.485753 | 0.787365 | 206.799145 | 222.351764 |  73.909154 |      0.0 |      0.0 | 0.037134 |      0.0 | 0.321338 |      0.0 |      0.0 | 0.037488 |      0.0 | 0.448265 | 0.134931 |
    | 03.01. |  2.4 |  1.157895 |  0.06363 |  2.88 |  1.957895 | 0.025452 | 0.012726 |  2.867365 |  0.060366 | 0.012726 |      0.0 |  1.00349 |  1.00349 |      0.0 |      0.0 | 0.036955 |      0.0 | 0.319546 |      0.0 | 0.484459 | 0.787274 | 205.856021 | 225.219129 |  73.552654 |      0.0 |      0.0 | 0.036955 |      0.0 | 0.319546 |      0.0 |      0.0 | 0.037466 |      0.0 | 0.446993 | 0.134572 |
    | 03.01. |  0.4 |  1.368421 |  0.10774 |  0.48 |  2.168421 | 0.043096 | 0.021548 |  0.467274 |       0.0 | 0.021548 |      0.0 | 1.111392 | 1.111392 |      0.0 |      0.0 | 0.036776 |      0.0 | 0.317763 |      0.0 | 0.483153 | 0.778452 | 204.744628 | 225.686403 |  73.198114 |      0.0 |      0.0 | 0.036776 |      0.0 | 0.317763 |      0.0 |      0.0 | 0.037437 |      0.0 | 0.445716 | 0.134209 |
    | 03.01. |  0.2 |  1.578947 | 0.325317 |  0.24 |  2.378947 | 0.130127 | 0.065063 |  0.218452 |       0.0 | 0.065063 |      0.0 | 1.219295 | 1.219295 |      0.0 |      0.0 | 0.036599 |      0.0 | 0.315991 |      0.0 | 0.481834 | 0.734937 | 203.525334 | 225.904855 |  72.845525 |      0.0 |      0.0 | 0.036599 |      0.0 | 0.315991 |      0.0 |      0.0 |   0.0374 |      0.0 | 0.444434 | 0.133843 |
    | 03.01. |  0.0 |  1.789474 | 0.811522 |   0.0 |  2.589474 | 0.324609 | 0.162304 |       0.0 |       0.0 | 0.162304 |      0.0 | 1.327197 | 1.327197 |      0.0 |      0.0 | 0.036423 |      0.0 | 0.314228 |      0.0 | 0.480504 | 0.572632 | 202.198137 | 225.904855 |  72.494874 |      0.0 |      0.0 | 0.036423 |      0.0 | 0.314228 |      0.0 |      0.0 | 0.037357 |      0.0 | 0.443147 | 0.133473 |
    | 03.01. |  0.0 |       2.0 | 2.973865 |   0.0 |       2.8 | 1.189546 | 0.594773 |       0.0 |       0.0 | 0.572632 | 0.016112 | 1.435099 | 1.435099 |      0.0 |      0.0 | 0.036247 |      0.0 | 0.312474 |      0.0 | 0.479163 |      0.0 | 200.763038 | 225.904855 |   72.13004 |      0.0 |      0.0 | 0.036247 |      0.0 | 0.312474 |      0.0 |      0.0 | 0.037307 |      0.0 | 0.441855 | 0.133101 |
    | 03.01. |  0.3 |  2.210526 |  3.12363 |  0.36 |  3.010526 | 1.249452 | 0.624726 |       0.0 |       0.0 |     0.36 | 0.192054 | 1.543001 | 1.543001 |      0.0 |      0.0 | 0.036065 |      0.0 |  0.31065 |      0.0 |  0.47781 |      0.0 | 199.220037 | 225.904855 |  71.591271 |      0.0 |      0.0 | 0.036065 |      0.0 |  0.31065 |      0.0 |      0.0 | 0.037251 |      0.0 | 0.440559 | 0.132725 |
    | 03.01. |  2.6 |  2.421053 | 3.798742 |  3.12 |  3.221053 | 1.519497 | 0.759748 |      2.32 |       0.0 | 0.759748 |      0.0 | 1.650903 | 1.650903 |      0.0 |      0.0 | 0.035796 |      0.0 | 0.307956 |      0.0 | 0.476439 | 0.040252 | 197.569134 | 228.224855 |  71.247519 |      0.0 |      0.0 | 0.035796 |      0.0 | 0.307956 |      0.0 |      0.0 | 0.037187 |      0.0 | 0.439253 | 0.132344 |
    | 03.01. |  0.7 |  2.631579 | 3.874722 |  0.84 |  3.431579 | 1.549889 | 0.774944 |  0.080252 |       0.0 | 0.774944 |      0.0 | 1.758805 | 1.758805 |      0.0 |      0.0 | 0.035624 |      0.0 | 0.306238 |      0.0 | 0.475052 | 0.025056 | 195.810328 | 228.305106 |  70.905657 |      0.0 |      0.0 | 0.035624 |      0.0 | 0.306238 |      0.0 |      0.0 | 0.037114 |      0.0 | 0.437938 | 0.131959 |
    | 03.01. |  0.3 |  2.842105 | 3.373822 |  0.36 |  3.642105 | 1.349529 | 0.674764 |       0.0 |       0.0 | 0.385056 | 0.207965 | 1.866708 | 1.866708 |      0.0 |      0.0 | 0.035453 |      0.0 | 0.304528 |      0.0 | 0.473657 |      0.0 |  193.94362 | 228.305106 |  70.357712 |      0.0 |      0.0 | 0.035453 |      0.0 | 0.304528 |      0.0 |      0.0 | 0.037038 |      0.0 | 0.436619 | 0.131571 |
    | 03.01. |  0.3 |  3.052632 | 5.667417 |  0.36 |  3.852632 | 2.266967 | 1.133483 |       0.0 |       0.0 |     0.36 | 0.552551 |  1.97461 |  1.97461 |      0.0 |      0.0 | 0.035179 |      0.0 | 0.301789 |      0.0 | 0.472245 |      0.0 | 191.969011 | 228.305106 |  69.468193 |      0.0 |      0.0 | 0.035179 |      0.0 | 0.301789 |      0.0 |      0.0 | 0.036954 |      0.0 | 0.435291 | 0.131179 |
    | 03.01. |  0.0 |  3.263158 | 3.434725 |   0.0 |  4.063158 |  1.37389 | 0.686945 |       0.0 |       0.0 |      0.0 | 0.486808 | 2.082512 | 2.082512 |      0.0 |      0.0 | 0.034734 |      0.0 | 0.297341 |      0.0 | 0.470797 |      0.0 | 189.886499 | 228.305106 |  68.649311 |      0.0 |      0.0 | 0.034734 |      0.0 | 0.297341 |      0.0 |      0.0 | 0.036856 |      0.0 |  0.43394 | 0.130777 |
    | 03.01. |  0.0 |  3.473684 | 3.787713 |   0.0 |  4.273684 | 1.515085 | 0.757543 |       0.0 |       0.0 |      0.0 | 0.532795 | 2.190414 | 2.190414 |      0.0 |      0.0 | 0.034325 |      0.0 | 0.293247 |      0.0 | 0.469303 |      0.0 | 187.696085 | 228.305106 |  67.788945 |      0.0 |      0.0 | 0.034325 |      0.0 | 0.293247 |      0.0 |      0.0 | 0.036743 |      0.0 | 0.432561 | 0.130362 |
    | 03.01. |  0.0 |  3.684211 | 2.368437 |   0.0 |  4.484211 | 0.947375 | 0.473687 |       0.0 |       0.0 |      0.0 | 0.330459 | 2.298316 | 2.298316 |      0.0 |      0.0 | 0.033894 |      0.0 | 0.288945 |      0.0 | 0.467767 |      0.0 | 185.397768 | 228.305106 |  67.135646 |      0.0 |      0.0 | 0.033894 |      0.0 | 0.288945 |      0.0 |      0.0 | 0.036614 |      0.0 | 0.431153 | 0.129935 |
    | 03.01. |  0.0 |  3.894737 | 1.120002 |   0.0 |  4.694737 | 0.448001 |    0.224 |       0.0 |       0.0 |      0.0 | 0.155289 | 2.406218 | 2.406218 |      0.0 |      0.0 | 0.033568 |      0.0 | 0.285678 |      0.0 | 0.466195 |      0.0 |  182.99155 | 228.305106 |  66.661111 |      0.0 |      0.0 | 0.033568 |      0.0 | 0.285678 |      0.0 |      0.0 | 0.036473 |      0.0 | 0.429722 | 0.129499 |
    | 03.01. |  0.0 |  4.105263 | 0.211415 |   0.0 |  4.905263 | 0.084566 | 0.042283 |       0.0 |       0.0 |      0.0 | 0.029177 | 2.514121 | 2.514121 |      0.0 |      0.0 | 0.033331 |      0.0 | 0.283306 |      0.0 | 0.464603 |      0.0 | 180.477429 | 228.305106 |  66.315298 |      0.0 |      0.0 | 0.033331 |      0.0 | 0.283306 |      0.0 |      0.0 | 0.036326 |      0.0 | 0.428277 | 0.129056 |
    | 03.01. |  0.0 |  4.315789 | 0.070242 |   0.0 |  5.115789 | 0.028097 | 0.014048 |       0.0 |       0.0 |      0.0 | 0.009661 | 2.622023 | 2.622023 |      0.0 |      0.0 | 0.033158 |      0.0 | 0.281576 |      0.0 | 0.463001 |      0.0 | 177.855406 | 228.305106 |  65.990902 |      0.0 |      0.0 | 0.033158 |      0.0 | 0.281576 |      0.0 |      0.0 | 0.036175 |      0.0 | 0.426826 | 0.128611 |
    | 03.01. |  0.0 |  4.526316 |  0.07067 |   0.0 |  5.326316 | 0.028268 | 0.014134 |       0.0 |       0.0 |      0.0 | 0.009688 | 2.729925 | 2.729925 |      0.0 |      0.0 | 0.032995 |      0.0 | 0.279955 |      0.0 | 0.461397 |      0.0 | 175.125481 | 228.305106 |  65.668264 |      0.0 |      0.0 | 0.032995 |      0.0 | 0.279955 |      0.0 |      0.0 | 0.036024 |      0.0 | 0.425372 | 0.128166 |
    | 03.01. |  0.0 |  4.736842 |   0.0711 |   0.0 |  5.536842 |  0.02844 |  0.01422 |       0.0 |       0.0 |      0.0 | 0.009716 | 2.837827 | 2.837827 |      0.0 |      0.0 | 0.032834 |      0.0 | 0.278341 |      0.0 |  0.45979 |      0.0 | 172.287654 | 228.305106 |  65.347373 |      0.0 |      0.0 | 0.032834 |      0.0 | 0.278341 |      0.0 |      0.0 | 0.035873 |      0.0 | 0.423917 | 0.127719 |
    | 03.01. |  0.0 |  4.947368 | 0.071525 |   0.0 |  5.747368 |  0.02861 | 0.014305 |       0.0 |       0.0 |      0.0 | 0.009742 | 2.945729 | 2.945729 |      0.0 |      0.0 | 0.032674 |      0.0 | 0.276737 |      0.0 | 0.458181 |      0.0 | 169.341925 | 228.305106 |   65.02822 |      0.0 |      0.0 | 0.032674 |      0.0 | 0.276737 |      0.0 |      0.0 |  0.03572 |      0.0 | 0.422461 | 0.127273 |
    | 04.01. |  0.0 |  5.157895 |  0.07195 |   0.0 |  5.957895 |  0.02878 |  0.01439 |       0.0 |       0.0 |      0.0 | 0.009768 | 3.053632 | 3.053632 |      0.0 |      0.0 | 0.032514 |      0.0 | 0.275141 |      0.0 | 0.456571 |      0.0 | 166.288293 | 228.305106 |  64.710797 |      0.0 |      0.0 | 0.032514 |      0.0 | 0.275141 |      0.0 |      0.0 | 0.035568 |      0.0 | 0.421003 | 0.126825 |
    | 04.01. |  0.0 |  5.368421 | 0.072375 |   0.0 |  6.168421 |  0.02895 | 0.014475 |       0.0 |       0.0 |      0.0 | 0.009794 | 3.161534 | 3.161534 |      0.0 |      0.0 | 0.032355 |      0.0 | 0.273554 |      0.0 | 0.454959 |      0.0 | 163.126759 | 228.305106 |  64.395094 |      0.0 |      0.0 | 0.032355 |      0.0 | 0.273554 |      0.0 |      0.0 | 0.035415 |      0.0 | 0.419544 | 0.126377 |
    | 04.01. |  0.0 |  5.578947 | 0.072795 |   0.0 |  6.378947 | 0.029118 | 0.014559 |       0.0 |       0.0 |      0.0 | 0.009818 | 3.269436 | 3.269436 | 4.504854 | 0.490564 | 0.032198 |      0.0 | 0.271975 | 0.490564 | 0.499288 |      0.0 | 159.857323 | 223.800253 |  68.095393 | 0.490564 |      0.0 | 0.032198 |      0.0 | 0.271975 | 0.045943 |      0.0 | 0.035262 |      0.0 | 0.418083 | 0.138691 |
    | 04.01. |  1.3 |  5.789474 | 0.073217 |  1.56 |  6.589474 | 0.029287 | 0.014643 |      0.76 |       0.0 | 0.014643 |      0.0 | 3.377338 | 3.377338 | 5.488273 | 0.641231 | 0.034048 |      0.0 | 0.290477 | 0.641231 | 0.592529 | 0.785357 | 156.479985 |  219.07198 |   72.61791 | 0.641231 |      0.0 | 0.034048 |      0.0 | 0.290477 |  0.14065 |      0.0 | 0.035158 |      0.0 | 0.416722 | 0.164592 |
    | 04.01. |  0.0 |       6.0 | 0.073637 |   0.0 |       6.8 | 0.029455 | 0.014727 |       0.0 |       0.0 | 0.014727 |      0.0 |  3.48524 |  3.48524 | 4.879336 | 0.610802 | 0.036309 |      0.0 |  0.31309 | 0.610802 | 0.679278 | 0.770629 | 152.994745 | 214.192643 |  76.537046 | 0.610802 |      0.0 | 0.036309 |      0.0 |  0.31309 |  0.22854 |      0.0 |  0.03516 |      0.0 | 0.415578 | 0.188688 |
    | 04.01. |  0.0 |  6.210526 | 0.074055 |   0.0 |  7.010526 | 0.029622 | 0.014811 |       0.0 |       0.0 | 0.014811 |      0.0 | 3.593142 | 3.593142 | 5.030399 | 0.669984 | 0.038269 |      0.0 | 0.332685 | 0.669984 | 0.753295 | 0.755818 | 149.401603 | 209.162244 |  80.526508 | 0.669984 |      0.0 | 0.038269 |      0.0 | 0.332685 | 0.303375 |      0.0 | 0.035264 |      0.0 | 0.414656 | 0.209249 |
    | 04.01. |  0.0 |  6.421053 |   0.2705 |   0.0 |  7.221053 |   0.1082 |   0.0541 |       0.0 |       0.0 |   0.0541 |      0.0 | 3.701045 | 3.701045 | 5.181462 | 0.733305 | 0.040263 |      0.0 | 0.352633 | 0.733305 | 0.825159 | 0.701718 | 145.700558 | 203.980781 |   84.58177 | 0.733305 |      0.0 | 0.040263 |      0.0 | 0.352633 |  0.37576 |      0.0 | 0.035459 |      0.0 |  0.41394 | 0.229211 |
    | 04.01. |  0.7 |  6.631579 |  1.23095 |  0.84 |  7.431579 |  0.49238 |  0.24619 |  0.741718 |       0.0 |  0.24619 |      0.0 | 3.808947 | 3.808947 | 6.074244 | 0.916515 | 0.042291 |      0.0 | 0.372909 | 0.916515 | 0.906905 |  0.55381 | 141.891611 | 198.648256 |  89.324299 | 0.916515 |      0.0 | 0.042291 |      0.0 | 0.372909 |  0.45773 |      0.0 | 0.035743 |      0.0 | 0.413431 | 0.251918 |
    | 04.01. |  0.4 |  6.842105 | 2.866595 |  0.48 |  7.642105 | 1.146638 | 0.573319 |   0.23381 |       0.0 | 0.573319 |      0.0 | 3.916849 | 3.916849 | 5.717399 | 0.919577 | 0.044662 |      0.0 | 0.396621 | 0.919577 | 0.990448 | 0.226681 | 137.974762 | 193.164667 |  93.680837 | 0.919577 |      0.0 | 0.044662 |      0.0 | 0.396621 | 0.541181 |      0.0 | 0.036121 |      0.0 | 0.413146 | 0.275124 |
    | 04.01. |  0.1 |  7.052632 |  4.66193 |  0.12 |  7.852632 | 1.864772 | 0.932386 |       0.0 |       0.0 | 0.346681 | 0.488981 | 4.024751 | 4.024751 | 5.634652 | 0.960633 |  0.04684 |      0.0 | 0.418404 | 0.960633 | 1.063298 |      0.0 | 133.950011 | 187.530016 |  97.400629 | 0.960633 |      0.0 |  0.04684 |      0.0 | 0.418404 | 0.613617 |      0.0 | 0.036591 |      0.0 |  0.41309 | 0.295361 |
    | 04.01. |  0.4 |  7.263158 | 6.271463 |  0.48 |  8.063158 | 2.508585 | 1.254293 |       0.0 |       0.0 |     0.48 | 0.657717 | 4.132653 | 4.132653 | 5.785715 | 1.036605 |   0.0487 |      0.0 | 0.437003 | 1.036605 | 1.134162 |      0.0 | 129.817358 | 181.744301 | 101.006318 | 1.035313 | 0.001293 |   0.0487 |      0.0 | 0.437003 | 0.683515 | 0.000275 | 0.037137 |      0.0 | 0.413236 | 0.315045 |
    | 04.01. |  0.0 |  7.473684 | 6.833792 |   0.0 |  8.273684 | 2.733517 | 1.366758 |       0.0 |       0.0 |      0.0 | 1.178878 | 4.240555 | 4.240555 | 5.936778 | 1.114955 | 0.050503 | 0.000909 | 0.455032 | 1.114955 |  1.20791 |      0.0 | 125.576803 | 175.807524 |  104.14282 | 1.103103 | 0.011852 | 0.050503 | 0.000909 | 0.455032 | 0.753634 | 0.002925 | 0.037745 | 0.000044 | 0.413562 | 0.335531 |
    | 04.01. |  0.0 |  7.684211 | 6.738053 |   0.0 |  8.484211 | 2.695221 | 1.347611 |       0.0 |       0.0 |      0.0 | 1.176649 | 4.348458 | 4.348458 | 6.087841 | 1.190331 | 0.052071 | 0.007589 | 0.470714 | 1.190331 | 1.285605 |      0.0 | 121.228345 | 169.719683 | 107.333305 | 1.159898 | 0.030433 | 0.052071 | 0.007589 | 0.470714 | 0.822301 | 0.010397 | 0.038406 | 0.000449 | 0.414053 | 0.357113 |
    | 04.01. |  0.0 |  7.894737 | 8.078045 |   0.0 |  8.694737 | 3.231218 | 1.615609 |       0.0 |       0.0 |      0.0 | 1.426926 |  4.45636 |  4.45636 | 6.238904 | 1.270041 | 0.053667 | 0.017873 | 0.486667 | 1.270041 | 1.367898 |      0.0 | 116.771985 | 163.480779 | 110.317036 | 1.212624 | 0.057417 | 0.053667 | 0.017873 | 0.486667 | 0.888435 |  0.02403 | 0.039112 | 0.001626 | 0.414696 | 0.379972 |
    | 04.01. |  0.0 |  8.105263 | 7.222285 |   0.0 |  8.905263 | 2.888914 | 1.444457 |       0.0 |       0.0 |      0.0 | 1.288488 | 4.564262 | 4.564262 | 6.389967 |  1.35009 | 0.055159 | 0.029825 | 0.501585 |  1.35009 | 1.454943 |      0.0 | 112.207723 | 157.090812 | 113.481855 | 1.259309 | 0.090782 | 0.055159 | 0.029825 | 0.501585 | 0.951572 | 0.044275 | 0.039858 | 0.003751 | 0.415486 | 0.404151 |
    | 04.01. |  0.0 |  8.315789 | 6.044787 |   0.0 |  9.115789 | 2.417915 | 1.208957 |       0.0 |       0.0 |      0.0 | 1.088977 | 4.672164 | 4.672164 |  6.54103 | 1.436822 | 0.056741 | 0.044552 | 0.517409 | 1.436822 | 1.547079 |      0.0 | 107.535559 | 150.549783 | 116.878384 |  1.30402 | 0.132803 | 0.056741 | 0.044552 | 0.517409 | 1.011543 | 0.071527 | 0.040643 | 0.006944 | 0.416422 | 0.429744 |
    | 04.01. |  0.0 |  8.526316 | 4.837587 |   0.0 |  9.326316 | 1.935035 | 0.967517 |       0.0 |       0.0 |      0.0 | 0.879931 | 4.780066 | 4.780066 | 6.692093 | 1.531798 | 0.058439 | 0.062408 | 0.534392 | 1.531798 | 1.645648 |      0.0 | 102.755493 |  143.85769 | 120.503509 | 1.347172 | 0.184625 | 0.058439 | 0.062408 | 0.534392 | 1.068601 | 0.106679 |  0.04147 | 0.011387 | 0.417511 | 0.457125 |
    | 04.01. |  0.0 |  8.736842 | 2.353282 |   0.0 |  9.536842 | 0.941313 | 0.470656 |       0.0 |       0.0 |      0.0 | 0.432096 | 4.887969 | 4.887969 | 6.843156 | 1.635847 | 0.060252 | 0.083557 | 0.552518 | 1.635847 | 1.752029 |      0.0 |  97.867524 | 137.014534 | 124.582395 | 1.388696 | 0.247151 | 0.060252 | 0.083557 | 0.552518 | 1.122987 |  0.15067 | 0.042342 | 0.017265 | 0.418765 | 0.486675 |
    | 04.01. |  0.0 |  8.947368 | 1.092032 |   0.0 |  9.747368 | 0.436813 | 0.218406 |       0.0 |       0.0 |      0.0 | 0.202449 | 4.995871 | 4.995871 | 6.994219 |  1.75458 | 0.062291 | 0.109693 | 0.572912 |  1.75458 | 1.868443 |      0.0 |  92.871653 | 130.020315 | 128.874689 | 1.430063 | 0.324517 | 0.062291 | 0.109693 | 0.572912 | 1.175026 | 0.205116 | 0.043266 | 0.024838 | 0.420197 | 0.519012 |
    | 04.01. |  0.0 |  9.157895 | 0.289745 |   0.0 |  9.957895 | 0.115898 | 0.057949 |       0.0 |       0.0 |      0.0 | 0.054207 | 5.103773 | 5.103773 | 7.145282 | 1.884901 | 0.064437 | 0.139643 | 0.594373 | 1.884901 | 1.996846 |      0.0 |  87.767881 | 122.875033 |  133.28241 | 1.469468 | 0.415433 | 0.064437 | 0.139643 | 0.594373 | 1.224947 | 0.271467 | 0.044247 | 0.034362 | 0.421824 | 0.554679 |
    | 04.01. |  0.0 |  9.368421 | 0.080167 |   0.0 | 10.168421 | 0.032067 | 0.016033 |       0.0 |       0.0 |      0.0 | 0.015124 | 5.211675 | 5.211675 | 7.296345 | 2.026056 | 0.066641 | 0.172808 | 0.616412 | 2.026056 | 2.137966 |      0.0 |  82.556206 | 115.578688 | 137.681713 |  1.50643 | 0.519626 | 0.066641 | 0.172808 | 0.616412 | 1.272732 | 0.350313 | 0.045285 | 0.045985 |  0.42365 | 0.593879 |
    | 04.01. |  0.0 |  9.578947 | 0.080565 |   0.0 | 10.378947 | 0.032226 | 0.016113 |       0.0 |       0.0 |      0.0 | 0.015314 | 5.319577 | 5.319577 | 7.447408 | 2.176305 | 0.068841 |  0.20818 | 0.638409 | 2.176305 | 2.291795 |      0.0 |  77.236628 |  108.13128 | 142.022072 | 1.540506 |   0.6358 | 0.068841 |  0.20818 | 0.638409 | 1.318286 | 0.441685 | 0.046381 | 0.059765 | 0.425678 |  0.63661 |
    | 04.01. |  0.0 |  9.789474 | 0.080962 |   0.0 | 10.589474 | 0.032385 | 0.016192 |       0.0 |       0.0 |      0.0 | 0.015492 | 5.427479 | 5.427479 | 7.598471 | 2.335218 | 0.071011 | 0.245165 |  0.66011 | 2.335218 | 2.457865 |      0.0 |  71.809149 | 100.532809 | 146.293547 | 1.571775 | 0.763444 | 0.071011 | 0.245165 |  0.66011 | 1.361496 | 0.545259 |  0.04753 | 0.075678 | 0.427903 |  0.68274 |
    | 04.01. |  0.0 |      10.0 | 0.081357 |   0.0 |      10.8 | 0.032543 | 0.016271 |       0.0 |       0.0 |      0.0 | 0.015659 | 5.535382 | 5.535382 | 7.749534 | 2.503295 | 0.073147 | 0.283481 | 0.681468 | 2.503295 | 2.635806 |      0.0 |  66.273767 |  92.783274 | 150.486032 | 1.600527 | 0.902769 | 0.073147 | 0.283481 | 0.681468 | 1.402306 | 0.660793 | 0.048727 |  0.09366 |  0.43032 | 0.732168 |

    .. raw:: html

        <iframe
            src="lland_v2_ex6.html"
            width="100%"
            height="830px"
            frameborder=0
        ></iframe>
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core import parametertools
from hydpy.core import sequencetools
# ...from lland
from hydpy.models.lland import lland_model
from hydpy.models.lland import lland_control
from hydpy.models.lland import lland_derived
from hydpy.models.lland import lland_inputs
from hydpy.models.lland import lland_fluxes
from hydpy.models.lland import lland_states
from hydpy.models.lland import lland_logs
from hydpy.models.lland import lland_aides
from hydpy.models.lland import lland_outlets
from hydpy.models.lland import lland_masks
from hydpy.models.lland.lland_constants import *


class Model(modeltools.Model):
    """External ET0 version of HydPy-L-Land (|lland_v2|)."""
    _RUN_METHODS = (lland_model.calc_nkor_v1,
                    lland_model.calc_tkor_v1,
                    lland_model.calc_et0_wet0_v1,
                    lland_model.calc_evpo_v1,
                    lland_model.calc_nbes_inzp_v1,
                    lland_model.calc_evi_inzp_v1,
                    lland_model.calc_sbes_v1,
                    lland_model.calc_wgtf_v1,
                    lland_model.calc_schm_wats_v1,
                    lland_model.calc_wada_waes_v1,
                    lland_model.calc_evb_v1,
                    lland_model.calc_qbb_v1,
                    lland_model.calc_qib1_v1,
                    lland_model.calc_qib2_v1,
                    lland_model.calc_qdb_v1,
                    lland_model.calc_bowa_v1,
                    lland_model.calc_qbgz_v1,
                    lland_model.calc_qigz1_v1,
                    lland_model.calc_qigz2_v1,
                    lland_model.calc_qdgz_v1,
                    lland_model.calc_qdgz1_qdgz2_v1,
                    lland_model.calc_qbga_v1,
                    lland_model.calc_qiga1_v1,
                    lland_model.calc_qiga2_v1,
                    lland_model.calc_qdga1_v1,
                    lland_model.calc_qdga2_v1,
                    lland_model.calc_q_v1)
    _OUTLET_METHODS = (lland_model.pass_q_v1,)


class ControlParameters(parametertools.SubParameters):
    """Control parameters of |lland_v2|, directly defined by the user."""
    CLASSES = (lland_control.FT,
               lland_control.NHRU,
               lland_control.Lnk,
               lland_control.FHRU,
               lland_control.KG,
               lland_control.KT,
               lland_control.KE,
               lland_control.WfET0,
               lland_control.FLn,
               lland_control.HInz,
               lland_control.LAI,
               lland_control.TRefT,
               lland_control.TRefN,
               lland_control.TGr,
               lland_control.TSp,
               lland_control.GTF,
               lland_control.RSchmelz,
               lland_control.CPWasser,
               lland_control.PWMax,
               lland_control.GrasRef_R,
               lland_control.NFk,
               lland_control.RelWZ,
               lland_control.RelWB,
               lland_control.Beta,
               lland_control.FBeta,
               lland_control.DMax,
               lland_control.DMin,
               lland_control.BSf,
               lland_control.A1,
               lland_control.A2,
               lland_control.TInd,
               lland_control.EQB,
               lland_control.EQI1,
               lland_control.EQI2,
               lland_control.EQD1,
               lland_control.EQD2,
               lland_control.NegQ)


class DerivedParameters(parametertools.SubParameters):
    """Derived parameters of |lland_v2|, indirectly defined by the user."""
    CLASSES = (lland_derived.MOY,
               lland_derived.AbsFHRU,
               lland_derived.KInz,
               lland_derived.WB,
               lland_derived.WZ,
               lland_derived.KB,
               lland_derived.KI1,
               lland_derived.KI2,
               lland_derived.KD1,
               lland_derived.KD2,
               lland_derived.QFactor)


class InputSequences(sequencetools.InputSequences):
    """Input sequences of |lland_v2|."""
    CLASSES = (lland_inputs.Nied,
               lland_inputs.TemL,
               lland_inputs.PET)


class FluxSequences(sequencetools.FluxSequences):
    """Flux sequences of |lland_v2|."""
    CLASSES = (lland_fluxes.NKor,
               lland_fluxes.TKor,
               lland_fluxes.ET0,
               lland_fluxes.EvPo,
               lland_fluxes.NBes,
               lland_fluxes.SBes,
               lland_fluxes.EvI,
               lland_fluxes.EvB,
               lland_fluxes.WGTF,
               lland_fluxes.Schm,
               lland_fluxes.WaDa,
               lland_fluxes.QDB,
               lland_fluxes.QIB1,
               lland_fluxes.QIB2,
               lland_fluxes.QBB,
               lland_fluxes.QDGZ,
               lland_fluxes.Q)


class StateSequences(sequencetools.StateSequences):
    """State sequences of |lland_v2|."""
    CLASSES = (lland_states.Inzp,
               lland_states.WATS,
               lland_states.WAeS,
               lland_states.BoWa,
               lland_states.QDGZ1,
               lland_states.QDGZ2,
               lland_states.QIGZ1,
               lland_states.QIGZ2,
               lland_states.QBGZ,
               lland_states.QDGA1,
               lland_states.QDGA2,
               lland_states.QIGA1,
               lland_states.QIGA2,
               lland_states.QBGA)


class LogSequences(sequencetools.LogSequences):
    """Log sequences of |lland_v2|."""
    CLASSES = (lland_logs.WET0,)


class AideSequences(sequencetools.AideSequences):
    """Aide sequences of |lland_v2|."""
    CLASSES = (lland_aides.SfA,
               lland_aides.Exz,
               lland_aides.BVl,
               lland_aides.MVl,
               lland_aides.RVl,
               lland_aides.EPW)


class OutletSequences(sequencetools.LinkSequences):
    """Downstream link sequences of |lland_v2|."""
    CLASSES = (lland_outlets.Q,)


class Masks(masktools.Masks):
    """Masks applicable to |lland_v2|."""
    CLASSES = lland_masks.Masks.CLASSES


autodoc_applicationmodel()

# pylint: disable=invalid-name
tester = Tester()
cythonizer = Cythonizer()
cythonizer.complete()
