"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? false;
        this.user = props.user ?? 25000;
        this.group = props.group ?? 26000;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-23.ContainerSecurityContext", version: "2.0.0-rc.103" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-23.EnvValue", version: "2.0.0-rc.103" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup;
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this._imagePullPolicyToKube(this.imagePullPolicy),
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
    _imagePullPolicyToKube(imagePullPolicy) {
        switch (imagePullPolicy) {
            case ImagePullPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.ALWAYS;
            case ImagePullPolicy.IF_NOT_PRESENT:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.IF_NOT_PRESENT;
            case ImagePullPolicy.NEVER:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.NEVER;
            default:
                throw new Error(`Unsupported image pull policy: ${imagePullPolicy}`);
        }
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-23.Container", version: "2.0.0-rc.103" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-23.Cpu", version: "2.0.0-rc.103" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-23.EnvFrom", version: "2.0.0-rc.103" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    const ports = [];
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers;
    for (const con of containers ?? []) {
        if (con.port) {
            ports.push(con.port);
        }
    }
    return ports;
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-23.Env", version: "2.0.0-rc.103" };
//# sourceMappingURL=data:application/json;base64,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