"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy ? this._restartPolicyToKube(this.restartPolicy) : undefined,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
    _restartPolicyToKube(restartPolicy) {
        switch (restartPolicy) {
            case RestartPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1PodSpecRestartPolicy.ALWAYS;
            case RestartPolicy.NEVER:
                return k8s.IoK8SApiCoreV1PodSpecRestartPolicy.NEVER;
            case RestartPolicy.ON_FAILURE:
                return k8s.IoK8SApiCoreV1PodSpecRestartPolicy.ON_FAILURE;
            default:
                throw new Error(`Unsupported restart policy: ${restartPolicy}`);
        }
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-23.AbstractPod", version: "2.0.0-rc.103" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-23.LabelSelector", version: "2.0.0-rc.103" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-23.Pod", version: "2.0.0-rc.103" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this._dnsPolicyToKube(this.policy),
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
    _dnsPolicyToKube(dnsPolicy) {
        switch (dnsPolicy) {
            case DnsPolicy.CLUSTER_FIRST:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.CLUSTER_FIRST;
            case DnsPolicy.CLUSTER_FIRST_WITH_HOST_NET:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.CLUSTER_FIRST_WITH_HOST_NET;
            case DnsPolicy.DEFAULT:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.DEFAULT;
            case DnsPolicy.NONE:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.NONE;
            default:
                throw new Error(`Unsupported dns policy: ${dnsPolicy}`);
        }
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-23.PodDns", version: "2.0.0-rc.103" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-23.PodSecurityContext", version: "2.0.0-rc.103" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.IN, values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.NOT_IN, values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.EXISTS, undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.DOES_NOT_EXIST, undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.GT, values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.LT, values);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            key: this.key,
            operator: this.operator,
            values: this.values,
        };
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-23.NodeLabelQuery", version: "2.0.0-rc.103" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-23.LabelExpression", version: "2.0.0-rc.103" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery(k8s.IoK8SApiCoreV1TolerationOperator.EQUAL, key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery(k8s.IoK8SApiCoreV1TolerationOperator.EXISTS, key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery(k8s.IoK8SApiCoreV1TolerationOperator.EXISTS);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            effect: this.effect ? this._taintEffectToKube(this.effect) : undefined,
            key: this.key,
            operator: this.operator,
            tolerationSeconds: this.evictAfter?.toSeconds(),
            value: this.value,
        };
    }
    _taintEffectToKube(taintEffect) {
        switch (taintEffect) {
            case TaintEffect.NO_EXECUTE:
                return k8s.IoK8SApiCoreV1TolerationEffect.NO_EXECUTE;
            case TaintEffect.NO_SCHEDULE:
                return k8s.IoK8SApiCoreV1TolerationEffect.NO_SCHEDULE;
            case TaintEffect.PREFER_NO_SCHEDULE:
                return k8s.IoK8SApiCoreV1TolerationEffect.PREFER_NO_SCHEDULE;
            default:
                throw new Error(`Unsupported taint effect: ${taintEffect}`);
        }
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-23.NodeTaintQuery", version: "2.0.0-rc.103" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-23.Pods", version: "2.0.0-rc.103" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-23.LabeledNode", version: "2.0.0-rc.103" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-23.TaintedNode", version: "2.0.0-rc.103" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-23.NamedNode", version: "2.0.0-rc.103" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-23.Node", version: "2.0.0-rc.103" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-23.Topology", version: "2.0.0-rc.103" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push(query._toKube());
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => s._toKube()) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-23.PodScheduling", version: "2.0.0-rc.103" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n));
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-23.PodConnections", version: "2.0.0-rc.103" };
//# sourceMappingURL=data:application/json;base64,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