# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_aicorebridge.ipynb.

# %% auto 0
__all__ = ['syslog', 'annotated_arg_builders', 'build_historic_args', 'AICoreModule']

# %% ../nbs/01_aicorebridge.ipynb 4
import typing
import logging
import traceback
import inspect
import datetime
import json
import os
import pandas, pandas as pd
import numpy, numpy as np

from dateutil import parser
from fastcore.basics import patch_to, patch
from .core import *
from . import __version__


# %% ../nbs/01_aicorebridge.ipynb 5
syslog = logging.getLogger(__name__)

# %% ../nbs/01_aicorebridge.ipynb 6
try:
    print(f"Loading {__name__} {__version__} from {__file__}")
except:
    pass

# %% ../nbs/01_aicorebridge.ipynb 8
def build_historic_args(data:pd.DataFrame, history:dict|list) -> dict:
    """Create a timeseries DataFrame from historic data defined in `history`.

    Parameters
    ----------
    data : pd.DataFrame
        The input time-series DataFrame.
    history : dict or list of dicts
        Historic data definition, each item in the list is a dictionary with a startDate key to set the start of a section of historic data in the result and a column-value pair for each of the columns in the 

    Returns
    -------
    historic_data : dict
        Historic data in dictionary format where keys are column names and values are the historic values as numpy array.
    """

    if not history:
        return {}
    
    if isinstance(history, dict):
        return history
    
    if not isinstance(history, list):
        return {}
    
    if isinstance(data, pd.DataFrame):
        dates = data.index.astype(np.int64).astype(np.float64) / 1e9
        dates = dates.to_numpy()
    elif data.dtype.names is not None:
        dates = data.view(dtype=np.float64).reshape(data.shape[0],len(data.dtype))[:,0]
    else:
        dates = data[:,0]
    dates = dates.astype(np.int64)
    
    columns = list(set([K for I in history for K in I.keys() if K != 'startDate']))
    column_data = {K:np.full(len(dates), np.nan, dtype=np.float64) for K in columns}

    for I in history:
        date = parser.parse(str((I.pop('startDate','2000-01-01T00:00:00+00:00')))).timestamp()
        mask = np.greater_equal(dates, date)
        for K,V in I.items():
            column_data[K][mask] = V
    
    return column_data
    #return pd.DataFrame(column_data, index=data.index)


# %% ../nbs/01_aicorebridge.ipynb 12
class AICoreModule(): pass

# %% ../nbs/01_aicorebridge.ipynb 13
@patch
def __init__(self:AICoreModule, 
             processor:typing.Callable, # data processing function
             save_dir:str, # path where the module can keep files 
             assets_dir:str,
             *args, **kwargs):
    
    self.init_time = datetime.datetime.now(datetime.UTC)
    self._init_processor(processor)

    self.init_args = args
    self.init_kwargs = dict(
        **kwargs,
        assets_dir=assets_dir,
        save_dir=save_dir
    )



# %% ../nbs/01_aicorebridge.ipynb 14
@patch
def _init_processor(
        self:AICoreModule, 
        processor:typing.Callable):
    """Initializes processor related variables on self"""
    
    self.processor = processor
    self.processor_signature = inspect.signature(self.processor)
    self.processor_params = dict(self.processor_signature.parameters)
    self.return_param = self.processor_params.pop('return', None)
    self.data_param, *self.call_params = list(self.processor_params.keys())


# %% ../nbs/01_aicorebridge.ipynb 15
# can be overloaded
@patch
def call_processor(self:AICoreModule, calldata, **callargs):
    return self.processor(calldata, **callargs)


# %% ../nbs/01_aicorebridge.ipynb 17
@patch
def infer(self:AICoreModule, data:dict, *_, **kwargs):
    try:

        msg=[
            f"Startup time: {self.init_time.isoformat()}",
            f"{self.processor.__name__}({self.processor_signature})",             
            f"init_args: {self.init_args}, init_kwargs: {self.init_kwargs}",
        ]

        # Pickup params, pop those that are not intended for the processor
        lastSeen = kwargs.pop('lastSeen', False)
        recordformat = kwargs.pop('format', "records").lower()
        timezone = kwargs.get('timezone', 'UTC')
        msg.append(f"lastSeen: {lastSeen}, recordformat: {recordformat}, timezone: {timezone}")

        samplerPeriod = kwargs.pop('samplerPeriod', self.init_kwargs.get('samplerPeriod','h'))
        samplerMethod = kwargs.pop('samplerMethod', self.init_kwargs.get('samplerMethod',None))
        reversed = kwargs.pop('reversed', False)

        calldata = self.get_call_data(
            data, 
            recordformat=recordformat,
            timezone=timezone)
        
        msg.append(f"calldata shape: {calldata.shape}")

        history = build_historic_args(calldata, kwargs.pop('history', {}))

        callargs = self.get_callargs(kwargs, history)

        for arg, val in callargs.items():
            msg.append(f"{arg}: {val}")
            
        result = timeseries_dataframe(
            self.call_processor(
                calldata, 
                **callargs), 
            timezone=timezone)
        
        msg.append(f"result shape: {result.shape}")

        if samplerMethod:
            msg.append(f"Sampler: {samplerMethod}, period: {samplerPeriod}")
            result = timeseries_dataframe_resample(result, samplerPeriod, samplerMethod)

        msg.append(f"return-data shape: {result.shape}")

        if reversed:
            result = result[::-1]

        return {
            'msg':msg,
            'data': timeseries_dataframe_to_datadict(
                result if not lastSeen else result[-1:],
                recordformat=recordformat,
                timezone=timezone,
                popNaN=True)
        }
    except Exception as err:
        msg.append(''.join(traceback.format_exception(None, err, err.__traceback__)))
        syslog.exception(f"Exception {str(err)} in infer()")
        return {
            'msg': f"Unexpected {err=}, {type(err)=}",
            'data': []
        }


# %% ../nbs/01_aicorebridge.ipynb 19
# Specialized types for initializing annotated parameters
# Add types by adding a tuple with the type name and a builder function
annotated_arg_builders = {
    str(B[0]):B[1] for B in [
        (numpy.ndarray, lambda X: numpy.array(X, dtype=X.dtype))
    ]
}

# %% ../nbs/01_aicorebridge.ipynb 20
@patch
def init_annotated_param(self:AICoreModule, param_name, value):
    """
    Initialize argument for the processor call
    
    param_name: name of the parameter to be initialized
    value: value of the parameter read from infer data to be used for initialization
    
    """

    annotation = self.processor_signature.parameters[param_name].annotation

    # try to convert value to one of the types in the annotation
    for T in typing.get_args(annotation):
        try:
            builder = annotated_arg_builders.get(str(T), T)
            return builder(value)
        except TypeError as err:
            continue
    try:
        return annotation(value)
    except TypeError as err:
        syslog.exception(f"Exception {str(err)} in fallback conversion to {annotation} of {type(value)}")

 

# %% ../nbs/01_aicorebridge.ipynb 21
@patch
def get_callargs(self:AICoreModule, kwargs, history):
    "Get arguments for the processor call"

    # Remove null / None values
    kwargs = {k:v for k,v in kwargs.items() if v is not None}
    
    return {
        K:self.init_annotated_param(
            K,
            history.get(
                K,
                self.init_kwargs.get(
                    K,
                    kwargs.get(
                        K, 
                        self.processor_signature.parameters[K].default
                    )
                )
            )
        )
        for K in self.call_params
    }


# %% ../nbs/01_aicorebridge.ipynb 25
@patch
def get_call_data(
        self:AICoreModule, 
        data:dict, 
        recordformat='records', 
        timezone='UTC'):
    
    "Convert data to the processor signature"

    df = set_time_index_zone(timeseries_dataframe_from_datadict(
        data, ['datetimeMeasure', 'time'], recordformat), timezone)

    df.sort_index(inplace=True)

    if self.processor_params[self.data_param].annotation == pd.DataFrame:
        return df
    elif len(df.columns) > 1:
        df.index = (df.index - datetime.datetime(1970,1,1, tzinfo=datetime.timezone.utc)) / datetime.timedelta(seconds=1)
        return df.to_records(index=True)
    else:
        df.index = (df.index - datetime.datetime(1970,1,1, tzinfo=datetime.timezone.utc)) / datetime.timedelta(seconds=1)
        return df.reset_index().to_numpy()
        
