"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MysqlDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const triggers_1 = require("aws-cdk-lib/triggers");
const base_database_1 = require("./base-database");
const HANDLER_PATH = path.join(__dirname, "../../assets/rds/triggers/mysql");
// https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.tls
const AMAZON_ROOT_CA1 = "https://www.amazontrust.com/repository/AmazonRootCA1.pem";
/**
 * A MySQL database.
 */
class MysqlDatabase extends base_database_1.BaseDatabase {
    /**
     * Creates a new MysqlDatabase.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param props - The configuration properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.ownerSecrets = [];
        this.readerSecrets = [];
        this.unprivilegedSecrets = [];
        const { adminSecret, vpc, characterSet = "utf8mb4" } = props;
        const environment = {
            AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
            ADMIN_SECRET_ARN: adminSecret.secretArn,
            OWNER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.ownerSecrets.map((s) => s.secretArn)),
            }),
            READER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.readerSecrets.map((s) => s.secretArn)),
            }),
            UNPRIVILEGED_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.unprivilegedSecrets.map((s) => s.secretArn)),
            }),
            DB_NAME: this.databaseName,
            DB_CHARACTER_SET: characterSet,
        };
        if (props.collation) {
            environment.DB_COLLATION = props.collation;
        }
        if (props.certificateAuthoritiesUrl) {
            environment.CA_CERTS_URL = props.certificateAuthoritiesUrl;
        }
        this.lambdaFunction = new aws_lambda_1.Function(this, "Function", {
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            handler: "index.handler",
            description: "Creates a schema and possibly some secondary users",
            vpc,
            vpcSubnets: this.subnetSelection,
            securityGroups: [this.securityGroup],
            environment,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        adminSecret.grantRead(this.lambdaFunction);
        this.trigger = new triggers_1.Trigger(this, "Trigger", {
            handler: this.lambdaFunction,
        });
        this.trigger.executeAfter(this.lambdaFunction);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forCluster(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            vpc: cluster.vpc,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forClusterFromSnapshot(scope, id, cluster, options) {
        // The DatabaseClusterFromSnapshot type is equivalent to DatabaseCluster.
        return MysqlDatabase.forCluster(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessCluster(scope, id, cluster, options) {
        // The ServerlessClusterFromSnapshot type is a subset of ServerlessCluster.
        return MysqlDatabase.forServerlessClusterFromSnapshot(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessClusterFromSnapshot(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            certificateAuthoritiesUrl: AMAZON_ROOT_CA1,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstance.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstance(scope, id, instance, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || instance.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: instance,
            endpoint: instance.instanceEndpoint,
            adminSecret: clusterSecret,
            vpc: instance.vpc,
            ...props,
        });
        database.trigger.executeAfter(instance);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstanceFromSnapshot.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstanceFromSnapshot(scope, id, instance, options) {
        // The DatabaseInstanceFromSnapshot type is equivalent to DatabaseInstance.
        return MysqlDatabase.forInstance(scope, id, instance, options);
    }
    addUserAsOwner(secret) {
        this.ownerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsReader(secret) {
        this.readerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsUnprivileged(secret) {
        this.unprivilegedSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
}
exports.MysqlDatabase = MysqlDatabase;
_a = JSII_RTTI_SYMBOL_1;
MysqlDatabase[_a] = { fqn: "shady-island.MysqlDatabase", version: "0.1.12" };
//# sourceMappingURL=data:application/json;base64,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