"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// eslint-disable-next-line import/no-extraneous-dependencies
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
// eslint-disable-next-line import/no-extraneous-dependencies
const promise_1 = require("mysql2/promise");
const util_1 = require("./util");
const adminSecretArn = process.env.ADMIN_SECRET_ARN;
const databaseName = process.env.DB_NAME;
const characterSet = process.env.DB_CHARACTER_SET || "utf8mb4";
const collation = process.env.DB_COLLATION;
const secretsManagerClient = new client_secrets_manager_1.SecretsManagerClient({});
const ownerSecretArns = util_1.parseJsonArrayFromEnv("OWNER_SECRETS");
const readerSecretArns = util_1.parseJsonArrayFromEnv("READER_SECRETS");
const unprivilegedSecretArns = util_1.parseJsonArrayFromEnv("UNPRIVILEGED_SECRETS");
async function createUser(connection, user, password) {
    const escapedPassword = connection.escape(password);
    let createUserSql = `CREATE USER IF NOT EXISTS ${user} IDENTIFIED BY ${escapedPassword}`;
    console.log(`About to create user: ${user}`);
    await connection.execute(createUserSql, [password]);
    console.log(`User created: ${user}`);
}
async function applyGrant(connection, user, privileges) {
    let grantPrivSql = `GRANT ${privileges} ON ${databaseName}.* TO ${user}`;
    console.log(`About to grant privileges to user: ${user}`);
    await connection.execute(grantPrivSql);
    console.log(`Privileges granted: ${privileges} ON ${databaseName}.*`);
}
const handler = async () => {
    const adminSecret = await util_1.fetchSecret(adminSecretArn, secretsManagerClient);
    const region = process.env.AWS_REGION || process.env.AWS_DEFAULT_REGION;
    const caBundle = await util_1.readRemote(process.env.CA_CERTS_URL ||
        `https://truststore.pki.rds.amazonaws.com/${region}/${region}-bundle.pem`);
    const connection = await promise_1.createConnection({
        host: adminSecret.host,
        user: adminSecret.username,
        password: adminSecret.password,
        port: adminSecret.port,
        ssl: { ca: caBundle },
        connectTimeout: 40000,
    });
    let createDbSql = `CREATE DATABASE IF NOT EXISTS ${databaseName} CHARACTER SET ${characterSet}`;
    if (collation) {
        createDbSql += ` COLLATE ${collation}`;
    }
    console.log(`About to create database: ${databaseName}`);
    await connection.execute(createDbSql);
    console.log(`Database created: ${databaseName}`);
    const ownerSecrets = await util_1.fetchAllSecrets(ownerSecretArns, secretsManagerClient);
    for (const userSecret of ownerSecrets) {
        const mysqlUser = `'${userSecret.username}'@'%'`;
        await createUser(connection, mysqlUser, userSecret.password);
        await applyGrant(connection, mysqlUser, "ALL PRIVILEGES");
    }
    const readerSecrets = await util_1.fetchAllSecrets(readerSecretArns, secretsManagerClient);
    for (const userSecret of readerSecrets) {
        const mysqlUser = `'${userSecret.username}'@'%'`;
        await createUser(connection, mysqlUser, userSecret.password);
        await applyGrant(connection, mysqlUser, "SELECT, SHOW VIEW");
    }
    const unprivilegedSecrets = await util_1.fetchAllSecrets(unprivilegedSecretArns, secretsManagerClient);
    for (const userSecret of unprivilegedSecrets) {
        const mysqlUser = `'${userSecret.username}'@'%'`;
        await createUser(connection, mysqlUser, userSecret.password);
    }
    let flushSql = "FLUSH PRIVILEGES";
    console.log("About to flush privileges");
    await connection.execute(flushSql);
    console.log("Privileges flushed");
    console.log("About to close the connection");
    await connection.end();
    console.log("Connection closed");
};
module.exports = { handler };
//# sourceMappingURL=data:application/json;base64,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