"""
Persistent Step-wise Backtest Implementation

This module provides a way to run backtesting.py strategies in a persistent manner,
adding one bar at a time instead of re-running the entire backtest each cycle.

This approach significantly reduces latency by avoiding the O(n) cost of reprocessing
historical data on every signal extraction.
"""

import logging
import numpy as np
import pandas as pd
from typing import Any, Dict, Optional

# Conditional import for backtesting library
try:
    from backtesting import Backtest, Strategy
    BACKTESTING_AVAILABLE = True
except ImportError as e:
    BACKTESTING_AVAILABLE = False
    Backtest = None
    Strategy = None

from ..core.signal_extractor import TradingSignal, SignalType

logger = logging.getLogger(__name__)


class PersistentBacktest:
    """
    A persistent backtesting engine that can accept new bars incrementally
    without re-running the entire backtest from scratch.
    
    This works by maintaining the internal state of a Backtest instance
    and manually stepping through new bars using private API access.
    """
    
    def __init__(self, initial_data: pd.DataFrame, strategy_class: type, max_bars: Optional[int] = None, **bt_kwargs):
        """
        Initialize the persistent backtest with initial historical data.
        
        Args:
            initial_data: DataFrame with OHLCV data for warm-up
            strategy_class: Strategy class (already converted to signal extractor)
            max_bars: Maximum number of bars to keep in memory (None for unlimited)
            **bt_kwargs: Additional arguments for Backtest constructor
        """
        if not BACKTESTING_AVAILABLE:
            raise ImportError("backtesting.py not available")
            
        self.strategy_class = strategy_class
        self.bt_kwargs = bt_kwargs
        self.max_bars = max_bars
        self.is_initialized = False
        
        # Internal state that will be maintained
        self.backtest = None
        self.strategy = None
        self.broker = None
        self.data = None
        self.current_index = 0
        
        # Initialize with the provided historical data
        self._initialize(initial_data)
    
    def _initialize(self, initial_data: pd.DataFrame):
        """Initialize the backtest with historical data and run to current state."""
        try:
            logger.debug(f"Initializing persistent backtest with {len(initial_data)} bars")
            
            # Create the backtest instance
            self.backtest = Backtest(
                initial_data,
                self.strategy_class,
                **self.bt_kwargs
            )
            
            # Run the backtest to process all initial data
            results = self.backtest.run()
            
            # Extract and store the internal components we need to maintain
            self.strategy = results._strategy
            self.broker = self.backtest._broker
            self.data = self.backtest._data
            
            # Track where we are in the data
            self.current_index = len(self.data) - 1
            
            self.is_initialized = True
            logger.debug(f"Persistent backtest initialized successfully, current index: {self.current_index}")
            
        except Exception as e:
            logger.error(f"Failed to initialize persistent backtest: {e}")
            raise
    
    def step(self, new_bar: pd.Series) -> TradingSignal:
        """
        Process a single new bar and return the trading signal.
        
        Args:
            new_bar: Series with OHLCV data for the new bar
            
        Returns:
            TradingSignal generated by the strategy for this bar
        """
        if not self.is_initialized:
            raise RuntimeError("PersistentBacktest not initialized")
        
        try:
            # Extend the data arrays with the new bar
            self._extend_data_arrays(new_bar)
            
            # Apply memory limit if specified
            if self.max_bars is not None and len(self.data) > self.max_bars:
                trim_count = len(self.data) - self.max_bars
                self._trim_arrays_front(trim_count)
            
            # Update the current data pointer
            self.current_index += 1
            self.data._i = self.current_index
            
            # Process the new bar through strategy and broker
            self._process_bar()
            
            # Extract and return the signal
            return self.strategy.get_current_signal()
            
        except Exception as e:
            logger.error(f"Error processing bar in persistent backtest: {e}")
            # Return safe fallback signal
            return TradingSignal(
                signal=SignalType.HOLD,
                price=float(new_bar.get('Close', 0)),
                timestamp=new_bar.name if hasattr(new_bar, 'name') else pd.Timestamp.now(),
                indicators={'error': str(e)},
                metadata={'persistent_error': True}
            )
    
    def _extend_data_arrays(self, new_bar: pd.Series):
        """
        Extend the internal data arrays with a new bar.
        
        This manipulates the numpy arrays that backtesting.py uses internally
        to store OHLCV data.
        """
        try:
            # Extract OHLCV values from the new bar
            timestamp = new_bar.name if hasattr(new_bar, 'name') else pd.Timestamp.now()
            
            # Get the values, handling different column name formats
            open_val = new_bar.get('Open', new_bar.get('open', new_bar.get('Close', new_bar.get('close', 0))))
            high_val = new_bar.get('High', new_bar.get('high', open_val))
            low_val = new_bar.get('Low', new_bar.get('low', open_val))
            close_val = new_bar.get('Close', new_bar.get('close', open_val))
            volume_val = new_bar.get('Volume', new_bar.get('volume', 0))
            
            # Extend each data array
            for col_name, value in [
                ('Open', open_val),
                ('High', high_val), 
                ('Low', low_val),
                ('Close', close_val),
                ('Volume', volume_val)
            ]:
                if hasattr(self.data, col_name):
                    # Get the underlying numpy array
                    array_attr = getattr(self.data, col_name)
                    if hasattr(array_attr, 'values'):
                        arr = array_attr.values
                    else:
                        arr = array_attr
                    
                    # Resize the array and add the new value
                    new_size = arr.shape[0] + 1
                    arr.resize(new_size, refcheck=False)
                    arr[-1] = float(value)
            
            # Extend the datetime index if it exists
            if hasattr(self.data, 'index') and hasattr(self.data.index, '_values'):
                # Convert timestamp to the format expected by backtesting.py
                if hasattr(timestamp, 'to_pydatetime'):
                    dt_val = timestamp.to_pydatetime()
                else:
                    dt_val = timestamp
                
                # Extend the index array
                index_arr = self.data.index._values
                new_index_size = index_arr.shape[0] + 1
                index_arr.resize(new_index_size, refcheck=False)
                
                # Convert datetime to the numeric format backtesting.py uses
                import matplotlib.dates as mdates
                index_arr[-1] = mdates.date2num(dt_val)
            
            logger.debug(f"Extended data arrays to length {len(self.data)}")
            
        except Exception as e:
            logger.error(f"Error extending data arrays: {e}")
            raise
    
    def _trim_arrays_front(self, n: int):
        """
        Remove the first n rows from every OHLCV numpy array and the index.
        
        This helps prevent unbounded memory growth in persistent mode.
        
        Args:
            n: Number of rows to remove from the front
        """
        try:
            logger.debug(f"Trimming {n} rows from front of data arrays")
            
            # Trim each OHLCV column
            for col_name in ['Open', 'High', 'Low', 'Close', 'Volume']:
                if hasattr(self.data, col_name):
                    # Get the underlying numpy array
                    array_attr = getattr(self.data, col_name)
                    if hasattr(array_attr, 'values'):
                        arr = array_attr.values
                    else:
                        arr = array_attr
                    
                    # Shift data left and resize
                    if len(arr) > n:
                        arr[:-n] = arr[n:]  # Shift left
                        arr.resize(len(arr) - n, refcheck=False)
            
            # Trim the datetime index if it exists
            if hasattr(self.data, 'index') and hasattr(self.data.index, '_values'):
                index_arr = self.data.index._values
                if len(index_arr) > n:
                    index_arr[:-n] = index_arr[n:]  # Shift left
                    index_arr.resize(len(index_arr) - n, refcheck=False)
            
            # Adjust current index
            self.current_index = max(0, self.current_index - n)
            
            logger.debug(f"Trimmed arrays, new length: {len(self.data)}, new current_index: {self.current_index}")
            
        except Exception as e:
            logger.error(f"Error trimming data arrays: {e}")
            # Don't raise - memory trimming failure shouldn't stop signal extraction
            pass
    
    def _process_bar(self):
        """
        Process the current bar through the strategy and broker.
        
        This mimics what Backtest._run() does for a single iteration.
        """
        try:
            # Call the strategy's next() method for the new bar
            if hasattr(self.strategy, 'next'):
                self.strategy.next()
            
            # Process any orders that were generated
            if hasattr(self.broker, '_next'):
                self.broker._next()
            elif hasattr(self.broker, 'next'):
                # Fallback for different backtesting.py versions
                self.broker.next()
            
            logger.debug(f"Processed bar {self.current_index}")
            
        except Exception as e:
            logger.error(f"Error processing bar through strategy/broker: {e}")
            raise
    
    def get_stats(self) -> Dict[str, Any]:
        """Get current statistics from the persistent backtest."""
        if not self.is_initialized:
            return {'error': 'not_initialized'}
        
        try:
            stats = {
                'initialized': True,
                'current_index': self.current_index,
                'total_bars': len(self.data),
                'max_bars': self.max_bars,
                'strategy_class': self.strategy_class.__name__,
            }
            
            # Add broker info if available
            if self.broker:
                if hasattr(self.broker, '_cash'):
                    stats['cash'] = float(self.broker._cash)
                if hasattr(self.broker, '_value'):
                    stats['portfolio_value'] = float(self.broker._value)
            
            return stats
            
        except Exception as e:
            return {'error': str(e)}
    
    def stop(self):
        """Clean up the persistent backtest."""
        logger.debug("Stopping persistent backtest")
        self.is_initialized = False
        self.backtest = None
        self.strategy = None
        self.broker = None
        self.data = None
        self.current_index = 0


class PersistentBacktestError(Exception):
    """Exception raised when persistent backtest operations fail."""
    pass 