"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MysqlDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const triggers_1 = require("aws-cdk-lib/triggers");
const base_database_1 = require("./base-database");
const HANDLER_PATH = path.join(__dirname, "../../assets/rds/triggers/mysql");
/**
 * A MySQL database.
 */
class MysqlDatabase extends base_database_1.BaseDatabase {
    /**
     * Creates a new MysqlDatabase.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param props - The configuration properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.ownerSecrets = [];
        this.readerSecrets = [];
        const { adminSecret, resource, vpc, characterSet = "utf8mb4" } = props;
        const environment = {
            AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
            ADMIN_SECRET_ARN: adminSecret.secretArn,
            OWNER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.ownerSecrets.map((s) => s.secretArn)),
            }),
            READER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.readerSecrets.map((s) => s.secretArn)),
            }),
            DB_NAME: this.databaseName,
            DB_CHARACTER_SET: characterSet,
        };
        if (props.collation) {
            environment.DB_COLLATION = props.collation;
        }
        const lambdaFunction = new aws_lambda_1.Function(this, "Function", {
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            handler: "index.handler",
            description: "Creates a schema and possibly some secondary users",
            vpc,
            vpcSubnets: this.subnetSelection,
            securityGroups: [this.securityGroup],
            environment,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        this.lambdaFunction = lambdaFunction;
        this.trigger = new triggers_1.Trigger(this, "Trigger", { handler: lambdaFunction });
        adminSecret.grantRead(lambdaFunction);
        this.trigger.executeAfter(resource, lambdaFunction);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forCluster(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        return new MysqlDatabase(scope, id, {
            resource: cluster,
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            vpc: cluster.vpc,
            ...props,
        });
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forClusterFromSnapshot(scope, id, cluster, options) {
        // The DatabaseClusterFromSnapshot type is equivalent to DatabaseCluster.
        return MysqlDatabase.forCluster(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessCluster(scope, id, cluster, options) {
        // The ServerlessClusterFromSnapshot type is a subset of ServerlessCluster.
        return MysqlDatabase.forServerlessClusterFromSnapshot(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessClusterFromSnapshot(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        return new MysqlDatabase(scope, id, {
            resource: cluster,
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            ...props,
        });
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstance.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstance(scope, id, instance, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || instance.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        return new MysqlDatabase(scope, id, {
            resource: instance,
            target: instance,
            endpoint: instance.instanceEndpoint,
            adminSecret: clusterSecret,
            vpc: instance.vpc,
            ...props,
        });
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstanceFromSnapshot.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstanceFromSnapshot(scope, id, instance, options) {
        // The DatabaseInstanceFromSnapshot type is equivalent to DatabaseInstance.
        return MysqlDatabase.forInstance(scope, id, instance, options);
    }
    addUserAsOwner(secret) {
        this.ownerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsReader(secret) {
        this.readerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
}
exports.MysqlDatabase = MysqlDatabase;
_a = JSII_RTTI_SYMBOL_1;
MysqlDatabase[_a] = { fqn: "shady-island.MysqlDatabase", version: "0.0.46" };
//# sourceMappingURL=data:application/json;base64,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