"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const fs = require("fs-extra");
const path = require("path");
const lib_1 = require("../lib");
const build_1 = require("../lib/build");
jest.mock('../lib/build', () => ({
    build: jest.fn((options) => {
        require('fs-extra').ensureDirSync(options.outDir); // eslint-disable-line @typescript-eslint/no-require-imports
    })
}));
let stack;
const buildDir = path.join(__dirname, '.build');
beforeEach(() => {
    stack = new core_1.Stack();
    fs.removeSync(buildDir);
});
afterEach(() => {
    fs.removeSync(buildDir);
});
test('NodejsFunction with .ts handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    expect(build_1.build).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler1.ts'),
        global: 'handler',
        outDir: expect.stringContaining(buildDir)
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('NodejsFunction with .js handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler2');
    // THEN
    expect(build_1.build).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler2.js'),
    }));
});
test('throws when entry is not js/ts', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'handler.py'
    })).toThrow(/Only JavaScript or TypeScript entry files are supported/);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'notfound.ts'
    })).toThrow(/Cannot find entry file at/);
});
test('throws when entry cannot be automatically found', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn')).toThrow(/Cannot find entry file./);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'handler1', {
        runtime: aws_lambda_1.Runtime.PYTHON_3_8
    })).toThrow(/Only `NODEJS` runtimes are supported/);
});
//# sourceMappingURL=data:application/json;base64,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