from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.http_validation_error import HTTPValidationError
from ...models.test_data_point_item import TestDataPointItem
from ...types import Response


def _get_kwargs(
    test_data_point_id: str,
    *,
    api_key: str,
) -> Dict[str, Any]:
    headers = {}
    headers["api-key"] = api_key

    return {
        "method": "get",
        "url": "/v0/test_data_points/{test_data_point_id}".format(
            test_data_point_id=test_data_point_id,
        ),
        "headers": headers,
    }


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[HTTPValidationError, TestDataPointItem]]:
    if response.status_code == HTTPStatus.CREATED:
        response_201 = TestDataPointItem.from_dict(response.json())

        return response_201
    if response.status_code == HTTPStatus.UNPROCESSABLE_ENTITY:
        response_422 = HTTPValidationError.from_dict(response.json())

        return response_422
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[HTTPValidationError, TestDataPointItem]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    test_data_point_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    api_key: str,
) -> Response[Union[HTTPValidationError, TestDataPointItem]]:
    """Get Test Data Point

     Get a Test Run

    Returns:
        the Test Run

    Args:
        test_data_point_id (str):
        api_key (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[HTTPValidationError, TestDataPointItem]]
    """

    kwargs = _get_kwargs(
        test_data_point_id=test_data_point_id,
        api_key=api_key,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    test_data_point_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    api_key: str,
) -> Optional[Union[HTTPValidationError, TestDataPointItem]]:
    """Get Test Data Point

     Get a Test Run

    Returns:
        the Test Run

    Args:
        test_data_point_id (str):
        api_key (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[HTTPValidationError, TestDataPointItem]
    """

    return sync_detailed(
        test_data_point_id=test_data_point_id,
        client=client,
        api_key=api_key,
    ).parsed


async def asyncio_detailed(
    test_data_point_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    api_key: str,
) -> Response[Union[HTTPValidationError, TestDataPointItem]]:
    """Get Test Data Point

     Get a Test Run

    Returns:
        the Test Run

    Args:
        test_data_point_id (str):
        api_key (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[HTTPValidationError, TestDataPointItem]]
    """

    kwargs = _get_kwargs(
        test_data_point_id=test_data_point_id,
        api_key=api_key,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    test_data_point_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    api_key: str,
) -> Optional[Union[HTTPValidationError, TestDataPointItem]]:
    """Get Test Data Point

     Get a Test Run

    Returns:
        the Test Run

    Args:
        test_data_point_id (str):
        api_key (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[HTTPValidationError, TestDataPointItem]
    """

    return (
        await asyncio_detailed(
            test_data_point_id=test_data_point_id,
            client=client,
            api_key=api_key,
        )
    ).parsed
