"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
//
// Utility to parse JSON safely
Object.defineProperty(exports, "__esModule", { value: true });
exports.safeJsonParse = exports.isJsonObject = void 0;
/**
 * Check if a piece of JSON is a JSON object, and not e.g. a mere string or null
 *
 * @param j - the JSON
 */
function isJsonObject(j) {
    // It is not enough to check that `typeof j === "object"`
    // because in JS `typeof null` is also "object", and so is `typeof []`.
    // So we need to check that j is an object, and not null, and not an array
    return typeof j === 'object' && !Array.isArray(j) && j !== null;
}
exports.isJsonObject = isJsonObject;
/**
 * Parse a string as JSON, while removing __proto__ and constructor, so JS prototype pollution is prevented
 *
 * @param s - the string to JSON parse
 */
function safeJsonParse(s) {
    return JSON.parse(s, (_, value) => {
        if (typeof value === 'object' && !Array.isArray(value) && value !== null) {
            delete value.__proto__;
            delete value.constructor;
        }
        return value;
    });
}
exports.safeJsonParse = safeJsonParse;
//# sourceMappingURL=data:application/json;base64,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