import unittest
import unittest.mock
from unittest.mock import patch, MagicMock
import json
import sys

# Mock missing dependencies
sys.modules['schedule'] = MagicMock()
sys.modules['sagemaker_sparkmonitor.aws_service_utils'] = MagicMock()

from sagemaker_sparkmonitor.sparkUtils import SparkInfoReporter


class TestSparkUtils(unittest.TestCase):
    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_spark_util(self):
        sparkInfoReporter = SparkInfoReporter(None, None)

        assert sparkInfoReporter.is_spark_job_running == False

    @patch('sagemaker_sparkmonitor.sparkUtils.get_sigv4_signed_header_for_emr_serverless')
    @patch('sagemaker_sparkmonitor.sparkUtils.requests.get')
    def test_get_emr_serverless_spark_ui_url_caching(self, mock_get, mock_sigv4):
        """Test EMR Serverless Spark UI URL caching functionality"""
        # Setup mock response
        mock_response = MagicMock()
        mock_response.content.decode.return_value = json.dumps({
            'appInfo': {
                'sparkUiUrl': 'https://test-spark-ui.amazonaws.com'
            }
        })
        mock_get.return_value = mock_response
        mock_sigv4.return_value = {'Authorization': 'test-auth'}
        
        # Setup SparkInfoReporter with connection details
        reporter = SparkInfoReporter(None, None)
        reporter.connection_details = {
            'livyEndpoint': 'https://test-livy.amazonaws.com',
            'session_id': 'test-session-123',
            'region': 'us-west-2'
        }
        reporter.connection_id = 'test-connection-id'
        
        # First call should make API request
        result1 = reporter.get_emr_serverless_spark_ui_url()
        self.assertEqual(result1, 'https://test-spark-ui.amazonaws.com')
        self.assertEqual(mock_get.call_count, 1)
        
        # Second call should use cache (no additional API call)
        result2 = reporter.get_emr_serverless_spark_ui_url()
        self.assertEqual(result2, 'https://test-spark-ui.amazonaws.com')
        self.assertEqual(mock_get.call_count, 1)  # Still 1, not 2
        
        # Verify correct API endpoint was called
        expected_url = 'https://test-livy.amazonaws.com/sessions/test-session-123'
        mock_get.assert_called_with(expected_url, headers={'Authorization': 'test-auth'})

    @patch('sagemaker_sparkmonitor.sparkUtils.get_sigv4_signed_header_for_emr_serverless')
    @patch('sagemaker_sparkmonitor.sparkUtils.requests.get')
    def test_get_emr_serverless_spark_ui_url_cache_clear(self, mock_get, mock_sigv4):
        """Test cache clearing functionality"""
        # Setup mock response
        mock_response = MagicMock()
        mock_response.content.decode.return_value = json.dumps({
            'appInfo': {
                'sparkUiUrl': 'https://test-spark-ui.amazonaws.com'
            }
        })
        mock_get.return_value = mock_response
        mock_sigv4.return_value = {'Authorization': 'test-auth'}
        
        # Setup SparkInfoReporter
        reporter = SparkInfoReporter(None, None)
        reporter.connection_details = {
            'livyEndpoint': 'https://test-livy.amazonaws.com',
            'session_id': 'test-session-123',
            'region': 'us-west-2'
        }
        reporter.connection_id = 'test-connection-id'
        
        # First call
        result1 = reporter.get_emr_serverless_spark_ui_url()
        self.assertEqual(mock_get.call_count, 1)
        
        # Clear cache
        reporter.get_emr_serverless_spark_ui_url.clear_cache()
        
        # Next call should make API request again
        result2 = reporter.get_emr_serverless_spark_ui_url()
        self.assertEqual(mock_get.call_count, 2)

    @patch('sagemaker_sparkmonitor.sparkUtils.get_sigv4_signed_header_for_emr_serverless')
    @patch('sagemaker_sparkmonitor.sparkUtils.requests.get')
    def test_get_emr_serverless_spark_ui_url_missing_appinfo(self, mock_get, mock_sigv4):
        """Test handling of missing appInfo in response"""
        # Setup mock response without appInfo
        mock_response = MagicMock()
        mock_response.content.decode.return_value = json.dumps({})
        mock_get.return_value = mock_response
        mock_sigv4.return_value = {'Authorization': 'test-auth'}
        
        # Setup SparkInfoReporter
        reporter = SparkInfoReporter(None, None)
        reporter.connection_details = {
            'livyEndpoint': 'https://test-livy.amazonaws.com',
            'session_id': 'test-session-123',
            'region': 'us-west-2'
        }
        reporter.connection_id = 'test-connection-id'
        
        # Should return None when appInfo is missing
        result = reporter.get_emr_serverless_spark_ui_url()
        self.assertIsNone(result)

    @patch('sagemaker_sparkmonitor.sparkUtils.get_sigv4_signed_header_for_emr_serverless')
    @patch('sagemaker_sparkmonitor.sparkUtils.requests.get')
    def test_get_emr_serverless_spark_ui_url_beta_endpoint(self, mock_get, mock_sigv4):
        """Test beta endpoint replacement"""
        # Setup mock response
        mock_response = MagicMock()
        mock_response.content.decode.return_value = json.dumps({
            'appInfo': {
                'sparkUiUrl': 'https://test-spark-ui.amazonaws.com'
            }
        })
        mock_get.return_value = mock_response
        mock_sigv4.return_value = {'Authorization': 'test-auth'}
        
        # Setup SparkInfoReporter with beta endpoint
        reporter = SparkInfoReporter(None, None)
        reporter.connection_details = {
            'livyEndpoint': 'https://test-livy-beta.amazonaws.com',
            'session_id': 'test-session-123',
            'region': 'us-west-2'
        }
        reporter.connection_id = 'test-connection-id'
        
        # Call method
        result = reporter.get_emr_serverless_spark_ui_url()
        
        # Verify beta was removed from endpoint
        expected_url = 'https://test-livy.amazonaws.com/sessions/test-session-123'
        mock_get.assert_called_with(expected_url, headers={'Authorization': 'test-auth'})

    # def glue_test():
    #     client = get_internal_glue_client()
    #     ret = client.get_dashboard_url(ResourceId='axb53q9y4xc80g-bd8c4383-8de6-44f3-a7df-c446803c845a', ResourceType='SESSION')
    #     session = requests.session()
    #     url = ret.get('Url', None)
    #     print(url)
    #     with requests.Session() as session:
    #         session.get(url, verify=False)
    #         for cookie in session.cookies:
    #             if cookie.name == 'VerifiedAuthToken':
    #                 expires = cookie.expires
    #                 print (expires)
    #         print (session.cookies.get('VerifiedAuthToken', None))

    # def serverless_test():
    #     application_id='00fm0aluav9roe0l'
    #     region = 'us-west-2'
    #     endpoint = f'https://{application_id}.livy.emr-serverless-services.{region}.amazonaws.com'
    #     emr_serverless_sigv4_header = get_sigv4_signed_header_for_emr_serverless(endpoint, region)
    #     canonical_uri = "/sessions"
    #     url=f'{endpoint}{canonical_uri}'
    #     response = requests.get(url, headers=emr_serverless_sigv4_header)
    #     # print(response.content.decode("utf-8"))
    #     res_json = json.loads(response.content.decode("utf-8"))
    #     # print(str(res_json))
    #     session_id = '1'
    #     spark_ui_url = None
    #     for livy_session in res_json['sessions']:
    #         print(str(livy_session['id']))
    #         if str(livy_session['id']) == session_id:
    #             spark_ui_url = livy_session['appInfo']['sparkUiUrl']
    #             # print(spark_ui_url)
    #             with requests.Session() as session:
    #                 session.get(spark_ui_url, verify=False)
    #                 emr_serverless_cookie = session.cookies.get('VerifiedAuthToken', None)
    #                 print(emr_serverless_cookie)
