"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dashboard = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_elasticache_1 = require("@aws-cdk/aws-elasticache");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
const auto_scaling_1 = require("./auto-scaling");
const load_balancer_1 = require("./load-balancer");
const rds_1 = require("./rds");
const redis_1 = require("./redis");
/**
 * An AWS CloudWatch Dashboard.
 *
 * @stability stable
 * @example
 *
 * // create a dashboard for AutoScaling
 * new Dashboard(this, 'dashboard', {
 *    autoScalingName: 'my-auto-scaling',
 *    autoScalingMaxCapacity: 10
 * });
 */
class Dashboard extends core_1.Construct {
    /**
     * Creates a Dashboard based on the Halloumi best practices.
     *
     * @param scope the scope into which to import this dashboard.
     * @param id the logical ID of the returned dashboard construct.
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let start = (props === null || props === void 0 ? void 0 : props.start) ? props.start : '-PT24H';
        let periodOverride = (props === null || props === void 0 ? void 0 : props.periodOverride) ? props.periodOverride : aws_cloudwatch_1.PeriodOverride.INHERIT;
        const dashboard = new aws_cloudwatch_1.Dashboard(this, id, {
            dashboardName: props === null || props === void 0 ? void 0 : props.dashboardName,
            start: start,
            end: props === null || props === void 0 ? void 0 : props.end,
            periodOverride: periodOverride,
            widgets: props === null || props === void 0 ? void 0 : props.widgets,
        });
        if (props === null || props === void 0 ? void 0 : props.loadBalancer) {
            for (let i = 0; i < props.loadBalancer.length; i++) {
                let loadBalancer = props.loadBalancer[i];
                let name;
                let full_name;
                if (loadBalancer instanceof aws_elasticloadbalancingv2_1.BaseLoadBalancer) {
                    name = loadBalancer.loadBalancerName;
                    full_name = loadBalancer.loadBalancerFullName;
                }
                else if (loadBalancer instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
                    name = loadBalancer.attrLoadBalancerName;
                    full_name = loadBalancer.attrLoadBalancerFullName;
                }
                else {
                    name = loadBalancer.name;
                    full_name = loadBalancer.full_name;
                }
                const lbWidgets = load_balancer_1.LoadBalancer.metrics(name, full_name);
                lbWidgets.forEach(widget => {
                    dashboard.addWidgets(widget);
                });
            }
        }
        if (props === null || props === void 0 ? void 0 : props.autoScaling) {
            for (let i = 0; i < props.autoScaling.length; i++) {
                let auto_scaling_group = props.autoScaling[i];
                let maxCapacity;
                let name;
                if (auto_scaling_group instanceof aws_autoscaling_1.CfnAutoScalingGroup) {
                    name = auto_scaling_group.ref;
                    maxCapacity = parseInt(auto_scaling_group.maxSize);
                    dashboard.node.addDependency(auto_scaling_group);
                }
                else if (auto_scaling_group instanceof aws_autoscaling_1.AutoScalingGroup) {
                    name = auto_scaling_group.autoScalingGroupName;
                    dashboard.node.addDependency(auto_scaling_group);
                }
                else {
                    name = auto_scaling_group.name;
                    maxCapacity = auto_scaling_group.max_capacity;
                }
                let autoScalingWidgets = auto_scaling_1.AutoScaling.metrics(name, maxCapacity);
                autoScalingWidgets.forEach(widget => {
                    dashboard.addWidgets(widget);
                });
            }
        }
        if (props === null || props === void 0 ? void 0 : props.elasticache) {
            for (let i = 0; i < props.elasticache.length; i++) {
                let elasticache = props.elasticache[i];
                let name;
                let numNodes;
                if (elasticache instanceof aws_elasticache_1.CfnReplicationGroup) {
                    name = elasticache.ref;
                    numNodes = elasticache.numNodeGroups || 1;
                }
                else {
                    name = elasticache.name;
                    numNodes = elasticache.nodes || 1;
                }
                for (let node = 1; node <= numNodes; node++) {
                    let nodeId = name + '-' + node.toString().padStart(3, '0');
                    let elasticacheWidgets = redis_1.Redis.metrics(nodeId);
                    elasticacheWidgets.forEach(widget => {
                        dashboard.addWidgets(widget);
                    });
                }
            }
        }
        if (props === null || props === void 0 ? void 0 : props.rds) {
            for (let i = 0; i < props.rds.length; i++) {
                let rds_cluster = props.rds[i];
                let name = rds_cluster.ref;
                let rdsWidgets = rds_1.Rds.metrics(name);
                rdsWidgets.forEach(widget => {
                    dashboard.addWidgets(widget);
                });
            }
        }
    }
}
exports.Dashboard = Dashboard;
_a = JSII_RTTI_SYMBOL_1;
Dashboard[_a] = { fqn: "halloumi-cloudwatch-dashboard.Dashboard", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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