"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../../lib");
const iam = require("aws-cdk-lib/aws-iam");
const sns = require("aws-cdk-lib/aws-sns");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs';
stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
const dlqEncryptionKey = new kms.Key(stack, 'ImportedSQSEncryptionKey', {
    enableKeyRotation: true,
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
});
const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {
    encryptionMasterKey: dlqEncryptionKey
});
// Definitions
const props = {
    sqsSubscriptionProps: {
        rawMessageDelivery: true,
        // To test this stack, when publishing the SNS message, add a message attribute:
        // Type: string, Name: store, Value: (any value)
        filterPolicy: {
            store: sns.SubscriptionFilter.existsFilter(),
        },
        deadLetterQueue: dlq,
    },
    topicEncryptionKeyProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
    },
    queueEncryptionKeyProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
    }
};
const snsToSqsStack = new lib_1.SnsToSqs(stack, 'test-sns-sqs-stack', props);
// Grant yourself permissions to use the Customer Managed KMS Key
const policyStatement = new iam.PolicyStatement({
    actions: ["kms:Encrypt", "kms:Decrypt"],
    effect: iam.Effect.ALLOW,
    principals: [new iam.AccountRootPrincipal()],
    resources: ["*"]
});
snsToSqsStack.queueEncryptionKey?.addToResourcePolicy(policyStatement);
snsToSqsStack.topicEncryptionKey?.addToResourcePolicy(policyStatement);
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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