"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const assertions_1 = require("aws-cdk-lib/assertions");
const utils_1 = require("./utils");
test('Pattern deployment w/ new Topic, new Queue and default props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {};
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    // Confirm subscription has proper target
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
    });
});
test('Pattern deployment w/ new topic, new queue, and overridden props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testTopicName = "topicehdjs";
    const testQueueName = "queueehdjs.fifo";
    const props = {
        topicProps: {
            topicName: testTopicName,
        },
        queueProps: {
            queueName: testQueueName,
            fifo: true
        },
        queueEncryptionKeyProps: {
            enableKeyRotation: false
        },
        encryptQueueWithCmk: true,
        encryptTopicWithCmk: true,
        deployDeadLetterQueue: false,
        maxReceiveCount: 0
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).not.toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 1);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Topic", {
        TopicName: testTopicName,
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: testQueueName,
        FifoQueue: true,
    });
    // One key has rotation (default), one does not (override)
    // NOTE - THIS IS FOR TESTING, Key Rotation is a best practice
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
test('Test getter methods', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
});
test('Test deployment w/ existing queue, and topic', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testTopicName = 'existing-topic-adfa';
    const testQueueName = 'existing-queue-zfx';
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: testTopicName
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: testQueueName
    });
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingTopicObj: topic,
        existingQueueObj: queue
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    // Since we used simple CDK calls to make topic/queue, there are no CMKs nor DLQ
    template.resourceCountIs("AWS::KMS::Key", 0);
    template.resourceCountIs("AWS::SQS::Queue", 1);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    // Confirm subscription has proper target
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
    });
});
test('Test deployment with imported encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testKeyAlias = 'key-asdg';
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false,
        alias: testKeyAlias
    });
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingQueueEncryptionKey: kmsKey
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false,
    });
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: true,
    });
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Topic", {
        KmsMasterKeyId: assertions_1.Match.anyValue()
    });
});
test('Test deployment with SNS managed KMS key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns')
        },
        queueProps: {
            encryptionMasterKey: new kms.Key(stack, 'test-key', {}),
        },
        enableEncryptionWithCustomerManagedKey: false
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.sse);
});
test('Test deployment with CMK encrypted SNS Topic (avoids interface)', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testDescription = 'someString-llasdj';
    const testTopicName = 'topic-hyuyj';
    const cmk = new kms.Key(stack, 'cmk', { description: testDescription });
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: cmk,
            topicName: testTopicName
        }
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::KMS::Key", {
        Description: testDescription
    });
});
test('Pattern deployment w/ existing topic and FIFO queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const topic = new sns.Topic(stack, 'TestTopic', {
        contentBasedDeduplication: true,
        displayName: 'Customer subscription topic',
        fifo: true,
        topicName: 'customerTopic',
    });
    const props = {
        encryptQueueWithCmk: false,
        existingTopicObj: topic,
        queueProps: {
            fifo: true,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: true,
        }
    };
    const app = () => {
        new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    };
    // Assertion
    expect(app).toThrowError("SQS queue encrypted by AWS managed KMS key cannot be used as SNS subscription");
});
test('Pattern deployment w/ existing topic and Standard queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const topic = new sns.Topic(stack, 'TestTopic', {
        displayName: 'Customer subscription topic',
        fifo: false,
        topicName: 'customerTopic',
    });
    const props = {
        encryptQueueWithCmk: false,
        existingTopicObj: topic,
        queueProps: {
            fifo: false,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: false,
        }
    };
    const app = () => {
        new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    };
    // Assertion
    expect(app).toThrowError("SQS queue encrypted by AWS managed KMS key cannot be used as SNS subscription");
});
test('Check sqsSubscriptionProps are used', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {
        sqsSubscriptionProps: {
            rawMessageDelivery: true,
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    allowlist: ['red', 'orange'],
                    matchPrefixes: ['bl'],
                })
            }
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
        RawMessageDelivery: true,
        FilterPolicy: {
            color: [
                "red",
                "orange",
                {
                    prefix: "bl"
                }
            ]
        }
    });
});
test('Check SNS dead letter queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {});
    const props = {
        sqsSubscriptionProps: {
            deadLetterQueue: dlq
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    // The SNS DLQ is a third queue and should be attached to the Subscription
    template.resourceCountIs("AWS::SQS::Queue", 3);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        RedrivePolicy: assertions_1.Match.anyValue()
    });
});
test('Construct uses topicProps.masterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const cmk = new kms.Key(stack, 'cmkfortopic');
    const noKeyQueue = new sqs.Queue(stack, 'placeHolderQueue', {});
    const props = {
        topicProps: {
            masterKey: cmk
        },
        existingQueueObj: noKeyQueue,
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).not.toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 1);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
});
test('Construct uses queueProps.encryptionMasterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const cmk = new kms.Key(stack, 'cmkforqueue', {});
    const props = {
        queueProps: {
            encryptionMasterKey: cmk
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.encryptionKey).not.toBeDefined();
    expect(testConstruct.queueEncryptionKey).not.toBeDefined();
    expect(testConstruct.topicEncryptionKey).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
});
test('Construct does not override unencrypted topic when passed in existingTopicObj prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const props = {
        existingTopicObj,
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
});
test('Test deployment with existing encryption key for Topic', () => {
    const testDescription = "someValue";
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false,
        description: testDescription
    });
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        encryptTopicWithCmk: true,
        existingTopicEncryptionKey: kmsKey
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false,
        Description: testDescription
    });
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
});
test('Test deployment with key props for Topic', () => {
    const testDescription = "test-description-lkjh";
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        encryptTopicWithCmk: true,
        topicEncryptionKeyProps: {
            enableKeyRotation: false,
            description: testDescription
        }
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false,
        Description: testDescription
    });
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
});
test('Test deployment with no key props or key for Topic', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        encryptTopicWithCmk: true
    });
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.encryptionKey).not.toBeDefined();
    expect(testConstruct.queueEncryptionKey).toBeDefined();
    expect(testConstruct.topicEncryptionKey).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
});
test('Old interface use with feature flag enabled', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {
        // Force a second key to avoid circular reference problem
        enableEncryptionWithCustomerManagedKey: true,
        queueProps: {
            encryptionMasterKey: new kms.Key(stack, 'queueKey', {})
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    utils_1.CheckKeyProperty(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    utils_1.CheckKeyProperty(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    // this is the key created for the topic
    utils_1.CheckKeyProperty(testConstruct.encryptionKey, utils_1.keyType.cmk);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
    });
});
/*******************************
 *
 * Input tests - these tests check that invalid inputs are caught
 * before any processing is attempted. Having both SNS and SQS in the same
 * Construct means the associated properties have unique names in the construct to
 * avoid collisions, so the standard checks will not find these.
 *
 *******************************/
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that Construct checks for mixed deprecated and active props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            enableEncryptionWithCustomerManagedKey: true,
            encryptQueueWithCmk: true
        });
    };
    // Assertion
    expect(app).toThrowError(/Cannot specify both deprecated key props and new key props/);
});
test('Confirm that Construct checks for mixed deprecated and active props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            enableEncryptionWithCustomerManagedKey: true,
            encryptQueueWithCmk: true
        });
    };
    // Assertion
    expect(app).toThrowError(/Cannot specify both deprecated key props and new key props/);
});
test('Confirm that queueProps and existingQueue is caught', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that existingTopic and topicProps is caught', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicObj: topic,
            topicProps: {
                topicName: 'topic-name'
            },
            existingQueueObj: queue
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps or existingTopicObj, but not both.\n/);
});
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Catch queue key and queue key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingQueueEncryptionKey: new kms.Key(stack, 'test', {}),
            queueEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide existingQueueEncryptionKey or queueEncryptionKeyProps, but not both.\n/);
});
test('Catch queueProps key and construct props key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingQueueEncryptionKey: new kms.Key(stack, 'firstKey', {}),
            queueProps: {
                encryptionMasterKey: new kms.Key(stack, 'secondKey', {})
            }
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide queueProps.encryptionMasterKey or existingQueueEncryptionKey, but not both.\n/);
});
test('Catch queueProps key and construct props queue Key Props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                encryptionMasterKey: new kms.Key(stack, 'secondKey', {})
            },
            queueEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide queueProps.encryptionMasterKey or queueEncryptionKeyProps, but not both./);
});
test('Catch topic key and topic key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicEncryptionKey: {},
            topicEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide existingTopicEncryptionKey or topicEncryptionKeyProps, but not both.\n/);
});
test('Catch topicProps key and construct props key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            topicProps: {
                masterKey: new kms.Key(stack, 'firstKey', {})
            },
            existingTopicEncryptionKey: new kms.Key(stack, 'secondKey', {})
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps.masterKey or existingTopicEncryptionKey, but not both.\n/);
});
test('Catch topicProps key and construct props Key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            topicProps: {
                masterKey: new kms.Key(stack, 'firstKey', {})
            },
            topicEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps.masterKey or topicEncryptionKeyProps, but not both.\n/);
});
test('Catch encryptTopicWithCmk false with topic key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptTopicWithCmk: false,
            topicEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptTopicWithCmk is false, submitting topicEncryptionKeyProps is invalid\n/);
});
test('Catch encryptTopicWithCmk false with topic key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptTopicWithCmk: false,
            existingTopicEncryptionKey: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptTopicWithCmk is false, submitting existingTopicEncryptionKey is invalid\n/);
});
test('Catch encryptQueueWithCmk false with queue key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptQueueWithCmk: false,
            queueEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptQueueWithCmk is false, submitting queueEncryptionKeyProps is invalid\n/);
});
test('Catch encryptQueueWithCmk false with queue key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptQueueWithCmk: false,
            existingQueueEncryptionKey: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptQueueWithCmk is false, submitting existingQueueEncryptionKey is invalid\n/);
});
test('Catch queueProps.encryption on new interface', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                encryption: sqs.QueueEncryption.UNENCRYPTED
            }
        });
    };
    // Assertion
    expect(app).toThrowError();
});
/*******************************
 * Tests for CreateRequiredKeys()
 *******************************/
test('test CreateRequiredKeys for no arguments', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {});
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Topic is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingTopicObj: {}
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Queue is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingQueueObj: {}
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Queue encryption is explicitly disabled', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        encryptQueueWithCmk: false
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Topic encryption is explicitly disabled', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        encryptTopicWithCmk: false
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Topic props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        topicProps: {
            masterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Queue props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        queueProps: {
            encryptionMasterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
//# sourceMappingURL=data:application/json;base64,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