"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsToSqs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
const core_1 = require("@aws-solutions-constructs/core");
const core_2 = require("aws-cdk-lib/core");
const cx_api_1 = require("aws-cdk-lib/cx-api");
/**
 * @summary The SnsToSqs class.
 */
class SnsToSqs extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the SnsToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {SnsToSqsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckSnsProps(props);
        defaults.CheckSqsProps(props);
        this.uniquePropChecks(props);
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        const activeKeys = SnsToSqs.createRequiredKeys(scope, id, props);
        if (!activeKeys.useDeprecatedInterface) {
            this.queueEncryptionKey = activeKeys.queueKey;
            this.topicEncryptionKey = activeKeys.topicKey;
        }
        // Setup the SNS topic
        if (!props.existingTopicObj) {
            // If an existingTopicObj was not specified create new topic
            const buildTopicResponse = defaults.buildTopic(this, id, {
                topicProps: props.topicProps,
                encryptionKey: activeKeys.topicKey,
                enableEncryptionWithCustomerManagedKey: activeKeys.encryptTopicWithCmk,
            });
            if (activeKeys.useDeprecatedInterface) {
                this.encryptionKey = buildTopicResponse.key;
            }
            this.snsTopic = buildTopicResponse.topic;
        }
        else {
            // If an existingTopicObj was specified utilize the provided topic
            this.snsTopic = props.existingTopicObj;
        }
        // Setup the queue
        const buildQueueResponse = defaults.buildQueue(this, 'queue', {
            existingQueueObj: props.existingQueueObj,
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue,
            enableEncryptionWithCustomerManagedKey: activeKeys.encryptQueueWithCmk,
            encryptionKey: activeKeys.queueKey,
        });
        this.sqsQueue = buildQueueResponse.queue;
        // Setup the SQS queue subscription to the SNS topic
        this.snsTopic.addSubscription(new subscriptions.SqsSubscription(this.sqsQueue, props.sqsSubscriptionProps));
    }
    /*
    *
    * With the old and new interfaces both supported, it got complex figuring out
    * what props to use for what keys. This complexity has all been extracted to this
    * function to keep the constructor itself more straightforward
    *
    * This function will interpret what info the client provided (throw an error if there are conflicts),
    * determine from the data and it's own internal logic whether to implement the deprecated or new interface,
    * CREATE ANY KEYS NEEDED (not passing a create key flag to the buildTopic and buildQueue functions), then
    * return the keys in a structure.
    *
    * Odd things to know:
    * -If it decides to implement the deprecated interface it will still fill in the topicKey and queueKey values - topicKey
    *      and queueKey will ALWAYS be set, for the old interface they will be set to the same key as singleKey
    * -If the client provides no key info, this function will use the FeatureFlag to determine which interface to use
    */
    static createRequiredKeys(scope, id, props) {
        let topicKey;
        let encryptTopicWithCmk = false;
        let queueKey;
        let encryptQueueWithCmk = false;
        let singleKey;
        // First - confirm that only 1 interface is being used
        let useDeprecatedInterface = false;
        let useCurrentInterface = false;
        if (props.enableEncryptionWithCustomerManagedKey || props.enableEncryptionWithCustomerManagedKey || props.encryptionKeyProps) {
            useDeprecatedInterface = true;
            defaults.printWarning('The enableEncryptionWithCustomerManagedKey, enableEncryptionWithCustomerManagedKey and encryptionKeyProps props values for SnsToSqsProps ' +
                'are deprecated. Consider moving to encryptQueueWithCmk, queueEncryptionKeyProps, existingQueueEncryptionKey, encryptTopicWithCmk, ' +
                'topicEncryptionKeyProps and existingTopicEncryptionKey.');
        }
        if (props.encryptQueueWithCmk ||
            props.queueEncryptionKeyProps ||
            props.existingQueueEncryptionKey ||
            props.encryptTopicWithCmk ||
            props.topicEncryptionKeyProps ||
            props.existingTopicEncryptionKey) {
            useCurrentInterface = true;
        }
        if (useCurrentInterface && useDeprecatedInterface) {
            throw new Error('Cannot specify both deprecated key props and new key props');
        }
        // If neither are set, use the feature flag choose the functionality
        if (!useCurrentInterface && !useDeprecatedInterface) {
            defaults.printWarning('Unable to determine whether to use current or deprecated Key functionality, using Feature Flag SNS_SUBSCRIPTIONS_SQS_DECRYPTION_POLICY to decide');
            if (core_2.FeatureFlags.of(scope).isEnabled(cx_api_1.SNS_SUBSCRIPTIONS_SQS_DECRYPTION_POLICY)) {
                useCurrentInterface = true;
            }
            else {
                useDeprecatedInterface = true;
            }
        }
        // If Deprecated functionality
        // Use code from above to create single key
        if (useDeprecatedInterface) {
            const queueKeyNeeded = DoWeNeedACmk(props.existingQueueObj, props.queueProps?.encryptionMasterKey, props.enableEncryptionWithCustomerManagedKey);
            const topicKeyNeeded = DoWeNeedACmk(props.existingTopicObj, props.topicProps?.masterKey, props.enableEncryptionWithCustomerManagedKey);
            if (queueKeyNeeded || topicKeyNeeded) {
                // We need to encrypt the resources with a single key
                singleKey = props.encryptionKey ?? core_1.buildEncryptionKey(scope, id, props.encryptionKeyProps);
                topicKey = topicKeyNeeded ? singleKey : undefined;
                queueKey = queueKeyNeeded ? singleKey : undefined;
                encryptQueueWithCmk = queueKeyNeeded;
                encryptTopicWithCmk = topicKeyNeeded;
            }
        }
        if (useCurrentInterface) {
            if (props.queueProps?.encryption) {
                throw new Error('The new interface of SnsToSqs does not support managing encryption using the queueProps.encryption setting. ' +
                    'To use a totally unencrypted queue (not recommended), create the queue separately and pass in the existingQueueObj prop');
            }
            if (DoWeNeedACmk(props.existingTopicObj, props.topicProps?.masterKey, props.encryptTopicWithCmk)) {
                topicKey = props.existingTopicEncryptionKey ?? core_1.buildEncryptionKey(scope, `${id}topic`, props.topicEncryptionKeyProps);
                encryptTopicWithCmk = true;
            }
            if (DoWeNeedACmk(props.existingQueueObj, props.queueProps?.encryptionMasterKey, props.encryptQueueWithCmk)) {
                queueKey = props.existingQueueEncryptionKey ?? core_1.buildEncryptionKey(scope, `${id}queue`, props.queueEncryptionKeyProps);
                encryptQueueWithCmk = true;
            }
        }
        return {
            useDeprecatedInterface,
            singleKey,
            topicKey,
            queueKey,
            encryptQueueWithCmk,
            encryptTopicWithCmk
        };
    }
    /*
     * Because the props for the queue and topic are unique on this construct, we need
     * to check for props issues that the standard checks won't catch
     */
    uniquePropChecks(props) {
        let errorMessages = '';
        let errorFound = false;
        if (props.topicProps?.masterKey && props.existingTopicEncryptionKey) {
            errorMessages += 'Error - Either provide topicProps.masterKey or existingTopicEncryptionKey, but not both.\n';
            errorFound = true;
        }
        if (props.topicProps?.masterKey && props.topicEncryptionKeyProps) {
            errorMessages += 'Error - Either provide topicProps.masterKey or topicEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if (props.existingTopicEncryptionKey && props.topicEncryptionKeyProps) {
            errorMessages += 'Error - Either provide existingTopicEncryptionKey or topicEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if (props.queueProps?.encryptionMasterKey && props.queueEncryptionKeyProps) {
            errorMessages += 'Error - Either provide queueProps.encryptionMasterKey or queueEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if (props.queueProps?.encryptionMasterKey && props.existingQueueEncryptionKey) {
            errorMessages += 'Error - Either provide queueProps.encryptionMasterKey or existingQueueEncryptionKey, but not both.\n';
            errorFound = true;
        }
        if (props.existingQueueEncryptionKey && props.queueEncryptionKeyProps) {
            errorMessages += 'Error - Either provide existingQueueEncryptionKey or queueEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if ((props.encryptQueueWithCmk === false) && (props.queueEncryptionKeyProps)) {
            errorMessages += 'Error - if encryptQueueWithCmk is false, submitting queueEncryptionKeyProps is invalid\n';
            errorFound = true;
        }
        if ((props.encryptQueueWithCmk === false) && (props.existingQueueEncryptionKey)) {
            errorMessages += 'Error - if encryptQueueWithCmk is false, submitting existingQueueEncryptionKey is invalid\n';
            errorFound = true;
        }
        if ((props.encryptTopicWithCmk === false) && (props.topicEncryptionKeyProps)) {
            errorMessages += 'Error - if encryptTopicWithCmk is false, submitting topicEncryptionKeyProps is invalid\n';
            errorFound = true;
        }
        if ((props.encryptTopicWithCmk === false) && (props.existingTopicEncryptionKey)) {
            errorMessages += 'Error - if encryptTopicWithCmk is false, submitting existingTopicEncryptionKey is invalid\n';
            errorFound = true;
        }
        if ((props.enableEncryptionWithCustomerManagedKey === false) && (props.encryptionKey || props.encryptionKeyProps)) {
            errorMessages += 'Error - if enableEncryptionWithCustomerManagedKey is false, submitting encryptionKey or encryptionKeyProps is invalid\n';
            errorFound = true;
        }
        if (errorFound) {
            throw new Error(errorMessages);
        }
    }
}
exports.SnsToSqs = SnsToSqs;
_a = JSII_RTTI_SYMBOL_1;
SnsToSqs[_a] = { fqn: "@aws-solutions-constructs/aws-sns-sqs.SnsToSqs", version: "2.58.1" };
function DoWeNeedACmk(existingResource, keyInResourceProps, encryptionFlag) {
    // We only need tp create a CMK if
    //   -The client did not supply an existing resource
    //   -The client did not specify an existing key in the resource properties
    //   -The client did not explicitly turn off encryption by setting the flag to false.
    if (!existingResource &&
        !keyInResourceProps &&
        defaults.CheckBooleanWithDefault(encryptionFlag, true)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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