"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const assertions_1 = require("aws-cdk-lib/assertions");
const utils_1 = require("./utils");
// This set of tests targets the legacy functionality, so each text MUST
// start with:
//        stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
test('Pattern deployment w/ new Topic, new Queue and default props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {};
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
        TopicArn: {
            Ref: "testsnssqsSnsTopic2CD0065B"
        },
        Endpoint: {
            "Fn::GetAtt": [
                "testsnssqsqueueB02504BF",
                "Arn"
            ]
        }
    });
});
test('Pattern deployment w/ new topic, new queue, and overridden props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {
        topicProps: {
            topicName: "new-topic",
        },
        queueProps: {
            queueName: "new-queue.fifo",
            fifo: true
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKeyProps: {
            enableKeyRotation: false
        },
        deployDeadLetterQueue: false,
        maxReceiveCount: 0
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Topic", {
        TopicName: "new-topic",
    });
    // Assertion 2
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "new-queue.fifo",
        FifoQueue: true
    });
    // Assertion 3
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
test('Test getter methods, old interface', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: true,
        deployDeadLetterQueue: true,
        maxReceiveCount: 0
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.resourceCountIs("AWS::KMS::Key", 1);
});
test('Test deployment w/ existing queue, and topic', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingTopicObj: topic,
        existingQueueObj: queue
    });
    expect(app.snsTopic).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SNS::Topic", {
        TopicName: "existing-topic-obj"
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
});
test('Test deployment with imported encryption key, old interface', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false
    });
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: kmsKey
    });
    // Assertion 2
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    // Assertion 3
    template.hasResourceProperties("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "importedkey38675D68",
                "Arn"
            ]
        }
    });
});
test('Test deployment with SNS managed KMS key, old interface', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns')
        },
        queueProps: {
            encryption: sqs.QueueEncryption.KMS
        },
        enableEncryptionWithCustomerManagedKey: false
    });
    // Assertion 2
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.sse);
});
test('Test deployment with CMK encrypted SNS Topic (avoids interface)', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const testDescription = 'someString-llasdj';
    const cmk = new kms.Key(stack, 'cmk', { description: testDescription });
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: cmk
        }
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::KMS::Key", {
        Description: testDescription
    });
    // A key is still created for the SQS queue, so there are 2 keys in the stack
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
});
test('Pattern deployment w/ existing topic and FIFO queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'TestTopic', {
        contentBasedDeduplication: true,
        displayName: 'Customer subscription topic',
        fifo: true,
        topicName: 'customerTopic',
    });
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        existingTopicObj: topic,
        queueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: true,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: true,
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    template.resourceCountIs("AWS::KMS::Key", 0);
    template.hasResourceProperties("AWS::SQS::Queue", {
        FifoQueue: true,
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testsnssqsdeadLetterQueue8DACC0A1",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
});
test('Pattern deployment w/ existing topic and Standard queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'TestTopic', {
        displayName: 'Customer subscription topic',
        fifo: false,
        topicName: 'customerTopic',
    });
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        existingTopicObj: topic,
        queueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: false,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: false,
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    template.hasResourceProperties("AWS::SQS::Queue", {
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testsnssqsdeadLetterQueue8DACC0A1",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
});
test('Check raw message delivery is true', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {
        sqsSubscriptionProps: {
            rawMessageDelivery: true
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
        TopicArn: {
            Ref: "testsnssqsSnsTopic2CD0065B"
        },
        Endpoint: {
            "Fn::GetAtt": [
                "testsnssqsqueueB02504BF",
                "Arn"
            ]
        },
        RawMessageDelivery: true
    });
});
test('Check for filter policy', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {
        sqsSubscriptionProps: {
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    allowlist: ['red', 'orange'],
                    matchPrefixes: ['bl'],
                })
            }
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        FilterPolicy: {
            color: [
                "red",
                "orange",
                {
                    prefix: "bl"
                }
            ]
        }
    });
});
test('Check SNS dead letter queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {});
    const props = {
        sqsSubscriptionProps: {
            deadLetterQueue: dlq
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "existingdlqobj784C5542",
                    "Arn"
                ]
            }
        }
    });
});
test('Construct uses topicProps.masterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const cmk = new kms.Key(stack, 'cmkfortopic');
    const props = {
        topicProps: {
            masterKey: cmk
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmkfortopic0904647B",
                "Arn"
            ]
        }
    });
});
test('Construct uses queueProps.encryptionMasterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const cmk = new kms.Key(stack, 'cmkforqueue');
    const props = {
        queueProps: {
            encryptionMasterKey: cmk
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmkforqueue4E093476",
                "Arn"
            ]
        }
    });
});
test('Construct does not override unencrypted topic when passed in existingTopicObj prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const props = {
        existingTopicObj,
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(testConstruct.snsTopic).toBeDefined();
    // Ensure that the existing topic above is the only topic in the stack
    template.resourceCountIs("AWS::SNS::Topic", 1);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
});
test('Confirm that CheckSnsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicObj: topic,
            topicProps: {
                topicName: 'topic-name'
            },
            existingQueueObj: queue
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps or existingTopicObj, but not both.\n/);
});
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that CheckSnsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicObj: topic,
            topicProps: {
                topicName: 'topic-name'
            },
            existingQueueObj: queue
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps or existingTopicObj, but not both.\n/);
});
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that Construct checks for mixed deprecated and active props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            enableEncryptionWithCustomerManagedKey: true,
            encryptQueueWithCmk: true
        });
    };
    expect(app).toThrowError(/Cannot specify both deprecated key props and new key props/);
});
test('Error if enableEncryption is false and encryption settings are provided', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            enableEncryptionWithCustomerManagedKey: false,
            encryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError("Error - if enableEncryptionWithCustomerManagedKey is false, submitting encryptionKey or encryptionKeyProps is invalid\n");
});
test('test CreateRequiredKeys for no arguments', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {});
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Topic is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingTopicObj: {}
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Queue is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingQueueObj: {}
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Topic props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        topicProps: {
            masterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Queue props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        queueProps: {
            encryptionMasterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).toBeDefined();
});
// ***************************
//
// Over the course of implementing the new interface and keeping the old interface capability,
// it became clear that the old interface had some irregular behavior that we did not want to alter.
// So we made a table of behavior for different inputs, then ran a test for each row of that table
// to capture the current behavior. These are those tests that protect existing behavior. Some
// repeat tests already implemented above, we chose not to worry about that.
//
// ***************************
test('1 Legacy Behavior - Queue Props, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        queueProps: {
            encryptionMasterKey: existingKey,
        }
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.sse);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.sse);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('2 Legacy Behavior - Queue Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        queueProps: {
            encryptionMasterKey: existingKey,
        }
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('4 Legacy Behavior - Topic Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        topicProps: {
            masterKey: existingKey,
        }
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('5 Legacy Behavior - Topic Props, Existing Key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
            encryptionKey: existingKey,
            topicProps: {
                masterKey: existingKey,
            }
        });
    };
    expect(app).toThrowError();
});
test('6 Legacy Behavior - Queue Props, Existing Key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
            encryptionKey: existingKey,
            queueProps: {
                encryptionMasterKey: existingKey,
            }
        });
    };
    expect(app).toThrowError();
});
test('7 Legacy Behavior - Queue Props, Key Props provided', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
            encryptionKeyProps: {},
            queueProps: {
                encryptionMasterKey: existingKey,
            }
        });
    };
    expect(app).toThrowError();
});
test('8 Legacy Behavior - Topic Props, Queue Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const existingKeyTwo = new kms.Key(stack, 'test-existing-key-two', {
        enableKeyRotation: true,
        alias: 'existingKeyTwo'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        topicProps: {
            masterKey: existingKey,
        },
        queueProps: {
            encryptionMasterKey: existingKeyTwo
        }
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    // Legacy code created an extra, unneeded key. New
    // code returns this to 2
    template.resourceCountIs("AWS::KMS::Key", 2);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('9 Legacy Behavior - Topic Props, Queue Props, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const existingKeyTwo = new kms.Key(stack, 'test-existing-key-two', {
        enableKeyRotation: true,
        alias: 'existingKeyTwo'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        topicProps: {
            masterKey: existingKey,
        },
        queueProps: {
            encryptionMasterKey: existingKeyTwo
        }
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('10 Legacy Behavior - Existing Topic and Queue, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        existingQueueObj: queue,
        existingTopicObj: topic
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 0);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
});
test('11 Legacy Behavior - Existing Topic and Queue, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        existingQueueObj: queue,
        existingTopicObj: topic
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    // Legacy code created an unused key here, new code
    // does not so this is 0
    template.resourceCountIs("AWS::KMS::Key", 0);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
});
test('13 Legacy Behavior - Existing Topic, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        existingTopicObj: topic
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('14 Legacy Behavior - Existing Topic, Key Props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const topic = new sns.Topic(stack, 'test-topic', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: existingKey,
        existingTopicObj: topic
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('15 Legacy Behavior - Existing Topic, Existing Key, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        encryptionKeyProps: { enableKeyRotation: false },
        existingTopicObj: topic
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.none);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.cmk);
});
test('16 Legacy Behavior - Existing Queue, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        existingQueueObj: queue
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.sse);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 0);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.sse);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
});
test('17 Legacy Behavior - Existing Queue, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        existingQueueObj: queue
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
});
test('18 Legacy Behavior - Existing Queue, Existing Key, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: existingKey,
        existingQueueObj: queue
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
});
test('19 Legacy Behavior - Existing Queue, Key Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKeyProps: { enableKeyRotation: false },
        existingQueueObj: queue
    });
    utils_1.CheckKeyProperty(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    utils_1.CheckTopicKeyType(template, utils_1.keyType.cmk);
    utils_1.CheckQueueKeyType(template, utils_1.keyType.none);
});
//# sourceMappingURL=data:application/json;base64,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