import { custom_resources, aws_iam as iam, CustomResource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The properties of an OU.
 *
 * @stability stable
 */
export interface OUProps {
    /**
     * The name of the OU.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Whether or not to import an existing OU if the new OU is a duplicate.
     *
     * If this is false and the OU already exists an error will be thrown.
     *
     * @default false
     * @stability stable
     */
    readonly importOnDuplicate?: boolean;
    /**
     * Whether or not a missing OU should be recreated during an update.
     *
     * If this is false and the OU does not exist an error will be thrown when you try to update it.
     *
     * @default false
     * @stability stable
     */
    readonly allowRecreateOnUpdate?: boolean;
}
/**
 * The properties of an Account.
 *
 * @stability stable
 */
export interface AccountProps {
    /**
     * The name of the account.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The email address of the account.
     *
     * Most be unique.
     *
     * @stability stable
     */
    readonly email: string;
    /**
     * Whether or not to import an existing account if the new account is a duplicate.
     *
     * If this is false and the account already exists an error will be thrown.
     *
     * @default false
     * @stability stable
     */
    readonly importOnDuplicate?: boolean;
    /**
     * Whether or not to allow this account to be moved between OUs.
     *
     * If importing is enabled
     * this will also allow imported accounts to be moved.
     *
     * @default false
     * @stability stable
     */
    readonly allowMove?: boolean;
    /**
     * Whether or not attempting to delete an account should raise an error.
     *
     * Accounts cannot be deleted programmatically, but they can be removed as a managed resource.
     * This property will allow you to control whether or not an error is thrown
     * when the stack wants to delete an account (orphan it) or if it should continue
     * silently.
     *
     * @default false
     * @see https://aws.amazon.com/premiumsupport/knowledge-center/close-aws-account/
     * @stability stable
     */
    readonly disableDelete?: boolean;
}
/**
 * The structure of an OrgObject.
 *
 * @stability stable
 */
export interface OUObject {
    /**
     * The unique id of the OUObject.
     *
     * This is used as the unique identifier when instantiating a construct object.
     * This is important for the CDK to be able to maintain a reference for the object when utilizing
     * the processOUObj function rather then using the name property of an object which could change.
     * If the id changes the CDK treats this as a new construct and will create a new construct resource and
     * destroy the old one.
     *
     * Not strictly required but useful when using the processOUObj function. If the id is not provided
     * the name property will be used as the id in processOUObj.
     *
     * You can create a unique id however you like. A bash example is provided.
     *
     * @stability stable
     * @example
     *
     * echo "ou-$( echo $RANDOM | md5sum | head -c 8 )"
     */
    readonly id?: string;
    /**
     * The OU object properties.
     *
     * @stability stable
     */
    readonly properties: OUProps;
    /**
     * Accounts that belong to this OU.
     *
     * @stability stable
     */
    readonly accounts?: AccountProps[];
    /**
     * OUs that are children of this OU.
     *
     * @stability stable
     */
    readonly children: OUObject[];
}
/**
 * @function processOUObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} ouProvider The custom resource provider for managing OUs
 * @param {custom_resources.Provider} accountProvider The custom resource provider for managing accounts
 * @param {OUObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
export declare function processOUObj(this: Construct, ouProvider: custom_resources.Provider, accountProvider: custom_resources.Provider, obj: OUObject, parent: string | OrganizationOU): void;
/**
 * The properties for the OU custom resource provider.
 *
 * @stability stable
 */
export interface OrganizationOUProviderProps {
    /**
     * The role the custom resource should use for taking actions on OUs if one is not provided one will be created automatically.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
}
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
export declare class OrganizationOUProvider extends Construct {
    /**
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: OrganizationOUProviderProps);
}
/**
 * The properties of an OrganizationOU custom resource.
 *
 * @stability stable
 */
export interface OUResourceProps extends OUProps {
    /**
     * The parent OU id.
     *
     * @stability stable
     */
    readonly parent: string | OrganizationOU;
    /**
     * The provider to use for the custom resource that will create the OU.
     *
     * You can create a provider with the OrganizationOuProvider class
     *
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
}
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
export declare class OrganizationOU extends Construct {
    /**
     * @stability stable
     */
    readonly resource: CustomResource;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: OUResourceProps);
}
/**
 * The properties for the account custom resource provider.
 *
 * @stability stable
 */
export interface OrganizationAccountProviderProps {
    /**
     * The role the custom resource should use for taking actions on OUs if one is not provided one will be created automatically.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
}
/**
 * The provider for account custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating accounts.
 *
 * @stability stable
 */
export declare class OrganizationAccountProvider extends Construct {
    /**
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: OrganizationOUProviderProps);
}
/**
 * The properties of an OrganizationAccount custom resource.
 *
 * @stability stable
 */
export interface AccountResourceProps extends AccountProps {
    /**
     * The parent OU id.
     *
     * @stability stable
     */
    readonly parent: string | OrganizationOU;
    /**
     * The provider to use for the custom resource that will create the OU.
     *
     * You can create a provider with the OrganizationOuProvider class
     *
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
}
/**
 * The construct to create or update an Organization account.
 *
 * This relies on the custom resource provider OrganizationAccountProvider.
 *
 * @stability stable
 */
export declare class OrganizationAccount extends Construct {
    /**
     * @stability stable
     */
    readonly resource: CustomResource;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: AccountResourceProps);
}
/**
 * The properties of an IPAM administrator custom resource provider.
 *
 * @stability stable
 */
export interface IPAMAdministratorProviderProps {
    /**
     * The role the custom resource should use for working with the IPAM administrator delegation if one is not provided one will be created automatically.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
}
/**
 * @stability stable
 */
export declare class IPAMAdministratorProvider extends Construct {
    /**
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: IPAMAdministratorProviderProps);
}
/**
 * The properties of an OrganizationAccount custom resource.
 *
 * @stability stable
 */
export interface IPAMAdministratorProps {
    /**
     * The account id of the IPAM administrator.
     *
     * @stability stable
     */
    readonly delegatedAdminAccountId: string;
    /**
     * The provider to use for the custom resource that will handle IPAM admin delegation.
     *
     * You can create a provider with the IPAMAdministratorProvider class
     *
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
}
/**
 * The construct to create or update the delegated IPAM administrator for an organization.
 *
 * This relies on the custom resource provider IPAMAdministratorProvider.
 *
 * @stability stable
 */
export declare class IPAMdministrator extends Construct {
    /**
     * @stability stable
     */
    readonly resource: CustomResource;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: IPAMAdministratorProps);
}
