"""
Settings
"""

from __future__ import annotations

from restfly import APIEndpoint
from restfly.utils import dict_clean

from .schema.settings import SettingsListSchema


class SettingsAPI(APIEndpoint):
    _path = "settings"

    def modify(self, settings: list[dict]) -> dict:
        """
        Modifies the advanced settings on the Nessus scanner.  Settings objects
        must contain an action and a name field.  They may also require a value
        field and/or an id field depending on the nature of the change.

        Args:
            settings (list[dict]): List of settings change objects

        Examples:

            Adding a new value:

            >>> nessus.settings.modify([{
            ...     'action': 'add',
            ...     'name': 'new_value',
            ...     'value': 'value_contents'
            ... }])

            Updating a default setting value:

            >>> nessus.settings.modify([{
            ...     'action': 'edit',
            ...     'name': 'allow_post_scan_editing',
            ...     'value': 'no'
            ... }])

            Removing a setting:

            >>> nessus.settings.modify([{
            ...     'action': 'remove',
            ...     'name': 'old_setting',
            ...     'id': 'abcdef1234567890abcdef'
            ... }])
        """
        schema = SettingsListSchema()
        payload = schema.dump(schema.load({"settings": settings}))
        return self._put("advanced", json=payload)

    def list(self) -> list[dict]:
        """
        Returns the list of advanced settings

        Returns:
            List[Dict]:
                List of settings objects.

        Example:

            >>> nessus.settings.list()
        """
        return self._get("advanced")["preferences"]

    def health(self) -> dict:
        """
        Returns the current health statistics fore the Nessus scanner

        Returns:
            Dict:
                Health stats information

        Example:

            >>> nessus.settings.health()
        """
        return self._get("health/stats")

    def alerts(self, start_time: int | None = None, end_time: int | None = None) -> list[dict]:
        """
        Returns the list of health alerts generated by the scanner

        Args:
            start_time (int, optional):
                Start time to query the historical data for.  Defaults to 24hrs
                ago.
            end_time (int, optional):
                End time to query the historicat data for.  Defaults to now.

        Returns:
            List[Dict]:
                List of alert objects matching the specified time range

        Example:

            >>> nessus.settings.alerts()
        """
        return self._get("health/alerts", params=dict_clean({"start_time": start_time, "end_time": end_time}))
