from __future__ import (absolute_import, division, print_function,
                        unicode_literals)

import functools
from contextlib import contextmanager
from time import time

from six import exec_


class TimedResult(object):
    def get_value(self):
        """
        :return: Duration in miliseconds
        :rtype: float
        """
        try:
            return self._value
        except AttributeError:
            raise Exception('You can only get the value after exiting timing context')

    def _set_value(self, value):
        self._value = value


@contextmanager
def timed(time=time):
    start = time()
    result = TimedResult()
    yield result
    duration = (time() - start) * 1000.0
    result._set_value(duration)


def slots_repr(self):
    """Generates object repr based on object's __slots__.

    Operates under assumption that __slots__ correspond to object's class' constructor
    parameters.
    """
    return '%s(%s)' % (
        type(self).__name__,
        ', '.join('%s=%r' % (key, getattr(self, key)) for key in self.__slots__)
    )


def memoized_by_args(*args):
    """Create memoization decorator more efficient than one generated by memoized.

    The downside is default values are not supported and you need to declare the arguments manually.

    >>> def x(a, b):
    ...     print('ping')
    ...     return a + b
    ...
    >>> memx = memoized_by_args('a', 'b')(x)
    >>> memx(1, 2)
    ping
    3
    >>> memx(1, 2)
    3
    """
    cache = {}  # noqa
    exec_(
        '''
def decorator(fun):
    @functools.wraps(fun)
    def wrapper(%(args)s, cache=cache, fun=fun):
        key = (%(args)s)
        try:
            return cache[key]
        except KeyError:
            value = cache[key] = fun(%(args)s)
            return value
    return wrapper''' % dict(args=', '.join(args)),
        dict(cache=cache, functools=functools), locals())
    return locals()['decorator']
