"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const defaults = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const iam = require("aws-cdk-lib/aws-iam");
const assertions_1 = require("aws-cdk-lib/assertions");
// -----------------------------------------------------------------------------------------
// Pattern deployment with new Lambda function, new Sagemaker endpoint and deployVpc = true
// -----------------------------------------------------------------------------------------
test('Pattern deployment with new Lambda function, new Sagemaker endpoint, deployVpc = true', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 128,
        },
        deployVpc: true,
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                SAGEMAKER_ENDPOINT_NAME: {
                    'Fn::GetAtt': ['testlambdasagemakerSagemakerEndpoint12803730', 'EndpointName'],
                },
            },
        },
        VpcConfig: {
            SecurityGroupIds: [
                {
                    'Fn::GetAtt': ['testlambdasagemakerReplaceDefaultSecurityGroupsecuritygroupB2FD7810', 'GroupId'],
                },
            ],
            SubnetIds: [
                {
                    Ref: 'VpcisolatedSubnet1SubnetE62B1B9B',
                },
                {
                    Ref: 'VpcisolatedSubnet2Subnet39217055',
                },
            ],
        },
    });
    // Assertion 3
    template.hasResourceProperties('AWS::SageMaker::Model', {
        ExecutionRoleArn: {
            'Fn::GetAtt': ['testlambdasagemakerSagemakerRoleD84546B8', 'Arn'],
        },
        PrimaryContainer: {
            Image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
            ModelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
        },
        VpcConfig: {
            SecurityGroupIds: [
                {
                    'Fn::GetAtt': ['testlambdasagemakerReplaceModelDefaultSecurityGroup7284AA24', 'GroupId'],
                },
            ],
            Subnets: [
                {
                    Ref: 'VpcisolatedSubnet1SubnetE62B1B9B',
                },
                {
                    Ref: 'VpcisolatedSubnet2Subnet39217055',
                },
            ],
        },
    });
    // Assertion 4
    template.hasResourceProperties('AWS::SageMaker::EndpointConfig', {
        ProductionVariants: [
            {
                InitialInstanceCount: 1,
                InitialVariantWeight: 1,
                InstanceType: 'ml.m4.xlarge',
                ModelName: {
                    'Fn::GetAtt': ['testlambdasagemakerSagemakerModelEC3E4E39', 'ModelName'],
                },
                VariantName: 'AllTraffic',
            },
        ],
        KmsKeyId: {
            Ref: 'testlambdasagemakertestlambdasagemakerKey5BF0746F',
        },
    });
    // Assertion 5
    template.hasResourceProperties('AWS::SageMaker::Endpoint', {
        EndpointConfigName: {
            'Fn::GetAtt': ['testlambdasagemakerSagemakerEndpointConfig6BABA334', 'EndpointConfigName'],
        },
    });
});
// ----------------------------------------------------------------------------------------------
// Pattern deployment with existing Lambda function, new Sagemaker endpoint and deployVpc = false
// ----------------------------------------------------------------------------------------------
test('Pattern deployment with existing Lambda function, new Sagemaker endpoint, deployVpc = false', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: aws_cdk_lib_1.Duration.minutes(5),
        memorySize: 128,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        existingLambdaObj: fn,
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SageMaker::Model', {
        ExecutionRoleArn: {
            'Fn::GetAtt': ['testlambdasagemakerSagemakerRoleD84546B8', 'Arn'],
        },
        PrimaryContainer: {
            Image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
            ModelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
        },
    });
    // Assertion 3
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                SAGEMAKER_ENDPOINT_NAME: {
                    'Fn::GetAtt': ['testlambdasagemakerSagemakerEndpoint12803730', 'EndpointName'],
                },
            },
        },
    });
});
// ------------------------------------------------------------------------------------------------------------------
// Pattern deployment with new Lambda function, new Sagemaker endpoint, deployVpc = true, and custom role
// ------------------------------------------------------------------------------------------------------------------
test('Pattern deployment with new Lambda function, new Sagemaker endpoint, deployVpc = true, and custom role', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    // Create IAM Role to be assumed by SageMaker
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    sagemakerRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'));
    const constructProps = {
        modelProps: {
            executionRoleArn: sagemakerRole.roleArn,
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        deployVpc: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 128,
        },
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'sagemaker.amazonaws.com',
                    },
                },
            ],
            Version: '2012-10-17',
        },
    });
    // Assertion 2: ReplaceDefaultSecurityGroup, ReplaceEndpointDefaultSecurityGroup, and ReplaceModelDefaultSecurityGroup
    template.resourceCountIs('AWS::EC2::SecurityGroup', 3);
    // Assertion 3
    template.resourceCountIs('AWS::EC2::Subnet', 2);
    // Assertion 4
    template.resourceCountIs('AWS::EC2::InternetGateway', 0);
    // Assertion 5: SAGEMAKER_RUNTIME VPC Interface
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
    // Assertion 6: S3 VPC Endpoint
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Gateway',
    });
    // Assertion 7
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// ---------------------------------------------------------------------------------
// Test for error when existing Lambda function does not have vpc and deployVpc = true
// ---------------------------------------------------------------------------------
test('Test for error when existing Lambda function does not have vpc and deployVpc = true ', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: aws_cdk_lib_1.Duration.minutes(5),
        memorySize: 128,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        deployVpc: true,
        existingLambdaObj: fn,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------------------------------------
// Pattern deployment with existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc
// -------------------------------------------------------------------------------------------------------
test('Pattern deployment with existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpoint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpoint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: aws_cdk_lib_1.Duration.minutes(5),
        memorySize: 128,
        vpc,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        existingVpc: vpc,
        existingLambdaObj: fn,
    };
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertion 2: ReplaceDefaultSecurityGroup, ReplaceEndpointDefaultSecurityGroup, and ReplaceModelDefaultSecurityGroup
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::SecurityGroup', 3);
    // Assertion 3
    template.resourceCountIs('AWS::EC2::Subnet', 2);
    // Assertion 4
    template.resourceCountIs('AWS::EC2::InternetGateway', 0);
    // Assertion 5: SAGEMAKER_RUNTIME VPC Interface
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
    // Assertion 6: S3 VPC Endpoint
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Gateway',
    });
    // Assertion 7
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// -----------------------------------------------------------------------------------------
// Test for error with existingLambdaObj/lambdaFunctionProps=undefined (not supplied by user)
// -----------------------------------------------------------------------------------------
test('Test for error with existingLambdaObj/lambdaFunctionProps=undefined (not supplied by user)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// --------------------------------------------------------------------
// Test for error with (props.deployVpc && props.existingVpc) is true
// --------------------------------------------------------------------
test('confirm CheckVpcProps is called', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpoint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpoint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 128,
        },
        deployVpc: true,
        existingVpc: vpc,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    };
    // Assertion 1
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
// ----------------------------------------------------------------------------------------------------------
// Test for error with primaryContainer=undefined (not supplied by user), and no existingSageMakerEndpointObj
// ----------------------------------------------------------------------------------------------------------
test('Test for error with primaryContainer=undefined (not supplied by user)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: aws_cdk_lib_1.Duration.minutes(5),
        memorySize: 128,
    });
    const constructProps = {
        modelProps: {},
        deployVpc: true,
        existingLambdaObj: fn,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------------------------------
// Test getter methods: existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc
// -------------------------------------------------------------------------------------------------
test('Test getter methods: existing Lambda function (with VPC), new Sagemaker endpoint, and existingVpc', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpoint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpoint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // deploy lambda function
    const fn = defaults.deployLambdaFunction(stack, {
        runtime: lambda.Runtime.PYTHON_3_8,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        handler: 'index.handler',
        timeout: aws_cdk_lib_1.Duration.minutes(5),
        memorySize: 128,
        vpc,
    });
    const constructProps = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        existingVpc: vpc,
        existingLambdaObj: fn,
    };
    const app = new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertions
    expect(app.lambdaFunction).toBeDefined();
    expect(app.sagemakerEndpoint).toBeDefined();
    expect(app.sagemakerEndpointConfig).toBeDefined();
    expect(app.sagemakerModel).toBeDefined();
    expect(app.vpc).toBeDefined();
});
// --------------------------------------------------------------------------------------------
// Test getter methods: new Lambda function, existingSagemakerendpointObj (no vpc)
// --------------------------------------------------------------------------------------------
test('Test getter methods: new Lambda function, existingSagemakerendpointObj (no vpc)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const deploySagemakerEndpointResponse = defaults.deploySagemakerEndpoint(stack, 'test', {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    const constructProps = {
        existingSagemakerEndpointObj: deploySagemakerEndpointResponse.endpoint,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 128,
        },
    };
    const app = new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertions
    expect(app.lambdaFunction).toBeDefined();
    expect(app.sagemakerEndpoint).toBeDefined();
    expect(app.sagemakerEndpointConfig).toBeUndefined();
    expect(app.sagemakerModel).toBeUndefined();
    expect(app.vpc).toBeUndefined();
});
// --------------------------------------------------------------------------------------------
// Test getter methods: new Lambda function, existingSagemakerendpointObj and deployVpc = true
// --------------------------------------------------------------------------------------------
test('Test getter methods: new Lambda function, existingSagemakerendpointObj and deployVpc = true', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const deploySagemakerEndpointResponse = defaults.deploySagemakerEndpoint(stack, 'test', {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    const constructProps = {
        existingSagemakerEndpointObj: deploySagemakerEndpointResponse.endpoint,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 128,
        },
        deployVpc: true,
    };
    const app = new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', constructProps);
    // Assertions
    expect(app.lambdaFunction).toBeDefined();
    expect(app.sagemakerEndpoint).toBeDefined();
    expect(app.sagemakerEndpointConfig).toBeUndefined();
    expect(app.sagemakerModel).toBeUndefined();
    expect(app.vpc).toBeDefined();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const deploySagemakerEndpointResponse = defaults.deploySagemakerEndpoint(stack, 'test', {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    new lib_1.LambdaToSagemakerEndpoint(stack, 'test-lambda-sagemaker', {
        existingSagemakerEndpointObj: deploySagemakerEndpointResponse.endpoint,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        sagemakerEnvironmentVariableName: 'CUSTOM_SAGEMAKER_ENDPOINT'
    });
    // Assertion
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'python3.8',
        Environment: {
            Variables: {
                CUSTOM_SAGEMAKER_ENDPOINT: {
                    'Fn::GetAtt': [
                        'SagemakerEndpoint',
                        'EndpointName'
                    ]
                }
            }
        }
    });
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
test('Confirm call to CheckSagemakerProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const deploySagemakerEndpointResponse = defaults.deploySagemakerEndpoint(stack, 'test', {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    const props = {
        existingSagemakerEndpointObj: deploySagemakerEndpointResponse.endpoint,
        endpointProps: { endpointConfigName: 'test' },
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
    };
    const app = () => {
        new lib_1.LambdaToSagemakerEndpoint(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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