#
#  Tests for UDMorphConverter. Smoke testing the main functionalities.
#  Code for extensive testing on the UD corpus can be found here: 
#    https://github.com/estnltk/estnltk-model-training/tree/main/ud_morph_tools/eval_ud_morph_conv
#

from collections import OrderedDict

from estnltk import Text
from estnltk.common import PACKAGE_PATH

from estnltk.converters.conll.conll_exporter import layer_to_conll

from estnltk.taggers.standard.morph_analysis.ud_morf import UDMorphConverter

from estnltk.converters import layer_to_dict
from estnltk.converters import dict_to_layer

import os, os.path

# -------------------------------------------------------------------------------

def test_ud_morph_converter_smoke():
    # Smoke test for UDMorphConverter
    
    # Case 1 
    text = Text('Ilus suur kass lesis kollasel diivanil.')
    # Tag required layers
    text.tag_layer(['words', 'sentences', 'morph_analysis'])
    # Convert morph categories to UD
    ud_converter = UDMorphConverter( remove_connegatives=True, 
                                     generate_num_cases=True )
    assert ud_converter.output_layer == 'ud_morph_analysis'
    ud_converter.tag( text )
    assert ud_converter.output_layer in text.layers
    
    expected_ud_morph_layer_dict = \
        {'ambiguous': True,
         'attributes': ('id', 'lemma', 'upostag', 'xpostag', 'feats', 'misc'),
         'enveloping': None,
         'meta': {},
         'name': 'ud_morph_analysis',
         'parent': 'morph_analysis',
         'secondary_attributes': (),
         'serialisation_module': None,
         'spans': [{'annotations': [{'feats': OrderedDict([('Degree', 'Pos'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 1,
                                     'lemma': 'ilus',
                                     'misc': '',
                                     'upostag': 'ADJ',
                                     'xpostag': 'A'}],
                    'base_span': (0, 4)},
                   {'annotations': [{'feats': OrderedDict([('Degree', 'Pos'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 2,
                                     'lemma': 'suur',
                                     'misc': '',
                                     'upostag': 'ADJ',
                                     'xpostag': 'A'}],
                    'base_span': (5, 9)},
                   {'annotations': [{'feats': OrderedDict([('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 3,
                                     'lemma': 'kass',
                                     'misc': '',
                                     'upostag': 'NOUN',
                                     'xpostag': 'S'}],
                    'base_span': (10, 14)},
                   {'annotations': [{'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Past'),
                                                           ('Mood', 'Ind'),
                                                           ('VerbForm', 'Fin'),
                                                           ('Number', 'Sing'),
                                                           ('Person', '3')]),
                                     'id': 4,
                                     'lemma': 'lesima',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'}],
                    'base_span': (15, 20)},
                   {'annotations': [{'feats': OrderedDict([('Degree', 'Pos'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ade')]),
                                     'id': 5,
                                     'lemma': 'kollane',
                                     'misc': '',
                                     'upostag': 'ADJ',
                                     'xpostag': 'A'}],
                    'base_span': (21, 29)},
                   {'annotations': [{'feats': OrderedDict([('Number', 'Sing'),
                                                           ('Case', 'Ade')]),
                                     'id': 6,
                                     'lemma': 'diivan',
                                     'misc': '',
                                     'upostag': 'NOUN',
                                     'xpostag': 'S'}],
                    'base_span': (30, 38)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 7,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (38, 39)}]}
    assert expected_ud_morph_layer_dict == layer_to_dict( text['ud_morph_analysis'] )
    
    #from pprint import pprint
    #pprint( layer_to_dict(text['ud_morph_analysis']) )

# -------------------------------------------------------------------------------

def test_ud_morph_converter_remove_connegatives():
    # Test that redundant connegative annotations are removed if possible.
    text = Text('Vist peaks tegutsema. Mingem edasi. Las jääb nii. Ei elata.')
    # Tag required layers
    text.tag_layer(['words', 'sentences', 'morph_analysis'])
    # Convert morph categories to UD
    ud_converter = UDMorphConverter( remove_connegatives=True )
    ud_converter.tag( text )
    
    expected_ud_morph_layer_dict = \
        {'ambiguous': True,
         'attributes': ('id', 'lemma', 'upostag', 'xpostag', 'feats', 'misc'),
         'enveloping': None,
         'meta': {},
         'name': 'ud_morph_analysis',
         'parent': 'morph_analysis',
         'secondary_attributes': (),
         'serialisation_module': None,
         'spans': [{'annotations': [{'feats': OrderedDict(),
                                     'id': 1,
                                     'lemma': 'vist',
                                     'misc': '',
                                     'upostag': 'ADV',
                                     'xpostag': 'D'}],
                    'base_span': (0, 4)},
                   {'annotations': [{'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Cnd'),
                                                           ('VerbForm', 'Fin')]),
                                     'id': 2,
                                     'lemma': 'pidama',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'},
                                    {'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Cnd'),
                                                           ('VerbForm', 'Fin')]),
                                     'id': 2,
                                     'lemma': 'pidama',
                                     'misc': '',
                                     'upostag': 'AUX',
                                     'xpostag': 'V'}],
                    'base_span': (5, 10)},
                   {'annotations': [{'feats': OrderedDict([('Voice', 'Act'),
                                                           ('VerbForm', 'Sup'),
                                                           ('Case', 'Ill')]),
                                     'id': 3,
                                     'lemma': 'tegutsema',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'}],
                    'base_span': (11, 20)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 4,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (20, 21)},
                   {'annotations': [{'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Imp'),
                                                           ('VerbForm', 'Fin'),
                                                           ('Number', 'Plur'),
                                                           ('Person', '1')]),
                                     'id': 1,
                                     'lemma': 'minema',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'}],
                    'base_span': (22, 28)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 2,
                                     'lemma': 'edasi',
                                     'misc': '',
                                     'upostag': 'ADV',
                                     'xpostag': 'D'}],
                    'base_span': (29, 34)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 3,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (34, 35)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 1,
                                     'lemma': 'las',
                                     'misc': '',
                                     'upostag': 'ADV',
                                     'xpostag': 'D'}],
                    'base_span': (36, 39)},
                   {'annotations': [{'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Ind'),
                                                           ('VerbForm', 'Fin'),
                                                           ('Number', 'Sing'),
                                                           ('Person', '3')]),
                                     'id': 2,
                                     'lemma': 'jääma',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'}],
                    'base_span': (40, 44)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 3,
                                     'lemma': 'nii',
                                     'misc': '',
                                     'upostag': 'ADV',
                                     'xpostag': 'D'}],
                    'base_span': (45, 48)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 4,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (48, 49)},
                   {'annotations': [{'feats': OrderedDict([('Polarity', 'Neg')]),
                                     'id': 1,
                                     'lemma': 'ei',
                                     'misc': '',
                                     'upostag': 'AUX',
                                     'xpostag': 'V'}],
                    'base_span': (50, 52)},
                   {'annotations': [{'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Ind'),
                                                           ('VerbForm', 'Fin'),
                                                           ('Connegative', 'Yes')]),
                                     'id': 2,
                                     'lemma': 'elatama',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'},
                                    {'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Imp'),
                                                           ('VerbForm', 'Fin'),
                                                           ('Number', 'Sing'),
                                                           ('Person', '2')]),
                                     'id': 2,
                                     'lemma': 'elatama',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'},
                                    {'feats': OrderedDict([('Voice', 'Act'),
                                                           ('Tense', 'Pres'),
                                                           ('Mood', 'Imp'),
                                                           ('VerbForm', 'Fin'),
                                                           ('Number', 'Sing'),
                                                           ('Person', '2'),
                                                           ('Connegative', 'Yes')]),
                                     'id': 2,
                                     'lemma': 'elatama',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'}],
                    'base_span': (53, 58)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 3,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (58, 59)}]}
    
    assert expected_ud_morph_layer_dict == layer_to_dict( text['ud_morph_analysis'] )
    
    #from pprint import pprint
    #pprint( layer_to_dict(text['ud_morph_analysis']) )


# -------------------------------------------------------------------------------

def test_ud_morph_converter_generate_num_cases():
    # Test that number/case information can be generated for numberics
    # Note that the current generation approach is excessive: it generates 
    # all 14 cases both in sing/plur. Disambiguation is desirable
    text = Text('Kunstielu 17. sajandil. Kell 20.00 ja saadetud 18 kirja.')
    # Tag required layers
    text.tag_layer(['words', 'sentences', 'morph_analysis'])
    # Convert morph categories to UD
    ud_converter = UDMorphConverter( generate_num_cases=True )
    ud_converter.tag( text )
    
    expected_ud_morph_layer_dict = \
        {'ambiguous': True,
         'attributes': ('id', 'lemma', 'upostag', 'xpostag', 'feats', 'misc'),
         'enveloping': None,
         'meta': {},
         'name': 'ud_morph_analysis',
         'parent': 'morph_analysis',
         'secondary_attributes': (),
         'serialisation_module': None,
         'spans': [{'annotations': [{'feats': OrderedDict([('Number', 'Sing'),
                                                           ('Case', 'Par')]),
                                     'id': 1,
                                     'lemma': 'kunstielu',
                                     'misc': '',
                                     'upostag': 'NOUN',
                                     'xpostag': 'S'}],
                    'base_span': (0, 9)},
                   {'annotations': [{'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Gen')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Par')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ill')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ine')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ela')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'All')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ade')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Abl')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Tra')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ter')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ess')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Abe')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Com')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Nom')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Gen')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Par')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ill')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ine')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ela')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'All')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ade')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Abl')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Tra')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ter')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ess')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Abe')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'},
                                    {'feats': OrderedDict([('NumType', 'Ord'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Com')]),
                                     'id': 2,
                                     'lemma': '17.',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'O'}],
                    'base_span': (10, 13)},
                   {'annotations': [{'feats': OrderedDict([('Number', 'Sing'),
                                                           ('Case', 'Ade')]),
                                     'id': 3,
                                     'lemma': 'sajand',
                                     'misc': '',
                                     'upostag': 'NOUN',
                                     'xpostag': 'S'}],
                    'base_span': (14, 22)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 4,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (22, 23)},
                   {'annotations': [{'feats': OrderedDict([('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 1,
                                     'lemma': 'kell',
                                     'misc': '',
                                     'upostag': 'NOUN',
                                     'xpostag': 'S'}],
                    'base_span': (24, 28)},
                   {'annotations': [{'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Gen')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Par')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ill')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ine')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ela')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'All')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ade')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Abl')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Tra')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ter')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ess')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Abe')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Com')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Nom')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Gen')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Par')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ill')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ine')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ela')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'All')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ade')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Abl')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Tra')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ter')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ess')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Abe')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Com')]),
                                     'id': 2,
                                     'lemma': '20.00',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'}],
                    'base_span': (29, 34)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 3,
                                     'lemma': 'ja',
                                     'misc': '',
                                     'upostag': 'CCONJ',
                                     'xpostag': 'J'},
                                    {'feats': OrderedDict(),
                                     'id': 3,
                                     'lemma': 'ja',
                                     'misc': '',
                                     'upostag': 'SCONJ',
                                     'xpostag': 'J'}],
                    'base_span': (35, 37)},
                   {'annotations': [{'feats': OrderedDict([('Degree', 'Pos'),
                                                           ('Voice', 'Pass'),
                                                           ('Tense', 'Past'),
                                                           ('VerbForm', 'Part')]),
                                     'id': 4,
                                     'lemma': 'saadetud',
                                     'misc': '',
                                     'upostag': 'ADJ',
                                     'xpostag': 'A'},
                                    {'feats': OrderedDict([('Voice', 'Pass'),
                                                           ('Tense', 'Past'),
                                                           ('VerbForm', 'Part')]),
                                     'id': 4,
                                     'lemma': 'saatma',
                                     'misc': '',
                                     'upostag': 'VERB',
                                     'xpostag': 'V'},
                                    {'feats': OrderedDict([('Degree', 'Pos'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Nom'),
                                                           ('Voice', 'Pass'),
                                                           ('Tense', 'Past'),
                                                           ('VerbForm', 'Part')]),
                                     'id': 4,
                                     'lemma': 'saadetud',
                                     'misc': '',
                                     'upostag': 'ADJ',
                                     'xpostag': 'A'},
                                    {'feats': OrderedDict([('Degree', 'Pos'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Nom'),
                                                           ('Voice', 'Pass'),
                                                           ('Tense', 'Past'),
                                                           ('VerbForm', 'Part')]),
                                     'id': 4,
                                     'lemma': 'saadetud',
                                     'misc': '',
                                     'upostag': 'ADJ',
                                     'xpostag': 'A'}],
                    'base_span': (38, 46)},
                   {'annotations': [{'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Nom')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Gen')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Par')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ill')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ine')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ela')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'All')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ade')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Abl')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Tra')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ter')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Ess')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Abe')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Sing'),
                                                           ('Case', 'Com')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Nom')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Gen')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Par')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ill')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ine')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ela')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'All')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ade')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Abl')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Tra')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ter')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Ess')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Abe')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'},
                                    {'feats': OrderedDict([('NumType', 'Card'),
                                                           ('NumForm', 'Digit'),
                                                           ('Number', 'Plur'),
                                                           ('Case', 'Com')]),
                                     'id': 5,
                                     'lemma': '18',
                                     'misc': '',
                                     'upostag': 'NUM',
                                     'xpostag': 'N'}],
                    'base_span': (47, 49)},
                   {'annotations': [{'feats': OrderedDict([('Number', 'Sing'),
                                                           ('Case', 'Par')]),
                                     'id': 6,
                                     'lemma': 'kiri',
                                     'misc': '',
                                     'upostag': 'NOUN',
                                     'xpostag': 'S'}],
                    'base_span': (50, 55)},
                   {'annotations': [{'feats': OrderedDict(),
                                     'id': 7,
                                     'lemma': '.',
                                     'misc': '',
                                     'upostag': 'PUNCT',
                                     'xpostag': 'Z'}],
                    'base_span': (55, 56)}]}
    
    assert expected_ud_morph_layer_dict == layer_to_dict( text['ud_morph_analysis'] )

    #from pprint import pprint
    #pprint( layer_to_dict(text['ud_morph_analysis']) )


# -------------------------------------------------------------------------------

def test_ud_morph_to_conllu_conversion():
    # Test converting UDMorphConverter's output layer to conllu
    
    text = Text('Rändur peaks kohe saabuma. Siis saame asjas selgust.')
    # Tag required layers
    text.tag_layer(['words', 'sentences', 'morph_analysis'])
    # Convert morph categories to UD
    ud_converter = UDMorphConverter( remove_connegatives=True, 
                                     generate_num_cases=True,
                                     add_deprel_attribs=True )
    assert ud_converter.output_attributes == ('id', 'lemma', 'upostag', 'xpostag', 'feats', 'head', 'deprel', 'deps', 'misc')
    ud_converter.tag( text )
    
    conll_str = layer_to_conll( text, ud_converter.output_layer, preserve_ambiguity=True, serialize=True )
    
    expected_conll_str = '''1	Rändur	rändur	NOUN	S	Number=Sing|Case=Nom	_	_	_	_
2	peaks	pidama	VERB	V	Voice=Act|Tense=Pres|Mood=Cnd|VerbForm=Fin	_	_	_	_
2	peaks	pidama	AUX	V	Voice=Act|Tense=Pres|Mood=Cnd|VerbForm=Fin	_	_	_	_
3	kohe	kohe	ADV	D	_	_	_	_	_
4	saabuma	saabuma	VERB	V	Voice=Act|VerbForm=Sup|Case=Ill	_	_	_	_
5	.	.	PUNCT	Z	_	_	_	_	_

1	Siis	siis	ADV	D	_	_	_	_	_
2	saame	saama	VERB	V	Voice=Act|Tense=Pres|Mood=Ind|VerbForm=Fin|Number=Plur|Person=1	_	_	_	_
2	saame	saama	AUX	V	Voice=Act|Tense=Pres|Mood=Ind|VerbForm=Fin|Number=Plur|Person=1	_	_	_	_
3	asjas	asi	NOUN	S	Number=Sing|Case=Ine	_	_	_	_
4	selgust	selgus	NOUN	S	Number=Sing|Case=Par	_	_	_	_
5	.	.	PUNCT	Z	_	_	_	_	_

'''
    assert conll_str == expected_conll_str