# -*- coding: utf-8 -*-
# Copyright 2024 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from typing import Callable, Dict, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import gapic_v1, grpc_helpers
import google.auth  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.cloud.location import locations_pb2  # type: ignore
from google.longrunning import operations_pb2  # type: ignore
from google.protobuf import empty_pb2  # type: ignore
import grpc  # type: ignore

from google.cloud.discoveryengine_v1.types import conversation as gcd_conversation
from google.cloud.discoveryengine_v1.types import answer
from google.cloud.discoveryengine_v1.types import conversation
from google.cloud.discoveryengine_v1.types import conversational_search_service
from google.cloud.discoveryengine_v1.types import session
from google.cloud.discoveryengine_v1.types import session as gcd_session

from .base import DEFAULT_CLIENT_INFO, ConversationalSearchServiceTransport


class ConversationalSearchServiceGrpcTransport(ConversationalSearchServiceTransport):
    """gRPC backend transport for ConversationalSearchService.

    Service for conversational search.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _stubs: Dict[str, Callable]

    def __init__(
        self,
        *,
        host: str = "discoveryengine.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        channel: Optional[Union[grpc.Channel, Callable[..., grpc.Channel]]] = None,
        api_mtls_endpoint: Optional[str] = None,
        client_cert_source: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        ssl_channel_credentials: Optional[grpc.ChannelCredentials] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to (default: 'discoveryengine.googleapis.com').
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if a ``channel`` instance is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if a ``channel`` instance is provided.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if a ``channel`` instance is provided.
            channel (Optional[Union[grpc.Channel, Callable[..., grpc.Channel]]]):
                A ``Channel`` instance through which to make calls, or a Callable
                that constructs and returns one. If set to None, ``self.create_channel``
                is used to create the channel. If a Callable is given, it will be called
                with the same arguments as used in ``self.create_channel``.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if a ``channel`` instance is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if a ``channel`` instance or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
          google.auth.exceptions.MutualTLSChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if isinstance(channel, grpc.Channel):
            # Ignore credentials if a channel was passed.
            credentials = None
            self._ignore_credentials = True
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None

        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            # initialize with the provided callable or the default channel
            channel_init = channel or type(self).create_channel
            self._grpc_channel = channel_init(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @classmethod
    def create_channel(
        cls,
        host: str = "discoveryengine.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> grpc.Channel:
        """Create and return a gRPC channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is mutually exclusive with credentials.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            grpc.Channel: A gRPC channel object.

        Raises:
            google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """

        return grpc_helpers.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    @property
    def grpc_channel(self) -> grpc.Channel:
        """Return the channel designed to connect to this service."""
        return self._grpc_channel

    @property
    def converse_conversation(
        self,
    ) -> Callable[
        [conversational_search_service.ConverseConversationRequest],
        conversational_search_service.ConverseConversationResponse,
    ]:
        r"""Return a callable for the converse conversation method over gRPC.

        Converses a conversation.

        Returns:
            Callable[[~.ConverseConversationRequest],
                    ~.ConverseConversationResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "converse_conversation" not in self._stubs:
            self._stubs["converse_conversation"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/ConverseConversation",
                request_serializer=conversational_search_service.ConverseConversationRequest.serialize,
                response_deserializer=conversational_search_service.ConverseConversationResponse.deserialize,
            )
        return self._stubs["converse_conversation"]

    @property
    def create_conversation(
        self,
    ) -> Callable[
        [conversational_search_service.CreateConversationRequest],
        gcd_conversation.Conversation,
    ]:
        r"""Return a callable for the create conversation method over gRPC.

        Creates a Conversation.

        If the
        [Conversation][google.cloud.discoveryengine.v1.Conversation] to
        create already exists, an ALREADY_EXISTS error is returned.

        Returns:
            Callable[[~.CreateConversationRequest],
                    ~.Conversation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_conversation" not in self._stubs:
            self._stubs["create_conversation"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/CreateConversation",
                request_serializer=conversational_search_service.CreateConversationRequest.serialize,
                response_deserializer=gcd_conversation.Conversation.deserialize,
            )
        return self._stubs["create_conversation"]

    @property
    def delete_conversation(
        self,
    ) -> Callable[
        [conversational_search_service.DeleteConversationRequest], empty_pb2.Empty
    ]:
        r"""Return a callable for the delete conversation method over gRPC.

        Deletes a Conversation.

        If the
        [Conversation][google.cloud.discoveryengine.v1.Conversation] to
        delete does not exist, a NOT_FOUND error is returned.

        Returns:
            Callable[[~.DeleteConversationRequest],
                    ~.Empty]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_conversation" not in self._stubs:
            self._stubs["delete_conversation"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/DeleteConversation",
                request_serializer=conversational_search_service.DeleteConversationRequest.serialize,
                response_deserializer=empty_pb2.Empty.FromString,
            )
        return self._stubs["delete_conversation"]

    @property
    def update_conversation(
        self,
    ) -> Callable[
        [conversational_search_service.UpdateConversationRequest],
        gcd_conversation.Conversation,
    ]:
        r"""Return a callable for the update conversation method over gRPC.

        Updates a Conversation.

        [Conversation][google.cloud.discoveryengine.v1.Conversation]
        action type cannot be changed. If the
        [Conversation][google.cloud.discoveryengine.v1.Conversation] to
        update does not exist, a NOT_FOUND error is returned.

        Returns:
            Callable[[~.UpdateConversationRequest],
                    ~.Conversation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_conversation" not in self._stubs:
            self._stubs["update_conversation"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/UpdateConversation",
                request_serializer=conversational_search_service.UpdateConversationRequest.serialize,
                response_deserializer=gcd_conversation.Conversation.deserialize,
            )
        return self._stubs["update_conversation"]

    @property
    def get_conversation(
        self,
    ) -> Callable[
        [conversational_search_service.GetConversationRequest],
        conversation.Conversation,
    ]:
        r"""Return a callable for the get conversation method over gRPC.

        Gets a Conversation.

        Returns:
            Callable[[~.GetConversationRequest],
                    ~.Conversation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_conversation" not in self._stubs:
            self._stubs["get_conversation"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/GetConversation",
                request_serializer=conversational_search_service.GetConversationRequest.serialize,
                response_deserializer=conversation.Conversation.deserialize,
            )
        return self._stubs["get_conversation"]

    @property
    def list_conversations(
        self,
    ) -> Callable[
        [conversational_search_service.ListConversationsRequest],
        conversational_search_service.ListConversationsResponse,
    ]:
        r"""Return a callable for the list conversations method over gRPC.

        Lists all Conversations by their parent
        [DataStore][google.cloud.discoveryengine.v1.DataStore].

        Returns:
            Callable[[~.ListConversationsRequest],
                    ~.ListConversationsResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_conversations" not in self._stubs:
            self._stubs["list_conversations"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/ListConversations",
                request_serializer=conversational_search_service.ListConversationsRequest.serialize,
                response_deserializer=conversational_search_service.ListConversationsResponse.deserialize,
            )
        return self._stubs["list_conversations"]

    @property
    def answer_query(
        self,
    ) -> Callable[
        [conversational_search_service.AnswerQueryRequest],
        conversational_search_service.AnswerQueryResponse,
    ]:
        r"""Return a callable for the answer query method over gRPC.

        Answer query method.

        Returns:
            Callable[[~.AnswerQueryRequest],
                    ~.AnswerQueryResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "answer_query" not in self._stubs:
            self._stubs["answer_query"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/AnswerQuery",
                request_serializer=conversational_search_service.AnswerQueryRequest.serialize,
                response_deserializer=conversational_search_service.AnswerQueryResponse.deserialize,
            )
        return self._stubs["answer_query"]

    @property
    def get_answer(
        self,
    ) -> Callable[[conversational_search_service.GetAnswerRequest], answer.Answer]:
        r"""Return a callable for the get answer method over gRPC.

        Gets a Answer.

        Returns:
            Callable[[~.GetAnswerRequest],
                    ~.Answer]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_answer" not in self._stubs:
            self._stubs["get_answer"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/GetAnswer",
                request_serializer=conversational_search_service.GetAnswerRequest.serialize,
                response_deserializer=answer.Answer.deserialize,
            )
        return self._stubs["get_answer"]

    @property
    def create_session(
        self,
    ) -> Callable[
        [conversational_search_service.CreateSessionRequest], gcd_session.Session
    ]:
        r"""Return a callable for the create session method over gRPC.

        Creates a Session.

        If the [Session][google.cloud.discoveryengine.v1.Session] to
        create already exists, an ALREADY_EXISTS error is returned.

        Returns:
            Callable[[~.CreateSessionRequest],
                    ~.Session]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_session" not in self._stubs:
            self._stubs["create_session"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/CreateSession",
                request_serializer=conversational_search_service.CreateSessionRequest.serialize,
                response_deserializer=gcd_session.Session.deserialize,
            )
        return self._stubs["create_session"]

    @property
    def delete_session(
        self,
    ) -> Callable[
        [conversational_search_service.DeleteSessionRequest], empty_pb2.Empty
    ]:
        r"""Return a callable for the delete session method over gRPC.

        Deletes a Session.

        If the [Session][google.cloud.discoveryengine.v1.Session] to
        delete does not exist, a NOT_FOUND error is returned.

        Returns:
            Callable[[~.DeleteSessionRequest],
                    ~.Empty]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_session" not in self._stubs:
            self._stubs["delete_session"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/DeleteSession",
                request_serializer=conversational_search_service.DeleteSessionRequest.serialize,
                response_deserializer=empty_pb2.Empty.FromString,
            )
        return self._stubs["delete_session"]

    @property
    def update_session(
        self,
    ) -> Callable[
        [conversational_search_service.UpdateSessionRequest], gcd_session.Session
    ]:
        r"""Return a callable for the update session method over gRPC.

        Updates a Session.

        [Session][google.cloud.discoveryengine.v1.Session] action type
        cannot be changed. If the
        [Session][google.cloud.discoveryengine.v1.Session] to update
        does not exist, a NOT_FOUND error is returned.

        Returns:
            Callable[[~.UpdateSessionRequest],
                    ~.Session]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_session" not in self._stubs:
            self._stubs["update_session"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/UpdateSession",
                request_serializer=conversational_search_service.UpdateSessionRequest.serialize,
                response_deserializer=gcd_session.Session.deserialize,
            )
        return self._stubs["update_session"]

    @property
    def get_session(
        self,
    ) -> Callable[[conversational_search_service.GetSessionRequest], session.Session]:
        r"""Return a callable for the get session method over gRPC.

        Gets a Session.

        Returns:
            Callable[[~.GetSessionRequest],
                    ~.Session]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_session" not in self._stubs:
            self._stubs["get_session"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/GetSession",
                request_serializer=conversational_search_service.GetSessionRequest.serialize,
                response_deserializer=session.Session.deserialize,
            )
        return self._stubs["get_session"]

    @property
    def list_sessions(
        self,
    ) -> Callable[
        [conversational_search_service.ListSessionsRequest],
        conversational_search_service.ListSessionsResponse,
    ]:
        r"""Return a callable for the list sessions method over gRPC.

        Lists all Sessions by their parent
        [DataStore][google.cloud.discoveryengine.v1.DataStore].

        Returns:
            Callable[[~.ListSessionsRequest],
                    ~.ListSessionsResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_sessions" not in self._stubs:
            self._stubs["list_sessions"] = self.grpc_channel.unary_unary(
                "/google.cloud.discoveryengine.v1.ConversationalSearchService/ListSessions",
                request_serializer=conversational_search_service.ListSessionsRequest.serialize,
                response_deserializer=conversational_search_service.ListSessionsResponse.deserialize,
            )
        return self._stubs["list_sessions"]

    def close(self):
        self.grpc_channel.close()

    @property
    def cancel_operation(
        self,
    ) -> Callable[[operations_pb2.CancelOperationRequest], None]:
        r"""Return a callable for the cancel_operation method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "cancel_operation" not in self._stubs:
            self._stubs["cancel_operation"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/CancelOperation",
                request_serializer=operations_pb2.CancelOperationRequest.SerializeToString,
                response_deserializer=None,
            )
        return self._stubs["cancel_operation"]

    @property
    def get_operation(
        self,
    ) -> Callable[[operations_pb2.GetOperationRequest], operations_pb2.Operation]:
        r"""Return a callable for the get_operation method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_operation" not in self._stubs:
            self._stubs["get_operation"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/GetOperation",
                request_serializer=operations_pb2.GetOperationRequest.SerializeToString,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["get_operation"]

    @property
    def list_operations(
        self,
    ) -> Callable[
        [operations_pb2.ListOperationsRequest], operations_pb2.ListOperationsResponse
    ]:
        r"""Return a callable for the list_operations method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_operations" not in self._stubs:
            self._stubs["list_operations"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/ListOperations",
                request_serializer=operations_pb2.ListOperationsRequest.SerializeToString,
                response_deserializer=operations_pb2.ListOperationsResponse.FromString,
            )
        return self._stubs["list_operations"]

    @property
    def kind(self) -> str:
        return "grpc"


__all__ = ("ConversationalSearchServiceGrpcTransport",)
