"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyType = exports.Policy = exports.POLICY_DESC_SUFFIX_FROM_FILE = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
exports.POLICY_DESC_SUFFIX_FROM_FILE = ' - Created by CDK';
class PolicyBase extends core_1.Resource {
}
class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope, id, policyId) {
        return Policy.fromPolicyAttributes(scope, id, { policyId });
    }
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope, id, attrs) {
        class Import extends PolicyBase {
            constructor(policyType, policyId) {
                super(scope, id);
                this.policyType = policyType;
                this.policyId = policyId;
            }
        }
        const policyType = attrs.policyType ?? PolicyType.STATIC;
        return new Import(policyType, attrs.policyId);
    }
    /**
     * Create a policy based on a file containing a cedar policy. Best practice would be
     * for the file name to end in `.cedar` but this is not required. Policy is parsed for valid
     * syntax but not validated against schema. In order to validate against schema, use
     * `PolicyStore.addPoliciesFromPath()`
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param props A `StaticPolicyFromFileProps` object.
     */
    static fromFile(scope, id, props) {
        const policyFileContents = fs.readFileSync(props.path).toString();
        (0, cedar_helpers_1.checkParsePolicy)(policyFileContents);
        let relativePath = path.basename(props.path);
        return new Policy(scope, id, {
            definition: {
                static: {
                    statement: policyFileContents,
                    description: props.description || `${relativePath}${exports.POLICY_DESC_SUFFIX_FROM_FILE}`,
                },
            },
            policyStore: props.policyStore,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        // validations
        if (props.definition.static && props.definition.templateLinked) {
            throw new Error('Policy can either be static or templateLinked');
        }
        let definition;
        if (props.definition.static) {
            (0, cedar_helpers_1.checkParsePolicy)(props.definition.static.statement);
            definition = {
                static: {
                    ...props.definition.static,
                    statement: props.definition.static.statement,
                },
            };
        }
        else if (props.definition.templateLinked) {
            definition = {
                templateLinked: {
                    policyTemplateId: props.definition.templateLinked.policyTemplate.policyTemplateId,
                    principal: props.definition.templateLinked.principal,
                    resource: props.definition.templateLinked.resource,
                },
            };
        }
        else {
            throw new Error('Policy must either be static or templateLinked');
        }
        // resource
        this.policy = new aws_verifiedpermissions_1.CfnPolicy(this, id, {
            definition: definition,
            policyStoreId: props.policyStore.policyStoreId,
        });
        // assign construct props
        this.policyId = this.policy.attrPolicyId;
        this.policyType = props.definition.static
            ? PolicyType.STATIC
            : PolicyType.TEMPLATELINKED;
        this.definition = props.definition;
        this.policyStoreId = props.policyStore.policyStoreId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@cdklabs/cdk-verified-permissions.Policy", version: "0.1.0" };
/**
 * PolicyType options
 */
var PolicyType;
(function (PolicyType) {
    PolicyType["STATIC"] = "Static";
    PolicyType["TEMPLATELINKED"] = "TemplateLinked";
})(PolicyType || (exports.PolicyType = PolicyType = {}));
//# sourceMappingURL=data:application/json;base64,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