import argparse

from dataclasses import dataclass


@dataclass
class ExploitConfigInfo:
    server_address: str
    tick_time: int
    thread_number: int
    port_service: int
    name_service: str
    test_mode: bool


def parse_exploit_config() -> ExploitConfigInfo:
    """
    Parse the exploit configuration from command line arguments.

    Returns:
        ExploitConfigInfo: The parsed configuration.
    """
    parser = argparse.ArgumentParser(
        description="Advanced Exploit Manager per CTF Attack/Defense"
    )

    parser.add_argument(
        "-s",
        "--server_address",
        type=str,
        help="Server address in the format <ip>:<port>",
        required=True,
    )
    parser.add_argument(
        "-t",
        "--tick_time",
        type=int,
        help="Time in seconds between each exploiting phase attempt",
        default=120,
    )
    parser.add_argument(
        "-T",
        "--thread_number",
        type=int,
        help="Number of threads to use for exploits",
        default=30,
    )
    parser.add_argument(
        "-p",
        "--port_service",
        type=int,
        help="Port number of the service to exploit",
        required=True,
    )
    parser.add_argument(
        "-n",
        "--name_service",
        type=str,
        help="Name of the service to exploit",
        default="UnknownService",
    )
    parser.add_argument(
        "-x",
        "--test",
        action="store_true",
        help="Run in test mode, exploit ONLY the NOP team",
    )

    args = parser.parse_args()

    return ExploitConfigInfo(
        server_address=args.server_address,
        tick_time=args.tick_time,
        thread_number=args.thread_number,
        port_service=args.port_service,
        name_service=args.name_service,
        test_mode=args.test if args.test else False,
    )
