import re
import time

from threading import Lock
from typing import Any, Callable
from itertools import count
from concurrent.futures import ThreadPoolExecutor, as_completed

from .logger import StatusCode, log_stats, log_status
from .flagids_handler import FlagIdsHandler

class ExploitExecutor:
    def __init__(
        self,
        exploit_function: Callable,
        teams: dict[int, str],
        port_service: int,
        name_service: str,
        flag_regex: re.Pattern,
        url_flag_ids: str,
        tick_time: int = 120,
        thread_number: int = 30,
    ):
        self.exploit_function = exploit_function
        self.teams = teams
        self.port_service = port_service
        self.name_service = name_service
        self.flag_regex = flag_regex
        self.tick_time = tick_time
        self.thread_number = thread_number
        self.handler_flag_ids = FlagIdsHandler(
            url=url_flag_ids,
            name_service=name_service
        )
        self.locker = Lock()
        self.stat = {
            "success_team": 0,
            "failed_team": 0,
            "total_flag": 0,
        }


    def __extract_flag(self, source: str | list[str]) -> set[str]:
        if isinstance(source, list):
            if len(source) == 0 or any(isinstance(item, str) for item in source):
                return set()

            return set(self.flag_regex.findall("\n".join(source)))

        elif isinstance(source, str):
            return set(self.flag_regex.findall(source))

        return set()


    def __single_exploit_execution(self, team_id: int, ip: str, flag_ids: list[dict[str, Any]]):
        try:
            result = self.exploit_function(ip, self.port_service, self.name_service, flag_ids)
            flags = self.__extract_flag(result)

            with self.locker:
                if flags:
                    self.stat["success_team"] += 1
                    self.stat["total_flag"] += len(flags)
                else:
                    self.stat["failed_team"] += 1

            return flags

        except Exception as e:
            log_status(
                StatusCode.ERROR,
                f"Error while exploiting team {team_id} ({ip}:{self.port_service}): {str(e)}",
                team_id=team_id,
                port_service=self.port_service,
                name_service=self.name_service,
            )
            return None


    def get_stat(self) -> dict[str, int]:
        """
        Returns the current statistics of the exploit execution.

        Returns:
            dict[str, int]: A dictionary containing the statistics.
        """
        with self.locker:

            current_stat = self.stat.copy()
            self.stat = {
                "success_team": 0,
                "failed_team": 0,
                "total_flag": 0,
            }

            return current_stat


    def __process_exploits(self):
        """
        Process the exploits for all teams concurrently.
        """

        with ThreadPoolExecutor(
            max_workers=min(self.thread_number, len(self.teams)),
            thread_name_prefix="exploit",
        ) as executor:

            flag_ids = self.handler_flag_ids.get_flag_ids()

            futures_to_team = {
                executor.submit(
                    self.__single_exploit_execution,
                    team_id,
                    ip,
                    flag_ids[team_id]
                ): team_id
                for team_id, ip in self.teams.items()
            }

            for future in as_completed(futures_to_team):
                team_id = futures_to_team[future]
                try:
                    flags = future.result()

                    if flags:
                        for flag in flags:
                            log_status(
                                StatusCode.SUCCESS,
                                f"Flag found for team {team_id}",
                                team_id=team_id,
                                port_service=self.port_service,
                                flag_code=flag,
                                name_service=self.name_service,
                            )

                except Exception as e:
                    log_status(
                        StatusCode.ERROR,
                        f"Error while processing team {team_id}: {str(e)}",
                        team_id=team_id,
                        port_service=self.port_service,
                        name_service=self.name_service,
                    )


    def execute(self):
        round = count(1)
        while True:
            start_time = time.time()

            self.__process_exploits()

            log_stats(
                f"Round {next(round)} - statistics",
                port_service=self.port_service,
                name_service=self.name_service,
                stats=self.get_stat(),
            )

            elapsed_time = time.time() - start_time
            time.sleep(max(0, self.tick_time - elapsed_time))
