"""
Exploit Manager
This module manages the exploit process by handling authentication,
configuration retrieval, and dispatching exploits to the target IP teams.

"""

import io
import sys

from threading import local
from .server_config import ServerConfig
from .test_executor import TestExploitExecutor
from .exploit_executor import ExploitExecutor
from .exploit_config import parse_exploit_config

thread_local = local()

# ty 0xMatte for the ThreadSafe <3
class ThreadSafeStdout:
    """
    A thread-safe wrapper for standard output to ensure that
    """

    def write(self, text):
        return self._get_buffer().write(text)

    def _get_buffer(self):
        return getattr(thread_local, "buf", sys.__stdout__)

    def __getattr__(self, attr):
        return getattr(self._get_buffer(), attr)


# Redirect standard output to the thread-local buffer
sys.stdout = ThreadSafeStdout()

def exploit_manager(func):

    def wrapper(*_args, **kwargs):

        def exploit_function_wrapper(*args, **kwargs):
            buf = io.StringIO()
            thread_local.buf = buf

            try:
                func(*args, **kwargs)
            finally:
                del thread_local.buf
                out = buf.getvalue()
                buf.close()
                return out


        exploit_config = parse_exploit_config()
        server_config = ServerConfig(
            server_address=exploit_config.server_address
        ).config()

        try:
            if exploit_config.test_mode:
                TestExploitExecutor(
                    exploit_function=exploit_function_wrapper,
                    port_service=exploit_config.port_service,
                    name_service=exploit_config.name_service,
                    flag_regex=server_config.regex_flag,
                    nop_team=server_config.nop_team,
                    url_flag_ids=server_config.url_flag_ids,
                ).execute()
            else:
                ExploitExecutor(
                    exploit_function=exploit_function_wrapper,
                    flag_regex=server_config.regex_flag,
                    teams=server_config.teams,
                    port_service=exploit_config.port_service,
                    name_service=exploit_config.name_service,
                    tick_time=exploit_config.tick_time,
                    thread_number=exploit_config.thread_number,
                    url_flag_ids=server_config.url_flag_ids,
                ).execute()
        except KeyboardInterrupt:
            sys.exit(0)

    return wrapper()
