"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *     lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        props.retryAttempts !== undefined && cdk.withResolved(props.retryAttempts, (attempts) => {
            if (attempts < 0 || attempts > 10000) {
                throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${attempts}`);
            }
        });
        props.parallelizationFactor !== undefined && cdk.withResolved(props.parallelizationFactor, (factor) => {
            if (factor < 1 || factor > 10) {
                throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${factor}`);
            }
        });
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
            topics: props.kafkaTopic !== undefined ? [props.kafkaTopic] : undefined,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading.
 */
var StartingPosition;
(function (StartingPosition) {
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,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