"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LayerVersion = void 0;
const core_1 = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    /**
     * Add permission for this layer version to specific entities.
     *
     * Usage within
     * the same account where the layer is defined is always allowed and does not
     * require calling this method. Note that the principal that creates the
     * Lambda function using the layer (for example, a CloudFormation changeset
     * execution role) also needs to have the ``lambda:GetLayerVersion``
     * permission on the layer version.
     */
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * Imports a layer version by ARN.
     *
     * Assumes it is compatible with all Lambda runtimes.
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
//# sourceMappingURL=data:application/json;base64,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