import * as iam from '@aws-cdk/aws-iam';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { FunctionProps } from './function';
import { FunctionBase } from './function-base';
import { Version } from './lambda-version';
import { Permission } from './permission';
/**
 * Properties for a newly created singleton Lambda.
 */
export interface SingletonFunctionProps extends FunctionProps {
    /**
     * A unique identifier to identify this lambda.
     *
     * The identifier should be unique across all custom resource providers.
     * We recommend generating a UUID per provider.
     */
    readonly uuid: string;
    /**
     * A descriptive name for the purpose of this Lambda.
     *
     * If the Lambda does not have a physical name, this string will be
     * reflected its generated name. The combination of lambdaPurpose
     * and uuid must be unique.
     *
     * @default SingletonLambda
     */
    readonly lambdaPurpose?: string;
}
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
export declare class SingletonFunction extends FunctionBase {
    /**
     * The principal this Lambda Function is running as.
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The name of the function.
     */
    readonly functionName: string;
    /**
     * The ARN fo the function.
     */
    readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     */
    readonly role?: iam.IRole;
    /**
     * The construct node where permissions are attached.
     */
    readonly permissionsNode: cdk.ConstructNode;
    /**
     * Whether the addPermission() call adds any permissions.
     *
     * True for new Lambdas, false for version $LATEST and imported Lambdas
     * from different accounts.
     */
    protected readonly canCreatePermissions: boolean;
    private lambdaFunction;
    /**
     *
     */
    constructor(scope: Construct, id: string, props: SingletonFunctionProps);
    /**
     * Returns a `lambda.Version` which represents the current version of this singleton Lambda function. A new version will be created every time the function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.SingletonFunction`.
     */
    get currentVersion(): Version;
    /**
     * Adds a permission to the Lambda resource policy.
     */
    addPermission(name: string, permission: Permission): void;
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up: cdk.IDependable[]): void;
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down: cdk.IConstruct): void;
    private ensureLambda;
}
