"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'can add same singleton Lambda multiple times, only instantiated once in template'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        for (let i = 0; i < 5; i++) {
            new lambda.SingletonFunction(stack, `Singleton${i}`, {
                uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
                code: new lambda.InlineCode('def hello(): pass'),
                runtime: lambda.Runtime.PYTHON_2_7,
                handler: 'index.hello',
                timeout: cdk.Duration.minutes(5),
            });
        }
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'lambda.amazonaws.com' },
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']],
                            },
                        ],
                    },
                },
                SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: {
                            ZipFile: 'def hello(): pass',
                        },
                        Handler: 'index.hello',
                        Role: { 'Fn::GetAtt': ['SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235', 'Arn'] },
                        Runtime: 'python2.7',
                        Timeout: 300,
                    },
                    DependsOn: ['SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235'],
                },
            },
        }));
        test.done();
    },
    'dependencies are correctly added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        const dependency = new iam.User(stack, 'dependencyUser');
        // WHEN
        singleton.addDependency(dependency);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            DependsOn: [
                'dependencyUser1B9CB07E',
                'SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'dependsOn are correctly added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        singleton.dependOn(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::User', {
            DependsOn: [
                'SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38',
                'SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grantInvoke works correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
        });
        // WHEN
        const invokeResult = singleton.grantInvoke(new iam.ServicePrincipal('events.amazonaws.com'));
        const statement = stack.resolve(invokeResult.resourceStatement);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'events.amazonaws.com',
        }));
        test.deepEqual(statement.action, ['lambda:InvokeFunction']);
        test.deepEqual(statement.principal, { Service: ['events.amazonaws.com'] });
        test.deepEqual(statement.effect, 'Allow');
        test.deepEqual(statement.resource, [{
                'Fn::GetAtt': ['SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38', 'Arn'],
            }]);
        test.done();
    },
    'check edge compatibility'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            environment: {
                KEY: 'value',
            },
        });
        // THEN
        test.throws(() => singleton._checkEdgeCompatibility(), /The function Default\/SingletonLambda84c0de93353f42179b0b45b6c993251a contains environment variables \[KEY\] and is not compatible with Lambda@Edge/);
        test.done();
    },
    'current version of a singleton function'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        // WHEN
        const version = singleton.currentVersion;
        version.addAlias('foo');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Version', {
            FunctionName: {
                Ref: 'SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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