"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const assets_1 = require("./assets");
const fs_1 = require("./fs");
const stage_1 = require("./stage");
const construct_compat_1 = require("./construct-compat");
const STAGING_TMP = '.cdk.staging';
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
        const outdir = (_c = stage_1.Stage.of(this)) === null || _c === void 0 ? void 0 : _c.outdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.stageAsset(outdir);
    }
    stageAsset(outdir) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Move bundling directory to staging directory
            fs.moveSync(this.bundleDir, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Temp staging directory in the working directory
        const stagingTmp = path.join('.', STAGING_TMP);
        fs.ensureDirSync(stagingTmp);
        // Create temp directory for bundling inside the temp staging directory
        const bundleDir = path.resolve(fs.mkdtempSync(path.join(stagingTmp, 'asset-bundle-')));
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            options.image._run({
                command: options.command,
                user,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : AssetStaging.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${AssetStaging.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                // Hash the hash to make sure we can use it in a file/directory name.
                // The resulting hash will also have the same length as for the other hash types.
                return crypto.createHash('sha256').update(props.assetHash).digest('hex');
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXNzZXQtc3RhZ2luZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFzc2V0LXN0YWdpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsaUNBQWlDO0FBQ2pDLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFDN0IseUNBQXlDO0FBQ3pDLCtCQUErQjtBQUMvQixxQ0FBdUQ7QUFFdkQsNkJBQXNEO0FBQ3RELG1DQUFnQztBQUNoQyx5REFBK0M7QUFFL0MsTUFBTSxXQUFXLEdBQUcsY0FBYyxDQUFDO0FBWW5DOzs7Ozs7Ozs7Ozs7Ozs7OztHQWlCRztBQUNILE1BQWEsWUFBYSxTQUFRLDRCQUFTO0lBNEN6QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXdCOztRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDO1FBRWhDLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNsQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzlDO1FBRUQsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTNDLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3JGLElBQUksZUFBZSxFQUFFO1lBQ25CLElBQUksQ0FBQyxVQUFVLFNBQUcsSUFBSSxDQUFDLFNBQVMsbUNBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQztTQUNyRDthQUFNO1lBQ0wsSUFBSSxDQUFDLFlBQVksR0FBRyxTQUFTLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sT0FBQyxJQUFJLENBQUMsU0FBUyxtQ0FBSSxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQztZQUNoRyxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUM7U0FDckM7UUFFRCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7UUFFakMsTUFBTSxNQUFNLFNBQUcsYUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsMENBQUUsTUFBTSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFDLDJIQUEySCxDQUFDLENBQUM7U0FDOUk7UUFFRCxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzFCLENBQUM7SUFFTyxVQUFVLENBQUMsTUFBYztRQUMvQixzQkFBc0I7UUFDdEIsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDdEIsT0FBTztTQUNSO1FBRUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRXhELGlCQUFpQjtRQUNqQixJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDN0IsT0FBTztTQUNSO1FBRUQseUJBQXlCO1FBQ3pCLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQiwrQ0FBK0M7WUFDL0MsRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBQ3hDLE9BQU87U0FDUjtRQUVELDJDQUEyQztRQUMzQyxNQUFNLElBQUksR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMxQyxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNqQixFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7U0FDOUM7YUFBTSxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUM3QixFQUFFLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3pCLGVBQVUsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxVQUFVLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUM7U0FDaEY7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsc0JBQXNCLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQzFEO0lBQ0gsQ0FBQztJQUVPLE1BQU0sQ0FBQyxPQUF3Qjs7UUFDckMsa0RBQWtEO1FBQ2xELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQy9DLEVBQUUsQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFN0IsdUVBQXVFO1FBQ3ZFLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdkYsc0NBQXNDO1FBQ3RDLEVBQUUsQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRS9CLElBQUksSUFBWSxDQUFDO1FBQ2pCLElBQUksT0FBTyxDQUFDLElBQUksRUFBRTtZQUNoQixJQUFJLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztTQUNyQjthQUFNLEVBQUUsMEJBQTBCO1lBQ2pDLE1BQU0sUUFBUSxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUMvQixJQUFJLEdBQUcsUUFBUSxDQUFDLEdBQUcsS0FBSyxDQUFDLENBQUMsQ0FBQyx1QkFBdUI7Z0JBQ2hELENBQUMsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxHQUFHLElBQUksUUFBUSxDQUFDLEdBQUcsRUFBRTtnQkFDbkMsQ0FBQyxDQUFDLFdBQVcsQ0FBQztTQUNqQjtRQUVELG9DQUFvQztRQUNwQyxNQUFNLE9BQU8sR0FBRztZQUNkO2dCQUNFLFFBQVEsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDekIsYUFBYSxFQUFFLFlBQVksQ0FBQyxrQkFBa0I7YUFDL0M7WUFDRDtnQkFDRSxRQUFRLEVBQUUsU0FBUztnQkFDbkIsYUFBYSxFQUFFLFlBQVksQ0FBQyxtQkFBbUI7YUFDaEQ7WUFDRCxTQUFHLE9BQU8sQ0FBQyxPQUFPLG1DQUFJLEVBQUU7U0FDekIsQ0FBQztRQUVGLElBQUk7WUFDRixPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLE9BQU8sQ0FBQyxDQUFDO1lBQzlELE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO2dCQUNqQixPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU87Z0JBQ3hCLElBQUk7Z0JBQ0osT0FBTztnQkFDUCxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVc7Z0JBQ2hDLGdCQUFnQixRQUFFLE9BQU8sQ0FBQyxnQkFBZ0IsbUNBQUksWUFBWSxDQUFDLGtCQUFrQjthQUM5RSxDQUFDLENBQUM7U0FDSjtRQUFDLE9BQU8sR0FBRyxFQUFFO1lBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssR0FBRyxFQUFFLENBQUMsQ0FBQztTQUM1RjtRQUVELElBQUksZUFBVSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLG9GQUFvRixZQUFZLENBQUMsbUJBQW1CLEdBQUcsQ0FBQyxDQUFDO1NBQzFJO1FBRUQsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVPLGFBQWEsQ0FBQyxLQUF3QjtRQUM1QyxJQUFJLFFBQXVCLENBQUM7UUFFNUIsSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQ25CLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsYUFBYSxLQUFLLHNCQUFhLENBQUMsTUFBTSxFQUFFO2dCQUN2RSxNQUFNLElBQUksS0FBSyxDQUFDLG9CQUFvQixLQUFLLENBQUMsYUFBYSxrR0FBa0csQ0FBQyxDQUFDO2FBQzVKO1lBQ0QsUUFBUSxHQUFHLHNCQUFhLENBQUMsTUFBTSxDQUFDO1NBQ2pDO2FBQU0sSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQzlCLFFBQVEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1NBQ2hDO2FBQU07WUFDTCxRQUFRLEdBQUcsc0JBQWEsQ0FBQyxNQUFNLENBQUM7U0FDakM7UUFFRCxRQUFRLFFBQVEsRUFBRTtZQUNoQixLQUFLLHNCQUFhLENBQUMsTUFBTTtnQkFDdkIsT0FBTyxlQUFVLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUM7WUFDMUUsS0FBSyxzQkFBYSxDQUFDLE1BQU07Z0JBQ3ZCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFO29CQUNuQixNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7aUJBQ3hGO2dCQUNELE9BQU8sZUFBVSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1lBQ3pFLEtBQUssc0JBQWEsQ0FBQyxNQUFNO2dCQUN2QixJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRTtvQkFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRkFBc0YsQ0FBQyxDQUFDO2lCQUN6RztnQkFDRCxxRUFBcUU7Z0JBQ3JFLGlGQUFpRjtnQkFDakYsT0FBTyxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzNFO2dCQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsMEJBQTBCLENBQUMsQ0FBQztTQUMvQztJQUNILENBQUM7O0FBL0xILG9DQWdNQztBQS9MQzs7O0dBR0c7QUFDb0IsK0JBQWtCLEdBQUcsY0FBYyxDQUFDO0FBRTNEOzs7R0FHRztBQUNvQixnQ0FBbUIsR0FBRyxlQUFlLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tICdAYXdzLWNkay9jeC1hcGknO1xuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMtZXh0cmEnO1xuaW1wb3J0IHsgQXNzZXRIYXNoVHlwZSwgQXNzZXRPcHRpb25zIH0gZnJvbSAnLi9hc3NldHMnO1xuaW1wb3J0IHsgQnVuZGxpbmdPcHRpb25zIH0gZnJvbSAnLi9idW5kbGluZyc7XG5pbXBvcnQgeyBGaWxlU3lzdGVtLCBGaW5nZXJwcmludE9wdGlvbnMgfSBmcm9tICcuL2ZzJztcbmltcG9ydCB7IFN0YWdlIH0gZnJvbSAnLi9zdGFnZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuL2NvbnN0cnVjdC1jb21wYXQnO1xuXG5jb25zdCBTVEFHSU5HX1RNUCA9ICcuY2RrLnN0YWdpbmcnO1xuXG4vKipcbiAqIEluaXRpYWxpemF0aW9uIHByb3BlcnRpZXMgZm9yIGBBc3NldFN0YWdpbmdgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFzc2V0U3RhZ2luZ1Byb3BzIGV4dGVuZHMgRmluZ2VycHJpbnRPcHRpb25zLCBBc3NldE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHNvdXJjZSBmaWxlIG9yIGRpcmVjdG9yeSB0byBjb3B5IGZyb20uXG4gICAqL1xuICByZWFkb25seSBzb3VyY2VQYXRoOiBzdHJpbmc7XG59XG5cbi8qKlxuICogU3RhZ2VzIGEgZmlsZSBvciBkaXJlY3RvcnkgZnJvbSBhIGxvY2F0aW9uIG9uIHRoZSBmaWxlIHN5c3RlbSBpbnRvIGEgc3RhZ2luZ1xuICogZGlyZWN0b3J5LlxuICpcbiAqIFRoaXMgaXMgY29udHJvbGxlZCBieSB0aGUgY29udGV4dCBrZXkgJ2F3czpjZGs6YXNzZXQtc3RhZ2luZycgYW5kIGVuYWJsZWRcbiAqIGJ5IHRoZSBDTEkgYnkgZGVmYXVsdCBpbiBvcmRlciB0byBlbnN1cmUgdGhhdCB3aGVuIHRoZSBDREsgYXBwIGV4aXN0cywgYWxsXG4gKiBhc3NldHMgYXJlIGF2YWlsYWJsZSBmb3IgZGVwbG95bWVudC4gT3RoZXJ3aXNlLCBpZiBhbiBhcHAgcmVmZXJlbmNlcyBhc3NldHNcbiAqIGluIHRlbXBvcmFyeSBsb2NhdGlvbnMsIHRob3NlIHdpbGwgbm90IGJlIGF2YWlsYWJsZSB3aGVuIGl0IGV4aXN0cyAoc2VlXG4gKiBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzE3MTYpLlxuICpcbiAqIFRoZSBgc3RhZ2VkUGF0aGAgcHJvcGVydHkgaXMgYSBzdHJpbmdpZmllZCB0b2tlbiB0aGF0IHJlcHJlc2VudHMgdGhlIGxvY2F0aW9uXG4gKiBvZiB0aGUgZmlsZSBvciBkaXJlY3RvcnkgYWZ0ZXIgc3RhZ2luZy4gSXQgd2lsbCBiZSByZXNvbHZlZCBvbmx5IGR1cmluZyB0aGVcbiAqIFwicHJlcGFyZVwiIHN0YWdlIGFuZCBtYXkgYmUgZWl0aGVyIHRoZSBvcmlnaW5hbCBwYXRoIG9yIHRoZSBzdGFnZWQgcGF0aFxuICogZGVwZW5kaW5nIG9uIHRoZSBjb250ZXh0IHNldHRpbmcuXG4gKlxuICogVGhlIGZpbGUvZGlyZWN0b3J5IGFyZSBzdGFnZWQgYmFzZWQgb24gdGhlaXIgY29udGVudCBoYXNoIChmaW5nZXJwcmludCkuIFRoaXNcbiAqIG1lYW5zIHRoYXQgb25seSBpZiBjb250ZW50IHdhcyBjaGFuZ2VkLCBjb3B5IHdpbGwgaGFwcGVuLlxuICovXG5leHBvcnQgY2xhc3MgQXNzZXRTdGFnaW5nIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFRoZSBkaXJlY3RvcnkgaW5zaWRlIHRoZSBidW5kbGluZyBjb250YWluZXIgaW50byB3aGljaCB0aGUgYXNzZXQgc291cmNlcyB3aWxsIGJlIG1vdW50ZWQuXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQlVORExJTkdfSU5QVVRfRElSID0gJy9hc3NldC1pbnB1dCc7XG5cbiAgLyoqXG4gICAqIFRoZSBkaXJlY3RvcnkgaW5zaWRlIHRoZSBidW5kbGluZyBjb250YWluZXIgaW50byB3aGljaCB0aGUgYnVuZGxlZCBvdXRwdXQgc2hvdWxkIGJlIHdyaXR0ZW4uXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQlVORExJTkdfT1VUUFVUX0RJUiA9ICcvYXNzZXQtb3V0cHV0JztcblxuICAvKipcbiAgICogVGhlIHBhdGggdG8gdGhlIGFzc2V0IChzdHJpbmdpbmZpZWQgdG9rZW4pLlxuICAgKlxuICAgKiBJZiBhc3NldCBzdGFnaW5nIGlzIGRpc2FibGVkLCB0aGlzIHdpbGwganVzdCBiZSB0aGUgb3JpZ2luYWwgcGF0aC5cbiAgICogSWYgYXNzZXQgc3RhZ2luZyBpcyBlbmFibGVkIGl0IHdpbGwgYmUgdGhlIHN0YWdlZCBwYXRoLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN0YWdlZFBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHBhdGggb2YgdGhlIGFzc2V0IGFzIGl0IHdhcyByZWZlcmVuY2VkIGJ5IHRoZSB1c2VyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNvdXJjZVBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogQSBjcnlwdG9ncmFwaGljIGhhc2ggb2YgdGhlIGFzc2V0LlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBzZWUgYGFzc2V0SGFzaGAuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc291cmNlSGFzaDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGNyeXB0b2dyYXBoaWMgaGFzaCBvZiB0aGUgYXNzZXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXNzZXRIYXNoOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBmaW5nZXJwcmludE9wdGlvbnM6IEZpbmdlcnByaW50T3B0aW9ucztcblxuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlUGF0aD86IHN0cmluZztcblxuICBwcml2YXRlIHJlYWRvbmx5IGJ1bmRsZURpcj86IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXNzZXRTdGFnaW5nUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5zb3VyY2VQYXRoID0gcHJvcHMuc291cmNlUGF0aDtcbiAgICB0aGlzLmZpbmdlcnByaW50T3B0aW9ucyA9IHByb3BzO1xuXG4gICAgaWYgKHByb3BzLmJ1bmRsaW5nKSB7XG4gICAgICB0aGlzLmJ1bmRsZURpciA9IHRoaXMuYnVuZGxlKHByb3BzLmJ1bmRsaW5nKTtcbiAgICB9XG5cbiAgICB0aGlzLmFzc2V0SGFzaCA9IHRoaXMuY2FsY3VsYXRlSGFzaChwcm9wcyk7XG5cbiAgICBjb25zdCBzdGFnaW5nRGlzYWJsZWQgPSB0aGlzLm5vZGUudHJ5R2V0Q29udGV4dChjeGFwaS5ESVNBQkxFX0FTU0VUX1NUQUdJTkdfQ09OVEVYVCk7XG4gICAgaWYgKHN0YWdpbmdEaXNhYmxlZCkge1xuICAgICAgdGhpcy5zdGFnZWRQYXRoID0gdGhpcy5idW5kbGVEaXIgPz8gdGhpcy5zb3VyY2VQYXRoO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJlbGF0aXZlUGF0aCA9IGBhc3NldC4ke3RoaXMuYXNzZXRIYXNofSR7cGF0aC5leHRuYW1lKHRoaXMuYnVuZGxlRGlyID8/IHRoaXMuc291cmNlUGF0aCl9YDtcbiAgICAgIHRoaXMuc3RhZ2VkUGF0aCA9IHRoaXMucmVsYXRpdmVQYXRoO1xuICAgIH1cblxuICAgIHRoaXMuc291cmNlSGFzaCA9IHRoaXMuYXNzZXRIYXNoO1xuXG4gICAgY29uc3Qgb3V0ZGlyID0gU3RhZ2Uub2YodGhpcyk/Lm91dGRpcjtcbiAgICBpZiAoIW91dGRpcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCd1bmFibGUgdG8gZGV0ZXJtaW5lIGNsb3VkIGFzc2VtYmx5IG91dHB1dCBkaXJlY3RvcnkuIEFzc2V0cyBtdXN0IGJlIGRlZmluZWQgaW5kaXJlY3RseSB3aXRoaW4gYSBcIlN0YWdlXCIgb3IgYW4gXCJBcHBcIiBzY29wZScpO1xuICAgIH1cblxuICAgIHRoaXMuc3RhZ2VBc3NldChvdXRkaXIpO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGFnZUFzc2V0KG91dGRpcjogc3RyaW5nKSB7XG4gICAgLy8gU3RhZ2luZyBpcyBkaXNhYmxlZFxuICAgIGlmICghdGhpcy5yZWxhdGl2ZVBhdGgpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCB0YXJnZXRQYXRoID0gcGF0aC5qb2luKG91dGRpciwgdGhpcy5yZWxhdGl2ZVBhdGgpO1xuXG4gICAgLy8gQWxyZWFkeSBzdGFnZWRcbiAgICBpZiAoZnMuZXhpc3RzU3luYyh0YXJnZXRQYXRoKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIC8vIEFzc2V0IGhhcyBiZWVuIGJ1bmRsZWRcbiAgICBpZiAodGhpcy5idW5kbGVEaXIpIHtcbiAgICAgIC8vIE1vdmUgYnVuZGxpbmcgZGlyZWN0b3J5IHRvIHN0YWdpbmcgZGlyZWN0b3J5XG4gICAgICBmcy5tb3ZlU3luYyh0aGlzLmJ1bmRsZURpciwgdGFyZ2V0UGF0aCk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgLy8gQ29weSBmaWxlL2RpcmVjdG9yeSB0byBzdGFnaW5nIGRpcmVjdG9yeVxuICAgIGNvbnN0IHN0YXQgPSBmcy5zdGF0U3luYyh0aGlzLnNvdXJjZVBhdGgpO1xuICAgIGlmIChzdGF0LmlzRmlsZSgpKSB7XG4gICAgICBmcy5jb3B5RmlsZVN5bmModGhpcy5zb3VyY2VQYXRoLCB0YXJnZXRQYXRoKTtcbiAgICB9IGVsc2UgaWYgKHN0YXQuaXNEaXJlY3RvcnkoKSkge1xuICAgICAgZnMubWtkaXJTeW5jKHRhcmdldFBhdGgpO1xuICAgICAgRmlsZVN5c3RlbS5jb3B5RGlyZWN0b3J5KHRoaXMuc291cmNlUGF0aCwgdGFyZ2V0UGF0aCwgdGhpcy5maW5nZXJwcmludE9wdGlvbnMpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gZmlsZSB0eXBlOiAke3RoaXMuc291cmNlUGF0aH1gKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGJ1bmRsZShvcHRpb25zOiBCdW5kbGluZ09wdGlvbnMpOiBzdHJpbmcge1xuICAgIC8vIFRlbXAgc3RhZ2luZyBkaXJlY3RvcnkgaW4gdGhlIHdvcmtpbmcgZGlyZWN0b3J5XG4gICAgY29uc3Qgc3RhZ2luZ1RtcCA9IHBhdGguam9pbignLicsIFNUQUdJTkdfVE1QKTtcbiAgICBmcy5lbnN1cmVEaXJTeW5jKHN0YWdpbmdUbXApO1xuXG4gICAgLy8gQ3JlYXRlIHRlbXAgZGlyZWN0b3J5IGZvciBidW5kbGluZyBpbnNpZGUgdGhlIHRlbXAgc3RhZ2luZyBkaXJlY3RvcnlcbiAgICBjb25zdCBidW5kbGVEaXIgPSBwYXRoLnJlc29sdmUoZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKHN0YWdpbmdUbXAsICdhc3NldC1idW5kbGUtJykpKTtcbiAgICAvLyBDaG1vZCB0aGUgYnVuZGxlRGlyIHRvIGZ1bGwgYWNjZXNzLlxuICAgIGZzLmNobW9kU3luYyhidW5kbGVEaXIsIDBvNzc3KTtcblxuICAgIGxldCB1c2VyOiBzdHJpbmc7XG4gICAgaWYgKG9wdGlvbnMudXNlcikge1xuICAgICAgdXNlciA9IG9wdGlvbnMudXNlcjtcbiAgICB9IGVsc2UgeyAvLyBEZWZhdWx0IHRvIGN1cnJlbnQgdXNlclxuICAgICAgY29uc3QgdXNlckluZm8gPSBvcy51c2VySW5mbygpO1xuICAgICAgdXNlciA9IHVzZXJJbmZvLnVpZCAhPT0gLTEgLy8gdWlkIGlzIC0xIG9uIFdpbmRvd3NcbiAgICAgICAgPyBgJHt1c2VySW5mby51aWR9OiR7dXNlckluZm8uZ2lkfWBcbiAgICAgICAgOiAnMTAwMDoxMDAwJztcbiAgICB9XG5cbiAgICAvLyBBbHdheXMgbW91bnQgaW5wdXQgYW5kIG91dHB1dCBkaXJcbiAgICBjb25zdCB2b2x1bWVzID0gW1xuICAgICAge1xuICAgICAgICBob3N0UGF0aDogdGhpcy5zb3VyY2VQYXRoLFxuICAgICAgICBjb250YWluZXJQYXRoOiBBc3NldFN0YWdpbmcuQlVORExJTkdfSU5QVVRfRElSLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgaG9zdFBhdGg6IGJ1bmRsZURpcixcbiAgICAgICAgY29udGFpbmVyUGF0aDogQXNzZXRTdGFnaW5nLkJVTkRMSU5HX09VVFBVVF9ESVIsXG4gICAgICB9LFxuICAgICAgLi4ub3B0aW9ucy52b2x1bWVzID8/IFtdLFxuICAgIF07XG5cbiAgICB0cnkge1xuICAgICAgcHJvY2Vzcy5zdGRlcnIud3JpdGUoYEJ1bmRsaW5nIGFzc2V0ICR7dGhpcy5ub2RlLnBhdGh9Li4uXFxuYCk7XG4gICAgICBvcHRpb25zLmltYWdlLl9ydW4oe1xuICAgICAgICBjb21tYW5kOiBvcHRpb25zLmNvbW1hbmQsXG4gICAgICAgIHVzZXIsXG4gICAgICAgIHZvbHVtZXMsXG4gICAgICAgIGVudmlyb25tZW50OiBvcHRpb25zLmVudmlyb25tZW50LFxuICAgICAgICB3b3JraW5nRGlyZWN0b3J5OiBvcHRpb25zLndvcmtpbmdEaXJlY3RvcnkgPz8gQXNzZXRTdGFnaW5nLkJVTkRMSU5HX0lOUFVUX0RJUixcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBGYWlsZWQgdG8gcnVuIGJ1bmRsaW5nIERvY2tlciBpbWFnZSBmb3IgYXNzZXQgJHt0aGlzLm5vZGUucGF0aH06ICR7ZXJyfWApO1xuICAgIH1cblxuICAgIGlmIChGaWxlU3lzdGVtLmlzRW1wdHkoYnVuZGxlRGlyKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBCdW5kbGluZyBkaWQgbm90IHByb2R1Y2UgYW55IG91dHB1dC4gQ2hlY2sgdGhhdCB5b3VyIGNvbnRhaW5lciB3cml0ZXMgY29udGVudCB0byAke0Fzc2V0U3RhZ2luZy5CVU5ETElOR19PVVRQVVRfRElSfS5gKTtcbiAgICB9XG5cbiAgICByZXR1cm4gYnVuZGxlRGlyO1xuICB9XG5cbiAgcHJpdmF0ZSBjYWxjdWxhdGVIYXNoKHByb3BzOiBBc3NldFN0YWdpbmdQcm9wcyk6IHN0cmluZyB7XG4gICAgbGV0IGhhc2hUeXBlOiBBc3NldEhhc2hUeXBlO1xuXG4gICAgaWYgKHByb3BzLmFzc2V0SGFzaCkge1xuICAgICAgaWYgKHByb3BzLmFzc2V0SGFzaFR5cGUgJiYgcHJvcHMuYXNzZXRIYXNoVHlwZSAhPT0gQXNzZXRIYXNoVHlwZS5DVVNUT00pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3Qgc3BlY2lmeSBcXGAke3Byb3BzLmFzc2V0SGFzaFR5cGV9XFxgIGZvciBcXGBhc3NldEhhc2hUeXBlXFxgIHdoZW4gXFxgYXNzZXRIYXNoXFxgIGlzIHNwZWNpZmllZC4gVXNlIFxcYENVU1RPTVxcYCBvciBsZWF2ZSBcXGB1bmRlZmluZWRcXGAuYCk7XG4gICAgICB9XG4gICAgICBoYXNoVHlwZSA9IEFzc2V0SGFzaFR5cGUuQ1VTVE9NO1xuICAgIH0gZWxzZSBpZiAocHJvcHMuYXNzZXRIYXNoVHlwZSkge1xuICAgICAgaGFzaFR5cGUgPSBwcm9wcy5hc3NldEhhc2hUeXBlO1xuICAgIH0gZWxzZSB7XG4gICAgICBoYXNoVHlwZSA9IEFzc2V0SGFzaFR5cGUuU09VUkNFO1xuICAgIH1cblxuICAgIHN3aXRjaCAoaGFzaFR5cGUpIHtcbiAgICAgIGNhc2UgQXNzZXRIYXNoVHlwZS5TT1VSQ0U6XG4gICAgICAgIHJldHVybiBGaWxlU3lzdGVtLmZpbmdlcnByaW50KHRoaXMuc291cmNlUGF0aCwgdGhpcy5maW5nZXJwcmludE9wdGlvbnMpO1xuICAgICAgY2FzZSBBc3NldEhhc2hUeXBlLkJVTkRMRTpcbiAgICAgICAgaWYgKCF0aGlzLmJ1bmRsZURpcikge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHVzZSBgQXNzZXRIYXNoVHlwZS5CVU5ETEVgIHdoZW4gYGJ1bmRsaW5nYCBpcyBub3Qgc3BlY2lmaWVkLicpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBGaWxlU3lzdGVtLmZpbmdlcnByaW50KHRoaXMuYnVuZGxlRGlyLCB0aGlzLmZpbmdlcnByaW50T3B0aW9ucyk7XG4gICAgICBjYXNlIEFzc2V0SGFzaFR5cGUuQ1VTVE9NOlxuICAgICAgICBpZiAoIXByb3BzLmFzc2V0SGFzaCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignYGFzc2V0SGFzaGAgbXVzdCBiZSBzcGVjaWZpZWQgd2hlbiBgYXNzZXRIYXNoVHlwZWAgaXMgc2V0IHRvIGBBc3NldEhhc2hUeXBlLkNVU1RPTWAuJyk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gSGFzaCB0aGUgaGFzaCB0byBtYWtlIHN1cmUgd2UgY2FuIHVzZSBpdCBpbiBhIGZpbGUvZGlyZWN0b3J5IG5hbWUuXG4gICAgICAgIC8vIFRoZSByZXN1bHRpbmcgaGFzaCB3aWxsIGFsc28gaGF2ZSB0aGUgc2FtZSBsZW5ndGggYXMgZm9yIHRoZSBvdGhlciBoYXNoIHR5cGVzLlxuICAgICAgICByZXR1cm4gY3J5cHRvLmNyZWF0ZUhhc2goJ3NoYTI1NicpLnVwZGF0ZShwcm9wcy5hc3NldEhhc2gpLmRpZ2VzdCgnaGV4Jyk7XG4gICAgICBkZWZhdWx0OlxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Vua25vd24gYXNzZXQgaGFzaCB0eXBlLicpO1xuICAgIH1cbiAgfVxufVxuIl19