"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const finder = this.options.finder;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: policy.EnableVersionUpgrade,
            useOnlineResharding: policy.UseOnlineResharding,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: p.WaitOnResourceSignals,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = specialCaseRefs(refTarget);
                if (specialRef) {
                    return specialRef;
                }
                else {
                    const refElement = this.options.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                // Fn::GetAtt takes a 2-element list as its argument
                const value = object[key];
                const target = this.options.finder.findResource(value[0]);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${value[0]}' not found`);
                }
                return target.getAtt(value[1]);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.findInMap(value[0], value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.options.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.options.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                throw new Error(`Unsupported CloudFormation function '${key}'`);
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = specialCaseSubRefs(refTarget);
        if (specialRef) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.options.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', true).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.options.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, true).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
}
exports.CfnParser = CfnParser;
function specialCaseRefs(value) {
    switch (value) {
        case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
        case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
        case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
        case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
        case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
        case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
        case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
        case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
        default: return undefined;
    }
}
function specialCaseSubRefs(value) {
    return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
}
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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