"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const cxapi = require("@aws-cdk/cx-api");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = scope.node.tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                scope.node.addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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