"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RememberingTokenResolver = exports.findTokens = exports.resolve = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x) { return resolve(x, { ...options, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return options.resolver.resolveString(fragments, makeContext()[0]);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) !== 'string') {
            // eslint-disable-next-line max-len
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}. Consider using "CfnJson" to delay resolution to deployment-time`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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