"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("@aws-cdk/cx-api");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const prepare_app_1 = require("./prepare-app");
const tree_metadata_1 = require("./tree-metadata");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        // hackery to be able to access some private members with strong types (yack!)
        const node = construct.node._actualNode;
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...node._aspects];
        const nodeAspectsCount = node._aspects.length;
        for (const aspect of allAspectsHere) {
            if (node.invokedAspects.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== node._aspects.length) {
                construct.node.addWarning('We detected an Aspect was added via another Aspect, and will not be applied');
                nestedAspectWarning = true;
            }
            node.invokedAspects.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, 'post', construct => {
        const session = {
            outdir: builder.outdir,
            assembly: builder,
        };
        if (construct instanceof stack_1.Stack) {
            construct._synthesizeTemplate(session);
        }
        else if (construct instanceof tree_metadata_1.TreeMetadata) {
            construct._synthesizeTree(session);
        }
        // this will soon be deprecated and removed in 2.x
        // see https://github.com/aws/aws-cdk-rfcs/issues/192
        construct.onSynthesize(session);
    });
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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