"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 * @experimental
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    /**
     * Create tree.json
     * @internal
     */
    _synthesizeTree(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    this.node.addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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