"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const cxapi = require("@aws-cdk/cx-api");
const construct_compat_1 = require("./construct-compat");
const runtime_info_1 = require("./private/runtime-info");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return construct.node.scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * The cloud assembly output directory.
     */
    get outdir() {
        return this._assemblyBuilder.outdir;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly || options.force) {
            const runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? undefined : runtime_info_1.collectRuntimeInformation();
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
                runtimeInfo,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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