"use strict";
const assert = require("assert");
const fs = require("fs");
const os = require("os");
const path = require("path");
const url = require("url");
const entrypoint = require("../../lib/custom-resource-provider/nodejs-entrypoint");
function makeEvent(req) {
    return {
        LogicalResourceId: '<LogicalResourceId>',
        RequestId: '<RequestId>',
        ResourceType: '<ResourceType>',
        ResponseURL: '<ResponseURL>',
        ServiceToken: '<ServiceToken>',
        StackId: '<StackId>',
        ResourceProperties: {
            ServiceToken: '<ServiceToken>',
            ...req.ResourceProperties,
        },
        ...req,
    };
}
async function invokeHandler(req, userHandler) {
    const parsedResponseUrl = url.parse(req.ResponseURL);
    // stage entry point and user handler.
    const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk-custom-resource-provider-handler-test-'));
    entrypoint.external.userHandlerIndex = path.join(workdir, 'index.js');
    fs.writeFileSync(entrypoint.external.userHandlerIndex, `exports.handler = ${userHandler.toString()};`);
    // do not include stack traces in failure responses so we can assert against them.
    entrypoint.external.includeStackTraces = false;
    // disable logging
    entrypoint.external.log = () => {
        return;
    };
    let actualResponse;
    entrypoint.external.sendHttpRequest = async (options, responseBody) => {
        assert(options.hostname === parsedResponseUrl.hostname, 'request hostname expected to be based on response URL');
        assert(options.path === parsedResponseUrl.path, 'request path expected to be based on response URL');
        assert(options.method === 'PUT', 'request method is expected to be PUT');
        actualResponse = responseBody;
    };
    await entrypoint.handler(req);
    if (!actualResponse) {
        throw new Error('no response sent to cloudformation');
    }
    return JSON.parse(actualResponse);
}
module.exports = {
    'handler return value is sent back to cloudformation as a success response': {
        async 'physical resource id (ref)'(test) {
            // GIVEN
            const createEvent = makeEvent({ RequestType: 'Create' });
            // WHEN
            const response = await invokeHandler(createEvent, async (_) => ({ PhysicalResourceId: 'returned-from-handler' }));
            // THEN
            test.deepEqual(response.Status, 'SUCCESS');
            test.deepEqual(response.PhysicalResourceId, 'returned-from-handler');
            test.done();
        },
        async 'data (attributes)'(test) {
            // GIVEN
            const createEvent = makeEvent({ RequestType: 'Create' });
            // WHEN
            const response = await invokeHandler(createEvent, async (_) => {
                return {
                    Data: {
                        Attribute1: 'hello',
                        Attribute2: {
                            Foo: 1111,
                        },
                    },
                };
            });
            // THEN
            test.deepEqual(response.Status, 'SUCCESS');
            test.deepEqual(response.PhysicalResourceId, '<RequestId>', 'physical id defaults to request id');
            test.deepEqual(response.Data, {
                Attribute1: 'hello',
                Attribute2: {
                    Foo: 1111,
                },
            });
            test.done();
        },
        async 'no echo'(test) {
            // GIVEN
            const createEvent = makeEvent({ RequestType: 'Create' });
            // WHEN
            const response = await invokeHandler(createEvent, async (_) => ({ NoEcho: true }));
            // THEN
            test.deepEqual(response.Status, 'SUCCESS');
            test.deepEqual(response.NoEcho, true);
            test.done();
        },
        async 'reason'(test) {
            // GIVEN
            const createEvent = makeEvent({ RequestType: 'Create' });
            // WHEN
            const response = await invokeHandler(createEvent, async (_) => ({ Reason: 'hello, reason' }));
            // THEN
            test.deepEqual(response.Status, 'SUCCESS');
            test.deepEqual(response.Reason, 'hello, reason');
            test.done();
        },
    },
    async 'an error thrown by the handler is sent as a failure response to cloudformation'(test) {
        // GIVEN
        const createEvent = makeEvent({ RequestType: 'Create' });
        // WHEN
        const response = await invokeHandler(createEvent, async (_) => {
            throw new Error('this is an error');
        });
        // THEN
        test.deepEqual(response, {
            Status: 'FAILED',
            Reason: 'this is an error',
            StackId: '<StackId>',
            RequestId: '<RequestId>',
            PhysicalResourceId: 'AWSCDK::CustomResourceProviderFramework::CREATE_FAILED',
            LogicalResourceId: '<LogicalResourceId>',
        });
        test.done();
    },
    async 'physical resource id cannot be changed in DELETE'(test) {
        // GIVEN
        const event = makeEvent({ RequestType: 'Delete' });
        // WHEN
        const response = await invokeHandler(event, async (_) => ({
            PhysicalResourceId: 'Changed',
        }));
        // THEN
        test.deepEqual(response, {
            Status: 'FAILED',
            Reason: 'DELETE: cannot change the physical resource ID from "undefined" to "Changed" during deletion',
            StackId: '<StackId>',
            RequestId: '<RequestId>',
            PhysicalResourceId: 'AWSCDK::CustomResourceProviderFramework::MISSING_PHYSICAL_ID',
            LogicalResourceId: '<LogicalResourceId>',
        });
        test.done();
    },
    async 'DELETE after CREATE is ignored with success'(test) {
        // GIVEN
        const event = makeEvent({
            RequestType: 'Delete',
            PhysicalResourceId: 'AWSCDK::CustomResourceProviderFramework::CREATE_FAILED',
        });
        // WHEN
        const response = await invokeHandler(event, async (_) => {
            test.ok(false, 'handler should not be called');
        });
        // THEN
        test.deepEqual(response, {
            Status: 'SUCCESS',
            Reason: 'SUCCESS',
            StackId: '<StackId>',
            RequestId: '<RequestId>',
            PhysicalResourceId: 'AWSCDK::CustomResourceProviderFramework::CREATE_FAILED',
            LogicalResourceId: '<LogicalResourceId>',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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