"use strict";
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const app_1 = require("../lib/app");
function withApp(props, block) {
    const app = new app_1.App({
        runtimeInfo: false,
        stackTraces: false,
        ...props,
    });
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        r2.node.addWarning('warning1');
        r2.node.addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStackByName(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.stackName, 'stack1');
        test.deepEqual(stack1.id, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, { Resources: { s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } } } });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.stackName, 'stack2');
        test.deepEqual(stack2.id, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, { Resources: { s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' } } });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2',
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context passed through CDK_CONTEXT has precedence'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2',
        });
        const prog = new app_1.App({
            context: {
                key1: 'val3',
                key2: 'val4',
            },
        });
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: 'DummyResource',
                    Properties: {
                        Prog2: 'Prog2',
                    },
                },
                s1c2r1D1791C01: {
                    Type: 'ResourceType1',
                },
                s1c2r25F685FFF: {
                    Type: 'ResourceType2',
                    Properties: {
                        FromContext: 'HELLO',
                    },
                },
            },
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App({
            context: {
                foo: 'bar',
            },
        });
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                    props: {
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: 'missing-context-key',
                provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: 'missing-context-key-2',
                provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                },
            },
        ]);
        test.done();
    },
    'runtime library versions disabled'(test) {
        const context = {};
        context[cxapi.DISABLE_VERSION_REPORTING] = true;
        const assembly = withApp(context, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        test.deepEqual(libs['jsii-runtime'], 'Java/1.2.3.4');
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs, {
            '@aws-cdk/core': version,
            '@aws-cdk/cx-api': version,
            '@aws-cdk/cloud-assembly-schema': version,
            'jsii-runtime': `node.js/${process.version}`,
        });
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.stackName, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
    'stacks are written to the assembly file in a topological order'(test) {
        // WHEN
        const assembly = withApp({}, (app) => {
            const stackC = new lib_1.Stack(app, 'StackC');
            const stackD = new lib_1.Stack(app, 'StackD');
            const stackA = new lib_1.Stack(app, 'StackA');
            const stackB = new lib_1.Stack(app, 'StackB');
            // Create the following dependency order:
            // A ->
            //      C -> D
            // B ->
            stackC.addDependency(stackA);
            stackC.addDependency(stackB);
            stackD.addDependency(stackC);
        });
        // THEN
        const artifactsIds = assembly.artifacts.map(a => a.id);
        test.ok(artifactsIds.indexOf('StackA') < artifactsIds.indexOf('StackC'));
        test.ok(artifactsIds.indexOf('StackB') < artifactsIds.indexOf('StackC'));
        test.ok(artifactsIds.indexOf('StackC') < artifactsIds.indexOf('StackD'));
        test.done();
    },
    'application support any type in context'(test) {
        const app = new app_1.App({
            context: {
                isString: 'string',
                isNumber: 10,
                isObject: { isString: 'string', isNumber: 10 },
            },
        });
        test.ok(app.node.tryGetContext('isString') === 'string');
        test.ok(app.node.tryGetContext('isNumber') === 10);
        test.deepEqual(app.node.tryGetContext('isObject'), { isString: 'string', isNumber: 10 });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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