"use strict";
const lib_1 = require("../lib");
const synthesis_1 = require("../lib/private/synthesis");
class TaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.STANDARD, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class AsgTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.AUTOSCALING_GROUP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class MapTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.MAP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
module.exports = {
    'Tag visit all children of the applied node'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(map.tags.renderTags(), { foo: 'bar' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'foo', value: 'bar', propagateAtLaunch: true }]);
        test.done();
    },
    'The last aspect applied takes precedence'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        res.node.applyAspect(new lib_1.Tag('foo', 'foobar'));
        res.node.applyAspect(new lib_1.Tag('foo', 'baz'));
        res2.node.applyAspect(new lib_1.Tag('foo', 'good'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'baz' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'good' }]);
        test.done();
    },
    'RemoveTag will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        root.node.applyAspect(new lib_1.Tag('root', 'was here'));
        res.node.applyAspect(new lib_1.Tag('first', 'there is only 1'));
        res.node.applyAspect(new lib_1.RemoveTag('root'));
        res.node.applyAspect(new lib_1.RemoveTag('doesnotexist'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'add will add a tag and remove will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Tag.add(root, 'root', 'was here');
        lib_1.Tag.add(res, 'first', 'there is only 1');
        lib_1.Tag.remove(res, 'root');
        lib_1.Tag.remove(res, 'doesnotexist');
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'the #visit function is idempotent'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.done();
    },
    'removeTag Aspects by default will override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.RemoveTag('key'));
        res2.node.applyAspect(new lib_1.Tag('key', 'value'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), undefined);
        test.done();
    },
    'removeTag Aspects with priority 0 will not override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.RemoveTag('key', { priority: 0 }));
        res2.node.applyAspect(new lib_1.Tag('key', 'value'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'key', value: 'value' }]);
        test.done();
    },
    'Aspects are merged with tags created by L1 Constructor'(test) {
        const root = new lib_1.Stack();
        const aspectBranch = new TaggableResource(root, 'FakeBranchA', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite' },
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        const asgResource = new AsgTaggableResource(aspectBranch, 'FakeAsg', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite', propagateAtLaunch: false },
                    { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                ],
            },
        });
        const mapTaggable = new MapTaggableResource(aspectBranch, 'FakeSam', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: {
                    aspects: 'overwrite',
                    cfn: 'is cool',
                },
            },
        });
        const cfnBranch = new TaggableResource(root, 'FakeBranchB', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        aspectBranch.node.applyAspect(new lib_1.Tag('aspects', 'rule'));
        synthesis_1.synthesize(root);
        test.deepEqual(aspectBranch.testProperties().tags, [{ key: 'aspects', value: 'rule' }, { key: 'cfn', value: 'is cool' }]);
        test.deepEqual(asgResource.testProperties().tags, [
            { key: 'aspects', value: 'rule', propagateAtLaunch: true },
            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
        ]);
        test.deepEqual(mapTaggable.testProperties().tags, {
            aspects: 'rule',
            cfn: 'is cool',
        });
        test.deepEqual(cfnBranch.testProperties().tags, [{ key: 'cfn', value: 'is cool' }]);
        test.done();
    },
    'when invalid tag properties are passed from L1s': {
        'map passed instead of array it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new TaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                        },
                    },
                });
            });
            test.throws(() => {
                new AsgTaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                            propagateAtLaunch: true,
                        },
                    },
                });
            });
            test.done();
        },
        'if array is passed instead of map it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new MapTaggableResource(root, 'FakeSam', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: [
                            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                            { key: 'aspects', value: 'overwrite' },
                        ],
                    },
                });
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC50YWctYXNwZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC50YWctYXNwZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFDQSxnQ0FBOEc7QUFDOUcsd0RBQXNEO0FBRXRELE1BQU0sZ0JBQWlCLFNBQVEsaUJBQVc7SUFFeEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztRQUNoRixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxhQUFPLENBQUMsUUFBUSxFQUFFLHFCQUFxQixFQUFFLElBQUksQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFDTSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQztJQUM1QixDQUFDO0NBQ0Y7QUFFRCxNQUFNLG1CQUFvQixTQUFRLGlCQUFXO0lBRTNDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEIsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLFVBQVUsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUM7UUFDaEYsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsYUFBTyxDQUFDLGlCQUFpQixFQUFFLHFCQUFxQixFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFDTSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQztJQUM1QixDQUFDO0NBQ0Y7QUFFRCxNQUFNLG1CQUFvQixTQUFRLGlCQUFXO0lBRTNDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEIsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLFVBQVUsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUM7UUFDaEYsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsYUFBTyxDQUFDLEdBQUcsRUFBRSxxQkFBcUIsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBQ00sY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxhQUFhLENBQUM7SUFDNUIsQ0FBQztDQUNGO0FBRUQsaUJBQVM7SUFDUCw0Q0FBNEMsQ0FBQyxJQUFVO1FBQ3JELE1BQU0sSUFBSSxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxHQUFHLEdBQUcsSUFBSSxtQkFBbUIsQ0FBQyxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7WUFDMUQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxpQkFBaUIsRUFBRTtZQUMxRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRTVDLHNCQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBQyxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzdGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCwwQ0FBMEMsQ0FBQyxJQUFVO1FBQ25ELE1BQU0sSUFBSSxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDNUMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFDL0MsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDOUMsc0JBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBQyxDQUFDLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsMENBQTBDLENBQUMsSUFBVTtRQUNuRCxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQWdCLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sR0FBRyxHQUFHLElBQUksbUJBQW1CLENBQUMsR0FBRyxFQUFFLGlCQUFpQixFQUFFO1lBQzFELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxHQUFHLEdBQUcsSUFBSSxtQkFBbUIsQ0FBQyxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7WUFDMUQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFNBQUcsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUNuRCxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFNBQUcsQ0FBQyxPQUFPLEVBQUUsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO1FBQzFELEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksZUFBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDNUMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxlQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztRQUNwRCxzQkFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbEYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLEVBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFDLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBQyxDQUFDLENBQUMsQ0FBQztRQUMzRyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25GLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCw4REFBOEQsQ0FBQyxJQUFVO1FBQ3ZFLE1BQU0sSUFBSSxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxHQUFHLEdBQUcsSUFBSSxtQkFBbUIsQ0FBQyxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7WUFDMUQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxpQkFBaUIsRUFBRTtZQUMxRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILFNBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQztRQUNsQyxTQUFHLENBQUMsR0FBRyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUN6QyxTQUFHLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN4QixTQUFHLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUVoQyxzQkFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbEYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLEVBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFDLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBQyxDQUFDLENBQUMsQ0FBQztRQUMzRyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25GLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLE1BQU0sSUFBSSxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBRUgsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDNUMsc0JBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNwRSxzQkFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BFLHNCQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELDhEQUE4RCxDQUFDLElBQVU7UUFDdkUsTUFBTSxJQUFJLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUN6QixNQUFNLEdBQUcsR0FBRyxJQUFJLGdCQUFnQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDckQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxjQUFjLEVBQUU7WUFDckQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQy9DLHNCQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsdUVBQXVFLENBQUMsSUFBVTtRQUNoRixNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQWdCLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksZUFBUyxDQUFDLEtBQUssRUFBRSxFQUFDLFFBQVEsRUFBRSxDQUFDLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDL0Msc0JBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdkUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELHdEQUF3RCxDQUFDLElBQVU7UUFDakUsTUFBTSxJQUFJLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUN6QixNQUFNLFlBQVksR0FBRyxJQUFJLGdCQUFnQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDN0QsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxFQUFFO29CQUNKLEVBQUMsR0FBRyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFDO29CQUNwQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBQztpQkFDL0I7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0sV0FBVyxHQUFHLElBQUksbUJBQW1CLENBQUMsWUFBWSxFQUFFLFNBQVMsRUFBRTtZQUNuRSxJQUFJLEVBQUUsa0JBQWtCO1lBQ3hCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUU7b0JBQ0osRUFBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxXQUFXLEVBQUUsaUJBQWlCLEVBQUUsS0FBSyxFQUFDO29CQUM5RCxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUM7aUJBQ3hEO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLFdBQVcsR0FBRyxJQUFJLG1CQUFtQixDQUFDLFlBQVksRUFBRSxTQUFTLEVBQUU7WUFDbkUsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxFQUFFO29CQUNKLE9BQU8sRUFBRSxXQUFXO29CQUNwQixHQUFHLEVBQUUsU0FBUztpQkFDZjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxTQUFTLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzFELElBQUksRUFBRSxrQkFBa0I7WUFDeEIsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRTtvQkFDSixFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBQztpQkFDL0I7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILFlBQVksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQzFELHNCQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUMsRUFBRSxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBQyxDQUFDLENBQUMsQ0FBQztRQUN0SCxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxJQUFJLEVBQUU7WUFDaEQsRUFBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFDO1lBQ3hELEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBQztTQUN4RCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxJQUFJLEVBQUU7WUFDaEQsT0FBTyxFQUFFLE1BQU07WUFDZixHQUFHLEVBQUUsU0FBUztTQUNmLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2xGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCxpREFBaUQsRUFBRTtRQUNqRCx1Q0FBdUMsQ0FBQyxJQUFVO1lBQ2hELE1BQU0sSUFBSSxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7WUFDekIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2YsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO29CQUN4QyxJQUFJLEVBQUUsa0JBQWtCO29CQUN4QixVQUFVLEVBQUU7d0JBQ1YsSUFBSSxFQUFFOzRCQUNKLEdBQUcsRUFBRSxTQUFTOzRCQUNkLE9BQU8sRUFBRSxXQUFXO3lCQUNyQjtxQkFDRjtpQkFDRixDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksbUJBQW1CLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtvQkFDM0MsSUFBSSxFQUFFLGtCQUFrQjtvQkFDeEIsVUFBVSxFQUFFO3dCQUNWLElBQUksRUFBRTs0QkFDSixHQUFHLEVBQUUsU0FBUzs0QkFDZCxPQUFPLEVBQUUsV0FBVzs0QkFDcEIsaUJBQWlCLEVBQUUsSUFBSTt5QkFDeEI7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsNkNBQTZDLENBQUMsSUFBVTtZQUN0RCxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1lBQ3pCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksbUJBQW1CLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtvQkFDdkMsSUFBSSxFQUFFLGtCQUFrQjtvQkFDeEIsVUFBVSxFQUFFO3dCQUNWLElBQUksRUFBRTs0QkFDSixFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUM7NEJBQ3ZELEVBQUMsR0FBRyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFDO3lCQUNyQztxQkFDRjtpQkFDRixDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIENmblJlc291cmNlUHJvcHMsIENvbnN0cnVjdCwgUmVtb3ZlVGFnLCBTdGFjaywgVGFnLCBUYWdNYW5hZ2VyLCBUYWdUeXBlIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IHN5bnRoZXNpemUgfSBmcm9tICcuLi9saWIvcHJpdmF0ZS9zeW50aGVzaXMnO1xuXG5jbGFzcyBUYWdnYWJsZVJlc291cmNlIGV4dGVuZHMgQ2ZuUmVzb3VyY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgdGFnczogVGFnTWFuYWdlcjtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENmblJlc291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICBjb25zdCB0YWdzID0gcHJvcHMucHJvcGVydGllcyA9PT0gdW5kZWZpbmVkID8gdW5kZWZpbmVkIDogcHJvcHMucHJvcGVydGllcy50YWdzO1xuICAgIHRoaXMudGFncyA9IG5ldyBUYWdNYW5hZ2VyKFRhZ1R5cGUuU1RBTkRBUkQsICdBV1M6OkZha2U6OlJlc291cmNlJywgdGFncyk7XG4gIH1cbiAgcHVibGljIHRlc3RQcm9wZXJ0aWVzKCkge1xuICAgIHJldHVybiB0aGlzLmNmblByb3BlcnRpZXM7XG4gIH1cbn1cblxuY2xhc3MgQXNnVGFnZ2FibGVSZXNvdXJjZSBleHRlbmRzIENmblJlc291cmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHRhZ3M6IFRhZ01hbmFnZXI7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZm5SZXNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gICAgY29uc3QgdGFncyA9IHByb3BzLnByb3BlcnRpZXMgPT09IHVuZGVmaW5lZCA/IHVuZGVmaW5lZCA6IHByb3BzLnByb3BlcnRpZXMudGFncztcbiAgICB0aGlzLnRhZ3MgPSBuZXcgVGFnTWFuYWdlcihUYWdUeXBlLkFVVE9TQ0FMSU5HX0dST1VQLCAnQVdTOjpGYWtlOjpSZXNvdXJjZScsIHRhZ3MpO1xuICB9XG4gIHB1YmxpYyB0ZXN0UHJvcGVydGllcygpIHtcbiAgICByZXR1cm4gdGhpcy5jZm5Qcm9wZXJ0aWVzO1xuICB9XG59XG5cbmNsYXNzIE1hcFRhZ2dhYmxlUmVzb3VyY2UgZXh0ZW5kcyBDZm5SZXNvdXJjZSB7XG4gIHB1YmxpYyByZWFkb25seSB0YWdzOiBUYWdNYW5hZ2VyO1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2ZuUmVzb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIGNvbnN0IHRhZ3MgPSBwcm9wcy5wcm9wZXJ0aWVzID09PSB1bmRlZmluZWQgPyB1bmRlZmluZWQgOiBwcm9wcy5wcm9wZXJ0aWVzLnRhZ3M7XG4gICAgdGhpcy50YWdzID0gbmV3IFRhZ01hbmFnZXIoVGFnVHlwZS5NQVAsICdBV1M6OkZha2U6OlJlc291cmNlJywgdGFncyk7XG4gIH1cbiAgcHVibGljIHRlc3RQcm9wZXJ0aWVzKCkge1xuICAgIHJldHVybiB0aGlzLmNmblByb3BlcnRpZXM7XG4gIH1cbn1cblxuZXhwb3J0ID0ge1xuICAnVGFnIHZpc2l0IGFsbCBjaGlsZHJlbiBvZiB0aGUgYXBwbGllZCBub2RlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJlcyA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgcmVzMiA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJlcywgJ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICBjb25zdCBhc2cgPSBuZXcgQXNnVGFnZ2FibGVSZXNvdXJjZShyZXMsICdBc2dGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG5cbiAgICBjb25zdCBtYXAgPSBuZXcgTWFwVGFnZ2FibGVSZXNvdXJjZShyZXMsICdNYXBGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnZm9vJywgJ2JhcicpKTtcblxuICAgIHN5bnRoZXNpemUocm9vdCk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAnZm9vJywgdmFsdWU6ICdiYXInfV0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlczIudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAnZm9vJywgdmFsdWU6ICdiYXInfV0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKG1hcC50YWdzLnJlbmRlclRhZ3MoKSwge2ZvbzogJ2Jhcid9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc2cudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAnZm9vJywgdmFsdWU6ICdiYXInLCBwcm9wYWdhdGVBdExhdW5jaDogdHJ1ZX1dKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ1RoZSBsYXN0IGFzcGVjdCBhcHBsaWVkIHRha2VzIHByZWNlZGVuY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcmVzID0gbmV3IFRhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICBjb25zdCByZXMyID0gbmV3IFRhZ2dhYmxlUmVzb3VyY2UocmVzLCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIHJlcy5ub2RlLmFwcGx5QXNwZWN0KG5ldyBUYWcoJ2ZvbycsICdiYXInKSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnZm9vJywgJ2Zvb2JhcicpKTtcbiAgICByZXMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdmb28nLCAnYmF6JykpO1xuICAgIHJlczIubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdmb28nLCAnZ29vZCcpKTtcbiAgICBzeW50aGVzaXplKHJvb3QpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlcy50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmb28nLCB2YWx1ZTogJ2Jheid9XSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmb28nLCB2YWx1ZTogJ2dvb2QnfV0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnUmVtb3ZlVGFnIHdpbGwgcmVtb3ZlIGEgdGFnIGlmIGl0IGV4aXN0cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlczIgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyZXMsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgYXNnID0gbmV3IEFzZ1RhZ2dhYmxlUmVzb3VyY2UocmVzLCAnQXNnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbWFwID0gbmV3IE1hcFRhZ2dhYmxlUmVzb3VyY2UocmVzLCAnTWFwRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIHJvb3Qubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdyb290JywgJ3dhcyBoZXJlJykpO1xuICAgIHJlcy5ub2RlLmFwcGx5QXNwZWN0KG5ldyBUYWcoJ2ZpcnN0JywgJ3RoZXJlIGlzIG9ubHkgMScpKTtcbiAgICByZXMubm9kZS5hcHBseUFzcGVjdChuZXcgUmVtb3ZlVGFnKCdyb290JykpO1xuICAgIHJlcy5ub2RlLmFwcGx5QXNwZWN0KG5ldyBSZW1vdmVUYWcoJ2RvZXNub3RleGlzdCcpKTtcbiAgICBzeW50aGVzaXplKHJvb3QpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZpcnN0JywgdmFsdWU6ICd0aGVyZSBpcyBvbmx5IDEnfV0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKG1hcC50YWdzLnJlbmRlclRhZ3MoKSwge2ZpcnN0OiAndGhlcmUgaXMgb25seSAxJ30pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzZy50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmaXJzdCcsIHZhbHVlOiAndGhlcmUgaXMgb25seSAxJywgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWV9XSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmaXJzdCcsIHZhbHVlOiAndGhlcmUgaXMgb25seSAxJ31dKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ2FkZCB3aWxsIGFkZCBhIHRhZyBhbmQgcmVtb3ZlIHdpbGwgcmVtb3ZlIGEgdGFnIGlmIGl0IGV4aXN0cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlczIgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyZXMsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgYXNnID0gbmV3IEFzZ1RhZ2dhYmxlUmVzb3VyY2UocmVzLCAnQXNnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbWFwID0gbmV3IE1hcFRhZ2dhYmxlUmVzb3VyY2UocmVzLCAnTWFwRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIFRhZy5hZGQocm9vdCwgJ3Jvb3QnLCAnd2FzIGhlcmUnKTtcbiAgICBUYWcuYWRkKHJlcywgJ2ZpcnN0JywgJ3RoZXJlIGlzIG9ubHkgMScpO1xuICAgIFRhZy5yZW1vdmUocmVzLCAncm9vdCcpO1xuICAgIFRhZy5yZW1vdmUocmVzLCAnZG9lc25vdGV4aXN0Jyk7XG5cbiAgICBzeW50aGVzaXplKHJvb3QpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZpcnN0JywgdmFsdWU6ICd0aGVyZSBpcyBvbmx5IDEnfV0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKG1hcC50YWdzLnJlbmRlclRhZ3MoKSwge2ZpcnN0OiAndGhlcmUgaXMgb25seSAxJ30pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzZy50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmaXJzdCcsIHZhbHVlOiAndGhlcmUgaXMgb25seSAxJywgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWV9XSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmaXJzdCcsIHZhbHVlOiAndGhlcmUgaXMgb25seSAxJ31dKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ3RoZSAjdmlzaXQgZnVuY3Rpb24gaXMgaWRlbXBvdGVudCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuXG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnZm9vJywgJ2JhcicpKTtcbiAgICBzeW50aGVzaXplKHJvb3QpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlcy50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmb28nLCB2YWx1ZTogJ2Jhcid9XSk7XG4gICAgc3ludGhlc2l6ZShyb290KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAnZm9vJywgdmFsdWU6ICdiYXInfV0pO1xuICAgIHN5bnRoZXNpemUocm9vdCk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnYmFyJ31dKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ3JlbW92ZVRhZyBBc3BlY3RzIGJ5IGRlZmF1bHQgd2lsbCBvdmVycmlkZSBjaGlsZCBUYWcgQXNwZWN0cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlczIgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyZXMsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFJlbW92ZVRhZygna2V5JykpO1xuICAgIHJlczIubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdrZXknLCAndmFsdWUnKSk7XG4gICAgc3ludGhlc2l6ZShyb290KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIHVuZGVmaW5lZCk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgdW5kZWZpbmVkKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ3JlbW92ZVRhZyBBc3BlY3RzIHdpdGggcHJpb3JpdHkgMCB3aWxsIG5vdCBvdmVycmlkZSBjaGlsZCBUYWcgQXNwZWN0cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlczIgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyZXMsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFJlbW92ZVRhZygna2V5Jywge3ByaW9yaXR5OiAwfSkpO1xuICAgIHJlczIubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdrZXknLCAndmFsdWUnKSk7XG4gICAgc3ludGhlc2l6ZShyb290KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIHVuZGVmaW5lZCk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdrZXknLCB2YWx1ZTogJ3ZhbHVlJ31dKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ0FzcGVjdHMgYXJlIG1lcmdlZCB3aXRoIHRhZ3MgY3JlYXRlZCBieSBMMSBDb25zdHJ1Y3RvcicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBhc3BlY3RCcmFuY2ggPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZUJyYW5jaEEnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHRhZ3M6IFtcbiAgICAgICAgICB7a2V5OiAnYXNwZWN0cycsIHZhbHVlOiAnb3ZlcndyaXRlJ30sXG4gICAgICAgICAge2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCd9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBhc2dSZXNvdXJjZSA9IG5ldyBBc2dUYWdnYWJsZVJlc291cmNlKGFzcGVjdEJyYW5jaCwgJ0Zha2VBc2cnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHRhZ3M6IFtcbiAgICAgICAgICB7a2V5OiAnYXNwZWN0cycsIHZhbHVlOiAnb3ZlcndyaXRlJywgcHJvcGFnYXRlQXRMYXVuY2g6IGZhbHNlfSxcbiAgICAgICAgICB7a2V5OiAnY2ZuJywgdmFsdWU6ICdpcyBjb29sJywgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWV9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBtYXBUYWdnYWJsZSA9IG5ldyBNYXBUYWdnYWJsZVJlc291cmNlKGFzcGVjdEJyYW5jaCwgJ0Zha2VTYW0nLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHRhZ3M6IHtcbiAgICAgICAgICBhc3BlY3RzOiAnb3ZlcndyaXRlJyxcbiAgICAgICAgICBjZm46ICdpcyBjb29sJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgY2ZuQnJhbmNoID0gbmV3IFRhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VCcmFuY2hCJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB0YWdzOiBbXG4gICAgICAgICAge2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCd9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBhc3BlY3RCcmFuY2gubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdhc3BlY3RzJywgJ3J1bGUnKSk7XG4gICAgc3ludGhlc2l6ZShyb290KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3BlY3RCcmFuY2gudGVzdFByb3BlcnRpZXMoKS50YWdzLCBbe2tleTogJ2FzcGVjdHMnLCB2YWx1ZTogJ3J1bGUnfSwge2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCd9XSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNnUmVzb3VyY2UudGVzdFByb3BlcnRpZXMoKS50YWdzLCBbXG4gICAgICB7a2V5OiAnYXNwZWN0cycsIHZhbHVlOiAncnVsZScsIHByb3BhZ2F0ZUF0TGF1bmNoOiB0cnVlfSxcbiAgICAgIHtrZXk6ICdjZm4nLCB2YWx1ZTogJ2lzIGNvb2wnLCBwcm9wYWdhdGVBdExhdW5jaDogdHJ1ZX0sXG4gICAgXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwobWFwVGFnZ2FibGUudGVzdFByb3BlcnRpZXMoKS50YWdzLCB7XG4gICAgICBhc3BlY3RzOiAncnVsZScsXG4gICAgICBjZm46ICdpcyBjb29sJyxcbiAgICB9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChjZm5CcmFuY2gudGVzdFByb3BlcnRpZXMoKS50YWdzLCBbe2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCd9XSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICd3aGVuIGludmFsaWQgdGFnIHByb3BlcnRpZXMgYXJlIHBhc3NlZCBmcm9tIEwxcyc6IHtcbiAgICAnbWFwIHBhc3NlZCBpbnN0ZWFkIG9mIGFycmF5IGl0IHJhaXNlcycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgcm9vdCA9IG5ldyBTdGFjaygpO1xuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZUJyYW5jaEEnLCB7XG4gICAgICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIHRhZ3M6IHtcbiAgICAgICAgICAgICAgY2ZuOiAnaXMgY29vbCcsXG4gICAgICAgICAgICAgIGFzcGVjdHM6ICdvdmVyd3JpdGUnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgQXNnVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZUJyYW5jaEEnLCB7XG4gICAgICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIHRhZ3M6IHtcbiAgICAgICAgICAgICAgY2ZuOiAnaXMgY29vbCcsXG4gICAgICAgICAgICAgIGFzcGVjdHM6ICdvdmVyd3JpdGUnLFxuICAgICAgICAgICAgICBwcm9wYWdhdGVBdExhdW5jaDogdHJ1ZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2lmIGFycmF5IGlzIHBhc3NlZCBpbnN0ZWFkIG9mIG1hcCBpdCByYWlzZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgbmV3IE1hcFRhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VTYW0nLCB7XG4gICAgICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIHRhZ3M6IFtcbiAgICAgICAgICAgICAge2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCcsIHByb3BhZ2F0ZUF0TGF1bmNoOiB0cnVlfSxcbiAgICAgICAgICAgICAge2tleTogJ2FzcGVjdHMnLCB2YWx1ZTogJ292ZXJ3cml0ZSd9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcbn07XG4iXX0=