import evaluate as hf_evaluate
import torch
import transformers

from enum import Enum
from abc import ABC, abstractmethod
from typing import Any, Dict, List
from transformers import pipeline, AutoConfig

TOXIGEN_SCORE_NAME = "toxicity"

DETOXIFY_SCORE_TOXICITY = "toxicity"
DETOXIFY_SCORE_SEVERE_TOXICITY = "severe_toxicity"
DETOXIFY_SCORE_OBSCENE = "obscene"
DETOXIFY_SCORE_IDENTITY_ATTACK = "identity_attack"
DETOXIFY_SCORE_INSULT = "insult"
DETOXIFY_SCORE_THREAT = "threat"
DETOXIFY_SCORE_SEXUAL_EXPLICIT = "sexual_explicit"
DETOXIFY_SCORE_NAMES = [
    DETOXIFY_SCORE_TOXICITY,
    DETOXIFY_SCORE_SEVERE_TOXICITY,
    DETOXIFY_SCORE_OBSCENE,
    DETOXIFY_SCORE_IDENTITY_ATTACK,
    DETOXIFY_SCORE_INSULT,
    DETOXIFY_SCORE_THREAT,
    DETOXIFY_SCORE_SEXUAL_EXPLICIT,
]


class BaseHelperModel(ABC):
    """
    Base class for 3P helper model invoker. Note: These Helper models are inherently
    Machine learning models being used by Evaluation algorithms.
    """

    @abstractmethod
    def get_helper_scores(self, text_input: str) -> Any:
        """
        Method to invoke helper model
        :param text_input: model text input
        :returns: model output
        """


class ToxigenHelperModel(BaseHelperModel):
    """
    Helper model for toxigen model: https://huggingface.co/tomh/toxigen_roberta/tree/main
    """

    TOXIGEN_MODEL_NAME = "tomh/toxigen_roberta"

    def __init__(self):
        """
        Constructor to locally load the helper model for inference.
        """
        self._model = pipeline("text-classification", model=self.TOXIGEN_MODEL_NAME)

    def __reduce__(self):
        """Serializer method so that instances of this class can be made into shared resources."""
        return self.__class__, ()

    def get_helper_scores(self, text_input: List[str]) -> Dict[str, List[float]]:  # type: ignore[override]
        """
        Method to get scores from ToxigenHelper
        :param text_input: list of text inputs for the model
        :returns: dict with key as score name and value being list of scores for text inputs

        Note: Toxigen scores are for label: LABEL_1
        """
        inference_output = self._model(text_input)
        result = {
            TOXIGEN_SCORE_NAME: [x["score"] if x["label"] == "LABEL_1" else 1.0 - x["score"] for x in inference_output]
        }
        return result

    @staticmethod
    def get_score_names() -> List[str]:
        """
        Util method to return name of scores generated by helper model
        :returns: List of score names
        """
        return [TOXIGEN_SCORE_NAME]


class DetoxifyHelperModel(BaseHelperModel):
    """
    Helper model for Detoxify: https://github.com/unitaryai/detoxify

    Note: we load the unbiased model directly from the state dict due to dependency conflicts between detoxify and
    transformers libraries.

    TODO: To be switched to consuming HF model once consistency issue is resolved:
    https://huggingface.co/unitary/unbiased-toxic-roberta. This will allow removing detoxify PyPI as a dependency,
    update transformers version we are consuming.
    """

    UNBIASED_MODEL_URL = (
        "https://github.com/unitaryai/detoxify/releases/download/v0.3-alpha/toxic_debiased-c7548aa0.ckpt"
    )

    def __init__(self):
        """
        Constructor to locally load the helper model for inference.
        """
        state_dict = torch.hub.load_state_dict_from_url(self.UNBIASED_MODEL_URL, map_location="cpu")
        config = state_dict["config"]["arch"]["args"]
        self._model = (
            getattr(transformers, config["model_name"])
            .from_pretrained(
                pretrained_model_name_or_path=None,
                config=AutoConfig.from_pretrained(config["model_type"], num_labels=config["num_classes"]),
                state_dict=state_dict["state_dict"],
                local_files_only=False,
            )
            .to("cpu")
        )
        self._tokenizer = getattr(transformers, config["tokenizer_name"]).from_pretrained(config["model_type"])

    def __reduce__(self):
        """Serializer method so that instances of this class can be made into shared resources."""
        return self.__class__, ()

    def get_helper_scores(self, text_input: List[str]) -> Dict[str, List[float]]:  # type: ignore[override]
        """
        Method to get scores from DetoxifyHelper
        :param text_input: list of text inputs for the model
        :returns: dict with keys as score name and value being list of scores for text inputs
        """
        inputs = self._tokenizer(text_input, return_tensors="pt", truncation=True, padding=True).to(self._model.device)
        scores = torch.sigmoid(self._model(**inputs)[0]).cpu().detach().numpy()
        return {
            score_name: [score[i].tolist() for score in scores]
            for i, score_name in enumerate(DetoxifyHelperModel.get_score_names())
        }

    @staticmethod
    def get_score_names() -> List[str]:
        """
        Util method to return name of scores generated by helper model
        :returns: List of score names
        """
        return DETOXIFY_SCORE_NAMES


class BertscoreHelperModel(BaseHelperModel):
    """
    BERTscore is a similarity-based metric that compares the embedding of the prediction and target sentences
    under a (learned) model, typically, from the BERT family.
    This score may lead to increased flexibility compared to rouge and METEOR in terms of rephrasing since
    semantically similar sentences are (typically) embedded similarly.
    https://huggingface.co/spaces/evaluate-metric/bertscore
    Note: we specify that this Ray actor requires num_cpus=1 in order to limit the number of concurrently
    running tasks or actors to avoid out of memory issues.
    See https://docs.ray.io/en/latest/ray-core/patterns/limit-running-tasks.html#core-patterns-limit-running-tasks
    for a detailed explanation.
    """

    def __init__(self, model_type: str):  # pragma: no cover
        """
        Default constructor
        :param model_type: Model type to be used for bertscore
        """
        self._bertscore = hf_evaluate.load("bertscore")
        self._model_type = model_type

    def __reduce__(self):
        """Serializer method so that instances of this class can be made into shared resources."""
        return self.__class__, (self._model_type,)

    def get_helper_scores(self, target_output: str, model_output: str) -> float:  # type: ignore[override]
        """
        Method to invoke the concerned model and get bertscore
        :param target_output: Reference text
        :model_output: Model prediction text
        """
        # Note: the following code is covered by unit tests,
        # but since it gets executed by Ray, Mypy marks it
        # as not covered.
        return self._bertscore.compute(  # pragma: no cover
            predictions=[model_output],
            references=[target_output],
            model_type=self._model_type,
        )["f1"][0]


class BertscoreHelperModelTypes(Enum):
    """This class holds the names of all the allowed models for computing the BERTScore."""

    MICROSOFT_DEBERTA_MODEL = "microsoft/deberta-xlarge-mnli"
    ROBERTA_MODEL = "roberta-large-mnli"

    @classmethod
    def model_is_allowed(cls, model_name: str) -> bool:
        """
        Given a model name like 'roberta-large-mnli', check if this is an allowed model for computing BERTScore.
        """
        for elem in iter(cls):
            if elem.value == model_name:
                return True
        return False

    @classmethod
    def model_list(cls) -> List[str]:
        """
        Return a list of all the allowed models for computing BERTScore.
        """
        return [elem.value for elem in iter(cls)]
