"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tier = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * A deployment environment with a specific purpose and audience.
 *
 * You can create any Tier you like, but we include those explained by DTAP.
 *
 * @see https://en.wikipedia.org/wiki/Development,_testing,_acceptance_and_production
 */
class Tier {
    /**
     * Creates a new Tier.
     *
     * @param id - The machine-readable identifier for this tier (e.g. prod)
     * @param label - The human-readable label for this tier (e.g. Production)
     */
    constructor(id, label) {
        this.id = id;
        this.label = label;
    }
    /**
     * Return the deployment tier that corresponds to the provided value.
     *
     * Production: "live", "prod", or "production".
     * Acceptance: "uat", "stage", "staging", or "acceptance".
     * Testing: "qc", "qa", "test", or "testing".
     * Development: anything else.
     *
     * @param value - The value to parse, case-insensitive
     * @returns The matching deployment tier, or else `DEVELOPMENT`.
     */
    static parse(value) {
        let tier;
        switch (value.trim().toLowerCase()) {
            case "production":
            case "prod":
            case "live":
                tier = Tier.PRODUCTION;
                break;
            case "uat":
            case "stage":
            case "staging":
            case "accept":
            case "acceptance":
                tier = Tier.ACCEPTANCE;
                break;
            case "qc":
            case "qa":
            case "test":
            case "testing":
                tier = Tier.TESTING;
                break;
            default:
                tier = Tier.DEVELOPMENT;
        }
        return tier;
    }
    /**
     * Compares this tier to the provided value and tests for equality.
     *
     * @param other - The value to compare.
     * @returns Whether the provided value is equal to this tier.
     */
    matches(other) {
        if (this === other) {
            return true;
        }
        return other instanceof Tier && this.id === other.id;
    }
    /**
     * Adds the label of this tier as a tag to the provided construct.
     */
    applyTags(construct) {
        aws_cdk_lib_1.Tags.of(construct).add("DeploymentTier", this.label);
    }
}
exports.Tier = Tier;
_a = JSII_RTTI_SYMBOL_1;
Tier[_a] = { fqn: "shady-island.Tier", version: "0.0.34" };
/**
 * A tier that represents a production environment.
 */
Tier.PRODUCTION = new Tier("prod", "Production");
/**
 * A tier that represents an acceptance environment.
 */
Tier.ACCEPTANCE = new Tier("uat", "Acceptance");
/**
 * A tier that represents a testing environment.
 */
Tier.TESTING = new Tier("test", "Testing");
/**
 * A tier that represents a development environment.
 */
Tier.DEVELOPMENT = new Tier("dev", "Development");
//# sourceMappingURL=data:application/json;base64,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