#  Copyright 2024 Palantir Technologies, Inc.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.


import typing
from functools import cached_property

from foundry_sdk import _core as core


class FilesystemClient:
    """
    The API client for the Filesystem Namespace.

    :param auth: Your auth configuration.
    :param hostname: Your Foundry hostname (for example, "myfoundry.palantirfoundry.com"). This can also include your API gateway service URI.
    :param config: Optionally specify the configuration for the HTTP session.
    """

    def __init__(
        self,
        auth: core.Auth,
        hostname: str,
        config: typing.Optional[core.Config] = None,
    ):
        self._auth = auth
        self._hostname = hostname
        self._config = config

    @cached_property
    def Folder(self):
        from foundry_sdk.v2.filesystem.folder import FolderClient

        return FolderClient(
            auth=self._auth,
            hostname=self._hostname,
            config=self._config,
        )

    @cached_property
    def Project(self):
        from foundry_sdk.v2.filesystem.project import ProjectClient

        return ProjectClient(
            auth=self._auth,
            hostname=self._hostname,
            config=self._config,
        )

    @cached_property
    def Resource(self):
        from foundry_sdk.v2.filesystem.resource import ResourceClient

        return ResourceClient(
            auth=self._auth,
            hostname=self._hostname,
            config=self._config,
        )

    @cached_property
    def Space(self):
        from foundry_sdk.v2.filesystem.space import SpaceClient

        return SpaceClient(
            auth=self._auth,
            hostname=self._hostname,
            config=self._config,
        )


class AsyncFilesystemClient:
    """
    The Async API client for the Filesystem Namespace.

    :param auth: Your auth configuration.
    :param hostname: Your Foundry hostname (for example, "myfoundry.palantirfoundry.com"). This can also include your API gateway service URI.
    :param config: Optionally specify the configuration for the HTTP session.
    """

    def __init__(
        self,
        auth: core.Auth,
        hostname: str,
        config: typing.Optional[core.Config] = None,
    ):
        from foundry_sdk.v2.filesystem.folder import AsyncFolderClient
        from foundry_sdk.v2.filesystem.project import AsyncProjectClient
        from foundry_sdk.v2.filesystem.resource import AsyncResourceClient
        from foundry_sdk.v2.filesystem.space import AsyncSpaceClient

        self.Folder = AsyncFolderClient(auth=auth, hostname=hostname, config=config)

        self.Project = AsyncProjectClient(auth=auth, hostname=hostname, config=config)

        self.Resource = AsyncResourceClient(auth=auth, hostname=hostname, config=config)

        self.Space = AsyncSpaceClient(auth=auth, hostname=hostname, config=config)
