# 🥑 Avocavo Nutrition API Python SDK

[![PyPI version](https://badge.fury.io/py/avocavo-nutrition.svg)](https://badge.fury.io/py/avocavo-nutrition)
[![Python Support](https://img.shields.io/pypi/pyversions/avocavo-nutrition.svg)](https://pypi.org/project/avocavo-nutrition/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Downloads](https://pepy.tech/badge/avocavo-nutrition)](https://pepy.tech/projects/avocavo-nutrition)

**Bulletproof nutrition data with anti-hallucination protection + enterprise-grade secure credential storage.**

Get 100% reliable USDA nutrition data powered by our bulletproof system. Multi-tier caching, intelligent parsing, and mathematical calculations ensure accuracy. Real FDC IDs, sub-second responses, and accepts any ingredient description format. **Built-in secure keyring storage** keeps your API credentials safe. Perfect for recipe apps, fitness trackers, meal planners, and food tech products.

### ✨ Key Features
- ✅ **Secure credential storage** - System keyring integration (macOS/Windows/Linux)
- ✅ **100% USDA verified** - Real FDC IDs with every response
- ✅ **Anti-hallucination** - GPT never provides nutrition data
- ✅ **Sub-second responses** - Multi-tier caching system
- ✅ **Any input format** - Natural language processing
- ✅ **Production ready** - OAuth, server-side key management, batch processing

## 🚀 Quick Start

### Installation

```bash
pip install avocavo-nutrition
```

### 🔐 Two-Step Authentication (Recommended)

Modern OAuth + API key system for maximum security:

```python
import avocavo_nutrition as av

# Step 1: OAuth login (stores JWT token securely)
av.login()  # Google OAuth by default
# av.login(provider="github")  # Or GitHub OAuth

# Step 2: Create/manage API keys with JWT authentication
result = av.create_api_key(
    name="My App Key",
    description="Production API key for my recipe app",
    environment="production"
)
print(f"Created key: {result['key']['api_key']}")

# Step 3: Use nutrition API (automatically uses stored API key)
nutrition = av.analyze_ingredient("2 cups chocolate chips")
print(f"Calories: {nutrition.nutrition.calories_total}")
print(f"Protein: {nutrition.nutrition.protein_total}g")
print(f"USDA: {nutrition.usda_match.description}")
```

### 🔑 API Key Management

```python
# List all your API keys
keys = av.list_api_keys()
for key in keys['keys']:
    print(f"{key['key_name']}: {key['monthly_usage']}/{key['monthly_limit']}")

# Switch to different API key
av.switch_to_api_key("sk_prod_abc123...")

# Delete old keys
av.delete_api_key(key_id=123)
```

### 🚀 Alternative Methods

```python
# Method 1: Direct API key (if you already have one)
client = av.NutritionAPI(api_key="sk_starter_your_api_key_here")

# Method 2: Environment variable (for CI/CD)
# export AVOCAVO_API_KEY=sk_starter_your_api_key_here
result = av.analyze_ingredient("2 cups flour")
```

### 🔐 Secure Credential Management

The Python SDK uses a modern two-step authentication system:

1. **JWT Tokens**: For identity verification and API key management
2. **API Keys**: For actual nutrition API requests

All credentials are stored securely using your system's keyring:

- **macOS**: Stored in Keychain (same as Safari, Chrome)
- **Windows**: Stored in Credential Manager  
- **Linux**: Stored in Secret Service (gnome-keyring, kwallet)

```python
# One-time OAuth login - JWT token stored securely
av.login()  # Opens browser for Google OAuth

# Create API keys using JWT authentication
new_key = av.create_api_key(
    name="Production Key",
    environment="production"
)

# Use anywhere in your projects - API key automatically used
result = av.analyze_ingredient("1 cup quinoa")
print(f"✅ {result.nutrition.calories_total} calories")

# Check login status
user = av.get_current_user()
if user:
    print(f"Logged in as: {user['email']}")
    
# Logout and clear credentials  
av.logout()
```

### 🔑 Direct API Key (Legacy)

If you already have an API key, you can use it directly:

```python
from avocavo_nutrition import NutritionAPI

# Use existing API key directly
client = NutritionAPI(api_key="sk_starter_your_api_key_here")
result = client.analyze_ingredient("1 cup rice")
print(f"Calories: {result.nutrition.calories_total}")

# Or via environment variable
import os
os.environ['AVOCAVO_API_KEY'] = 'sk_starter_your_api_key_here'
result = av.analyze_ingredient("1 cup rice")  # Uses env var automatically
```

## ⚡ Any Input Format

**No rigid formatting required - describe ingredients any way you want:**

```python
# Descriptive ingredients
result = av.analyze_ingredient("2 tablespoons of extra virgin olive oil")

# Any style works
result = av.analyze_ingredient("I'm using about 1 cup of chicken breast, grilled")

# Precise or approximate
result = av.analyze_ingredient("100g brown rice, cooked")

# Include preparation details
result = av.analyze_ingredient("2 cups flour (all-purpose, for baking)")
```

**Bulletproof USDA matching with 94%+ cache hit rate and anti-hallucination protection.**

## 🎯 What You Can Do

### 🥘 Analyze Ingredients
```python
# Any ingredient with quantity - flexible input formats
result = av.analyze_ingredient("2 tbsp olive oil")
if result.success:
    print(f"Calories: {result.nutrition.calories}")
    print(f"Fat: {result.nutrition.fat}g")
    print(f"Verify: {result.verification_url}")
```

### 🍳 Analyze Complete Recipes
```python
# Full recipe with per-serving calculations
recipe = av.analyze_recipe([
    "2 cups all-purpose flour",
    "1 cup whole milk",
    "2 large eggs", 
    "1/4 cup sugar"
], servings=8)

print(f"Per serving: {recipe.nutrition.per_serving.calories} calories")
print(f"Total recipe: {recipe.nutrition.total.calories} calories")
```

### ⚡ Batch Processing
```python
# Analyze multiple ingredients efficiently
# Batch limits: Free (5), Starter (10), Professional (20), Enterprise (50+) ingredients
batch = av.analyze_batch([
    "1 cup quinoa",
    "2 tbsp olive oil", 
    "4 oz salmon",
    "1 cup spinach"
])

for item in batch.results:
    if item.success:
        print(f"{item.ingredient}: {item.nutrition.calories} cal")
```

### 📊 Account Management
```python
# Check your usage and limits
account = av.get_account_usage()
print(f"Plan: {account.plan_name}")
print(f"Usage: {account.usage.current_month}/{account.usage.monthly_limit}")
print(f"Remaining: {account.usage.remaining}")
```

## ✨ Bulletproof Features

### 🛡️ **Anti-Hallucination Protection**
- **GPT Safe Normalization**: AI only normalizes ingredient names, never provides nutrition data
- **Validated Database Search**: All AI suggestions verified against real USDA database
- **Mathematical Calculation Only**: Nutrition = (USDA_per_100g × grams) ÷ 100
- **Zero AI Nutrition Data**: 100% database-sourced nutrition facts

### 🔄 **10-Layer Bulletproof Flow**
1. 🚀 **Redis Cache**: Instant exact match (sub-1ms)
2. 💾 **Supabase Cache**: Persistent exact match (1-5ms)
3. 🧠 **Intelligent Parsing**: Extract ingredient + quantity + unit
4. 📏 **Deterministic Conversion**: Unit → grams using conversion tables
5. 🎯 **SQL Exact Search**: Direct USDA database lookup
6. 🔍 **Fuzzy Match**: High-confidence fuzzy matching (90%+)
7. 🤖 **GPT Normalization**: Safe ingredient term normalization only
8. 🛡️ **Validated Search**: Verify all GPT terms before use
9. 🧮 **Mathematical Calculation**: Scale nutrition to actual quantity
10. 💾 **Multi-Tier Caching**: Store for future instant access

### 🎯 **Bulletproof USDA Accuracy**
- **Anti-hallucination protection**: GPT never provides nutrition data
- Real FDC IDs from USDA FoodData Central
- **Mathematical calculations**: Handles missing calories (Protein×4 + Carbs×4 + Fat×9)
- **Smart zero-calorie detection** for ingredients like salt and water
- Verification URLs for manual checking
- **10-layer bulletproof flow** ensures 100% reliability

### ⚡ **Bulletproof Performance**
- **94%+ cache hit rate** = sub-second responses  
- **8,000+ requests/hour** throughput
- **Multi-tier caching**: Redis → Supabase → Local USDA
- **Anti-hallucination protection** with verified calculations
- **Mathematical calorie calculation** for missing nutrients

### 🔧 **Any Input Format**
- Handles "2 cups flour" or "1 lb chicken breast"
- Any ingredient description style
- Automatic quantity and measurement parsing
- No rigid formatting requirements

### 🛠️ **Developer Friendly**
- Secure credential storage with `keyring`
- Type hints and comprehensive error handling
- Works with environment variables
- Detailed documentation and examples

## 📊 Complete Nutrition Data

```python
result = av.analyze_ingredient("1 cup cooked rice")
nutrition = result.nutrition

# All available nutrients
print(f"Calories: {nutrition.calories}")
print(f"Protein: {nutrition.protein}g")
print(f"Carbs: {nutrition.carbs}g") 
print(f"Fat: {nutrition.fat}g")
print(f"Fiber: {nutrition.fiber}g")
print(f"Sugar: {nutrition.sugar}g")
print(f"Sodium: {nutrition.sodium}mg")
print(f"Calcium: {nutrition.calcium}mg")
print(f"Iron: {nutrition.iron}mg")
```

## 💰 Pricing Plans

| Plan | Monthly Calls | Price | Batch Limit | Features |
|------|---------------|-------|-------------|----------|
| **Free** | 1,000 | **Free** | 5 ingredients | Basic nutrition analysis |
| **Starter** | 5,000 | $4.99/month | 10 ingredients | Full API access, priority support |
| **Professional** | 10,000 | $8.99/month | 20 ingredients | Advanced features, production use |
| **Business** | 50,000 | $39.99/month | 50+ ingredients | High volume, custom limits available |

### Pay-As-You-Go Credits
Need more calls? Purchase credits anytime:
- **$1** = 1,000 additional calls
- **$5** = 5,000 additional calls  
- **$10** = 10,000 additional calls
- **$25** = 25,000 additional calls
- **$50** = 50,000 additional calls
- **$100** = 100,000 additional calls

Credits never expire and work with any plan!

[**Get your API key →**](https://nutrition.avocavo.app)

## 🔐 Authentication Options

### Option 1: Login (Recommended)
```python
import avocavo_nutrition as av

# Login once, use everywhere
av.login("user@example.com", "password")

# Credentials stored securely with keyring
result = av.analyze_ingredient("1 cup rice")
```

### Option 2: API Key
```python
from avocavo_nutrition import NutritionAPI

# Direct API key usage
client = NutritionAPI(api_key="your_api_key")
result = client.analyze_ingredient("1 cup rice")
```

### Option 3: Environment Variable
```bash
export AVOCAVO_API_KEY="your_api_key_here"
```

```python
import avocavo_nutrition as av
# API key automatically detected from environment
result = av.analyze_ingredient("1 cup rice")
```

## 🏗️ Real-World Examples

### Recipe App Integration
```python
import avocavo_nutrition as av

def calculate_recipe_nutrition(ingredients, servings=1):
    """Calculate nutrition for any recipe"""
    recipe = av.analyze_recipe(ingredients, servings)
    
    if recipe.success:
        return {
            'calories_per_serving': recipe.nutrition.per_serving.calories,
            'protein_per_serving': recipe.nutrition.per_serving.protein,
            'total_calories': recipe.nutrition.total.calories,
            'usda_verified_ingredients': recipe.usda_matches
        }
    else:
        return {'error': recipe.error}

# Usage
recipe_nutrition = calculate_recipe_nutrition([
    "2 cups flour",
    "1 cup milk", 
    "2 eggs"
], servings=6)
```

### Fitness Tracker Integration  
```python
def track_daily_nutrition(food_entries):
    """Track daily nutrition from food entries"""
    total_nutrition = {
        'calories': 0,
        'protein': 0,
        'carbs': 0,
        'fat': 0
    }
    
    for food in food_entries:
        result = av.analyze_ingredient(food)
        if result.success:
            total_nutrition['calories'] += result.nutrition.calories
            total_nutrition['protein'] += result.nutrition.protein
            total_nutrition['carbs'] += result.nutrition.carbs
            total_nutrition['fat'] += result.nutrition.fat
    
    return total_nutrition

# Usage
daily_foods = [
    "1 cup oatmeal",
    "1 medium banana", 
    "6 oz grilled chicken",
    "2 cups steamed broccoli"
]
daily_totals = track_daily_nutrition(daily_foods)
```

### Restaurant Menu Analysis
```python
def analyze_menu_item(ingredients):
    """Analyze nutrition for restaurant menu items"""
    # Use batch processing for efficiency
    # Batch limits: Free (5), Starter (10), Professional (20), Enterprise (50+) ingredients  
    batch = av.analyze_batch(ingredients)
    
    total_calories = sum(
        item.nutrition.calories 
        for item in batch.results 
        if item.success
    )
    
    return {
        'total_calories': total_calories,
        'success_rate': batch.success_rate,
        'ingredients_analyzed': batch.successful_matches
    }
```

## 🛠️ Advanced Usage

### Error Handling
```python
from avocavo_nutrition import ApiError, RateLimitError, AuthenticationError

try:
    result = av.analyze_ingredient("mystery ingredient")
    if result.success:
        print(f"Found: {result.usda_match.description}")
    else:
        print(f"No match: {result.error}")
        
except RateLimitError as e:
    print(f"Rate limit exceeded. Limit: {e.limit}, Usage: {e.usage}")
except AuthenticationError as e:
    print(f"Auth error: {e.message}")
except ApiError as e:
    print(f"API Error: {e.message}")
```

### Configuration
```python
# Use development environment
client = NutritionAPI(
    api_key="your_key",
    base_url="https://devapp.avocavo.app",  # Dev environment
    timeout=60  # Custom timeout
)

# Check API health
health = client.health_check()
print(f"API Status: {health['status']}")
print(f"Cache Hit Rate: {health['cache']['hit_rate']}")
```

### User Management
```python
# Check login status
if av.is_logged_in():
    user = av.get_current_user()
    print(f"Logged in as: {user['email']}")
else:
    print("Please login: av.login()")  # OAuth browser login

# Login with different provider
av.login(provider="github")  # GitHub OAuth instead of Google

# Logout
result = av.logout()
print(result['message'])  # "Successfully logged out"
```

## 🔍 What Information You Get

The Avocavo Nutrition API provides comprehensive nutrition data with USDA verification:

### Core Nutrition Facts
- **Calories** - Energy content
- **Macronutrients** - Protein, carbohydrates, total fat
- **Fiber & Sugar** - Detailed carbohydrate breakdown  
- **Minerals** - Sodium, calcium, iron
- **Fats** - Saturated fat, cholesterol

### USDA Verification
- **Real FDC IDs** from USDA FoodData Central
- **Verification URLs** for manual checking
- **Data source types** (Foundation, SR Legacy, Survey, Branded)
- **Confidence scores** for match quality

### Bulletproof System Metrics
- **Cache layer hit** - Which cache tier provided the data
- **Processing method** - Exact path through bulletproof flow
- **Anti-hallucination status** - Verification that no AI nutrition data was used
- **Mathematical verification** - Confirmation of calculated vs database nutrition
- **Response times** - Sub-second bulletproof performance tracking

### Bulletproof System Response
```python
result = av.analyze_ingredient("1 cup cooked brown rice")

# Bulletproof nutrition data (100% mathematical calculation)
print(f"Calories: {result.nutrition.calories}")           # 216.0
print(f"Protein: {result.nutrition.protein}g")            # 5.0
print(f"Carbs: {result.nutrition.carbs}g")               # 45.0
print(f"Fiber: {result.nutrition.fiber}g")               # 3.5

# USDA verification (bulletproof source)
print(f"FDC ID: {result.usda_match.fdc_id}")             # 168880
print(f"Description: {result.usda_match.description}")    # "Rice, brown, long-grain, cooked"
print(f"Data Type: {result.usda_match.data_type}")       # "SR Legacy"
print(f"Verify: {result.verification_url}")              # USDA verification link

# Bulletproof system metrics
print(f"Method: {result.metadata.method_used}")           # "sql_exact" | "fuzzy_match" | "gpt_normalized"
print(f"Cache Layer: {result.from_cache}")               # Redis | Supabase | Fresh
print(f"Anti-Hallucination: {result.metadata.verified}") # True (GPT never provided nutrition)
print(f"Processing Time: {result.processing_time_ms}ms")  # 15.2 (bulletproof speed)
print(f"Confidence: {result.confidence_score}%")          # 95 (bulletproof accuracy)
```

## 📚 Documentation

- **[Complete API Documentation](https://docs.avocavo.app)** - Full reference
- **[Get API Key](https://nutrition.avocavo.app)** - Sign up for free
- **[GitHub Repository](https://github.com/avocavo/nutrition-api-python)** - Source code
- **[Support](mailto:api-support@avocavo.com)** - Get help

## 🤝 Support

- **Email**: [api-support@avocavo.com](mailto:api-support@avocavo.com)
- **Documentation**: [docs.avocavo.app](https://docs.avocavo.app)
- **Status Page**: [status.avocavo.app](https://status.avocavo.app)
- **Feature Requests**: [GitHub Issues](https://github.com/avocavo/nutrition-api-python/issues)

## 📄 License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

---

**Made with ❤️ by the Avocavo team**

*Get started in 30 seconds: `pip install avocavo-nutrition`*