# Copyright (C) 2018-2021, earthobservations developers.
# Distributed under the MIT License. See LICENSE for more info.
from __future__ import annotations

import datetime as dt
import logging
from enum import Enum
from io import StringIO
from typing import TYPE_CHECKING
from urllib.parse import urljoin

import polars as pl

from wetterdienst.core.timeseries.request import TimeseriesRequest
from wetterdienst.core.timeseries.values import TimeseriesValues
from wetterdienst.exceptions import InvalidEnumerationError
from wetterdienst.metadata.columns import Columns
from wetterdienst.metadata.datarange import DataRange
from wetterdienst.metadata.kind import Kind
from wetterdienst.metadata.period import Period, PeriodType
from wetterdienst.metadata.provider import Provider
from wetterdienst.metadata.resolution import Resolution, ResolutionType
from wetterdienst.metadata.timezone import Timezone
from wetterdienst.metadata.unit import OriginUnit, SIUnit, UnitEnum
from wetterdienst.provider.dwd.metadata.datetime import DatetimeFormat
from wetterdienst.provider.dwd.mosmix.access import KMLReader
from wetterdienst.util.cache import CacheExpiry
from wetterdienst.util.enumeration import parse_enumeration_from_template
from wetterdienst.util.geo import convert_dm_to_dd
from wetterdienst.util.network import download_file, list_remote_files_fsspec
from wetterdienst.util.parameter import DatasetTreeCore
from wetterdienst.util.polars_util import read_fwf_from_df
from wetterdienst.util.python import to_list

if TYPE_CHECKING:
    from collections.abc import Sequence

    from wetterdienst.core.timeseries.result import StationsResult
    from wetterdienst.metadata.parameter import Parameter
    from wetterdienst.settings import Settings

try:
    from backports.datetime_fromisoformat import MonkeyPatch
except ImportError:
    pass
else:
    MonkeyPatch.patch_fromisoformat()

log = logging.getLogger(__name__)

DWD_MOSMIX_S_PATH = "weather/local_forecasts/mos/MOSMIX_S/all_stations/kml/"
DWD_MOSMIX_L_PATH = "weather/local_forecasts/mos/MOSMIX_L/all_stations/kml/"
DWD_MOSMIX_L_SINGLE_PATH = "weather/local_forecasts/mos/MOSMIX_L/single_stations/{station_id}/kml/"


class DwdMosmixPeriod(Enum):
    FUTURE = Period.FUTURE.value


class DwdMosmixDataset(Enum):
    SMALL = "small"
    LARGE = "large"


class DwdMosmixStationGroup(Enum):
    SINGLE_STATIONS = "single_stations"
    ALL_STATIONS = "all_stations"


class DwdForecastDate(Enum):
    """
    Enumeration for pointing to different mosmix dates.
    """

    LATEST = "latest"


class DwdMosmixType(Enum):
    SMALL = "SMALL"
    LARGE = "LARGE"


class DwdMosmixParameter(DatasetTreeCore):
    # https://opendata.dwd.de/weather/lib/MetElementDefinition.xml
    class SMALL(DatasetTreeCore):
        class SMALL(Enum):
            CLOUD_COVER_ABOVE_7KM = "nh"
            CLOUD_COVER_BELOW_500FT = "n05"
            CLOUD_COVER_BELOW_1000FT = "nl"
            CLOUD_COVER_BETWEEN_2KM_TO_7KM = "nm"
            CLOUD_COVER_EFFECTIVE = "neff"
            CLOUD_COVER_TOTAL = "n"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = "rr1c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = "rr3c"
            PRESSURE_AIR_SITE_REDUCED = "pppp"
            PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = "fxh25"
            PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = "fxh40"
            PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = "fxh55"
            PROBABILITY_FOG_LAST_1H = "wwm"
            PROBABILITY_FOG_LAST_6H = "wwm6"
            PROBABILITY_FOG_LAST_12H = "wwmh"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = "rh00"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = "r602"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = "rh02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = "rd02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = "rh10"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = "r650"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = "rh50"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = "rd50"
            RADIATION_GLOBAL = "rad1h"
            SUNSHINE_DURATION = "sund1"
            TEMPERATURE_AIR_MAX_2M = "tx"
            TEMPERATURE_AIR_MEAN_0_05M = "t5cm"
            TEMPERATURE_AIR_MEAN_2M = "ttt"
            TEMPERATURE_AIR_MIN_2M = "tn"
            TEMPERATURE_DEW_POINT_MEAN_2M = "td"
            VISIBILITY_RANGE = "vv"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = "rrs1c"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = "rrs3c"
            WEATHER_LAST_6H = "w1w2"
            WEATHER_SIGNIFICANT = "ww"
            WIND_DIRECTION = "dd"
            WIND_GUST_MAX_LAST_1H = "fx1"
            WIND_GUST_MAX_LAST_3H = "fx3"
            WIND_GUST_MAX_LAST_12H = "fxh"
            WIND_SPEED = "ff"

        CLOUD_COVER_ABOVE_7KM = SMALL.CLOUD_COVER_ABOVE_7KM
        CLOUD_COVER_BELOW_500FT = SMALL.CLOUD_COVER_BELOW_500FT
        CLOUD_COVER_BELOW_1000FT = SMALL.CLOUD_COVER_BELOW_1000FT
        CLOUD_COVER_BETWEEN_2KM_TO_7KM = SMALL.CLOUD_COVER_BETWEEN_2KM_TO_7KM
        CLOUD_COVER_EFFECTIVE = SMALL.CLOUD_COVER_EFFECTIVE
        CLOUD_COVER_TOTAL = SMALL.CLOUD_COVER_TOTAL
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = SMALL.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = SMALL.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H
        PRESSURE_AIR_SITE_REDUCED = SMALL.PRESSURE_AIR_SITE_REDUCED
        PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = SMALL.PROBABILITY_WIND_GUST_GE_25KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = SMALL.PROBABILITY_WIND_GUST_GE_40KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = SMALL.PROBABILITY_WIND_GUST_GE_55KN_LAST_12H
        PROBABILITY_FOG_LAST_1H = SMALL.PROBABILITY_FOG_LAST_1H
        PROBABILITY_FOG_LAST_6H = SMALL.PROBABILITY_FOG_LAST_6H
        PROBABILITY_FOG_LAST_12H = SMALL.PROBABILITY_FOG_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = SMALL.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H
        RADIATION_GLOBAL = SMALL.RADIATION_GLOBAL
        SUNSHINE_DURATION = SMALL.SUNSHINE_DURATION
        TEMPERATURE_AIR_MAX_2M = SMALL.TEMPERATURE_AIR_MAX_2M
        TEMPERATURE_AIR_MEAN_0_05M = SMALL.TEMPERATURE_AIR_MEAN_0_05M
        TEMPERATURE_AIR_MEAN_2M = SMALL.TEMPERATURE_AIR_MEAN_2M
        TEMPERATURE_AIR_MIN_2M = SMALL.TEMPERATURE_AIR_MIN_2M
        TEMPERATURE_DEW_POINT_MEAN_2M = SMALL.TEMPERATURE_DEW_POINT_MEAN_2M
        VISIBILITY_RANGE = SMALL.VISIBILITY_RANGE
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = SMALL.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = SMALL.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H
        WEATHER_LAST_6H = SMALL.WEATHER_LAST_6H
        WEATHER_SIGNIFICANT = SMALL.WEATHER_SIGNIFICANT
        WIND_DIRECTION = SMALL.WIND_DIRECTION
        WIND_GUST_MAX_LAST_1H = SMALL.WIND_GUST_MAX_LAST_1H
        WIND_GUST_MAX_LAST_3H = SMALL.WIND_GUST_MAX_LAST_3H
        WIND_GUST_MAX_LAST_12H = SMALL.WIND_GUST_MAX_LAST_12H
        WIND_SPEED = SMALL.WIND_SPEED

    class LARGE(DatasetTreeCore):
        class LARGE(Enum):
            CLOUD_BASE_CONVECTIVE = "h_bsc"
            CLOUD_COVER_ABOVE_7KM = "nh"
            CLOUD_COVER_BELOW_500FT = "n05"
            CLOUD_COVER_BELOW_1000FT = "nl"
            CLOUD_COVER_BELOW_7KM = "nlm"
            CLOUD_COVER_BETWEEN_2KM_TO_7KM = "nm"
            CLOUD_COVER_EFFECTIVE = "neff"
            CLOUD_COVER_TOTAL = "n"
            ERROR_ABSOLUTE_PRESSURE_AIR_SITE = "e_ppp"
            ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_2M = "e_ttt"
            ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_2M = "e_td"
            ERROR_ABSOLUTE_WIND_DIRECTION = "e_dd"
            ERROR_ABSOLUTE_WIND_SPEED = "e_ff"
            EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = "pevap"
            PRECIPITATION_DURATION = "drr1"
            PRECIPITATION_HEIGHT_LAST_1H = "rr1"
            PRECIPITATION_HEIGHT_LAST_3H = "rr3"
            PRECIPITATION_HEIGHT_LAST_6H = "rr6"
            PRECIPITATION_HEIGHT_LAST_12H = "rrh"
            PRECIPITATION_HEIGHT_LAST_24H = "rrd"
            PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = "rrl1c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = "rr1c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = "rr3c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = "rr6c"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = "rrhc"
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = "rrdc"
            PRESSURE_AIR_SITE_REDUCED = "pppp"
            PROBABILITY_DRIZZLE_LAST_1H = "wwz"
            PROBABILITY_DRIZZLE_LAST_6H = "wwz6"
            PROBABILITY_DRIZZLE_LAST_12H = "wwzh"
            PROBABILITY_FOG_LAST_1H = "wwm"
            PROBABILITY_FOG_LAST_6H = "wwm6"
            PROBABILITY_FOG_LAST_12H = "wwmh"
            PROBABILITY_FOG_LAST_24H = "wwmd"
            PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = "wwf"
            PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = "wwf6"
            PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = "wwfh"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = "rh00"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = "r602"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = "rh02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = "rd02"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = "rh10"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = "r650"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = "rh50"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = "rd50"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_6H = "r600"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_24H = "rd00"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1MM_LAST_1H = "r101"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_1H = "r102"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3MM_LAST_1H = "r103"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5MM_LAST_1H = "r105"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7MM_LAST_1H = "r107"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_1H = "r110"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_6H = "r610"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_24H = "rd10"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_2MM_LAST_1H = "r120"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_3MM_LAST_1H = "r130"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_1H = "r150"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_10MM_LAST_1H = "rr1o1"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_15MM_LAST_1H = "rr1w1"
            PROBABILITY_PRECIPITATION_HEIGHT_GT_25MM_LAST_1H = "rr1u1"
            PROBABILITY_PRECIPITATION_LAST_1H = "wwp"
            PROBABILITY_PRECIPITATION_LAST_6H = "wwp6"
            PROBABILITY_PRECIPITATION_LAST_12H = "wwph"
            PROBABILITY_PRECIPITATION_LAST_24H = "wwpd"
            PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = "wwl"
            PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = "wwl6"
            PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = "wwlh"
            PROBABILITY_PRECIPITATION_SOLID_LAST_1H = "wws"
            PROBABILITY_PRECIPITATION_SOLID_LAST_6H = "wws6"
            PROBABILITY_PRECIPITATION_SOLID_LAST_12H = "wwsh"
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = "wwd"
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = "wwd6"
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = "wwdh"
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = "wwc"
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = "wwc6"
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = "wwch"
            PROBABILITY_RADIATION_GLOBAL_LAST_1H = "rrad1"
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0PCT_LAST_24H = "psd00"
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30PCT_LAST_24H = "psd30"
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60PCT_LAST_24H = "psd60"
            PROBABILITY_THUNDER_LAST_1H = "wwt"
            PROBABILITY_THUNDER_LAST_6H = "wwt6"
            PROBABILITY_THUNDER_LAST_12H = "wwth"
            PROBABILITY_THUNDER_LAST_24H = "wwtd"
            PROBABILITY_VISIBILITY_BELOW_1000M = "vv10"
            PROBABILITY_WIND_GUST_GE_25KN_LAST_6H = "fx625"
            PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = "fxh25"
            PROBABILITY_WIND_GUST_GE_40KN_LAST_6H = "fx640"
            PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = "fxh40"
            PROBABILITY_WIND_GUST_GE_55KN_LAST_6H = "fx655"
            PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = "fxh55"
            RADIATION_GLOBAL = "rad1h"
            RADIATION_GLOBAL_LAST_3H = "rads3"
            RADIATION_SKY_LONG_WAVE_LAST_3H = "radl3"
            SUNSHINE_DURATION = "sund1"
            SUNSHINE_DURATION_LAST_3H = "sund3"
            SUNSHINE_DURATION_RELATIVE_LAST_24H = "rsund"
            SUNSHINE_DURATION_YESTERDAY = "sund"
            TEMPERATURE_AIR_MAX_2M = "tx"
            TEMPERATURE_AIR_MEAN_0_05M = "t5cm"
            TEMPERATURE_AIR_MEAN_2M = "ttt"
            TEMPERATURE_AIR_MEAN_2M_LAST_24H = "tm"
            TEMPERATURE_AIR_MIN_0_05M_LAST_12H = "tg"
            TEMPERATURE_AIR_MIN_2M = "tn"
            TEMPERATURE_DEW_POINT_MEAN_2M = "td"
            VISIBILITY_RANGE = "vv"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = "rrs1c"
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = "rrs3c"
            WEATHER_LAST_6H = "w1w2"
            WEATHER_SIGNIFICANT = "ww"
            WEATHER_SIGNIFICANT_LAST_3H = "ww3"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = "wpc11"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = "wpc31"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = "wpc61"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = "wpch1"
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = "wpcd1"
            WIND_DIRECTION = "dd"
            WIND_GUST_MAX_LAST_1H = "fx1"
            WIND_GUST_MAX_LAST_3H = "fx3"
            WIND_GUST_MAX_LAST_12H = "fxh"
            WIND_SPEED = "ff"

        CLOUD_BASE_CONVECTIVE = LARGE.CLOUD_BASE_CONVECTIVE
        CLOUD_COVER_ABOVE_7KM = LARGE.CLOUD_COVER_ABOVE_7KM
        CLOUD_COVER_BELOW_500FT = LARGE.CLOUD_COVER_BELOW_500FT
        CLOUD_COVER_BELOW_1000FT = LARGE.CLOUD_COVER_BELOW_1000FT
        CLOUD_COVER_BELOW_7KM = LARGE.CLOUD_COVER_BELOW_7KM
        CLOUD_COVER_BETWEEN_2KM_TO_7KM = LARGE.CLOUD_COVER_BETWEEN_2KM_TO_7KM
        CLOUD_COVER_EFFECTIVE = LARGE.CLOUD_COVER_EFFECTIVE
        CLOUD_COVER_TOTAL = LARGE.CLOUD_COVER_TOTAL
        ERROR_ABSOLUTE_PRESSURE_AIR_SITE = LARGE.ERROR_ABSOLUTE_PRESSURE_AIR_SITE
        ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_2M = LARGE.ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_2M
        ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_2M = LARGE.ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_2M
        ERROR_ABSOLUTE_WIND_DIRECTION = LARGE.ERROR_ABSOLUTE_WIND_DIRECTION
        ERROR_ABSOLUTE_WIND_SPEED = LARGE.ERROR_ABSOLUTE_WIND_SPEED
        EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = LARGE.EVAPOTRANSPIRATION_POTENTIAL_LAST_24H
        PRECIPITATION_DURATION = LARGE.PRECIPITATION_DURATION
        PRECIPITATION_HEIGHT_LAST_1H = LARGE.PRECIPITATION_HEIGHT_LAST_1H
        PRECIPITATION_HEIGHT_LAST_3H = LARGE.PRECIPITATION_HEIGHT_LAST_3H
        PRECIPITATION_HEIGHT_LAST_6H = LARGE.PRECIPITATION_HEIGHT_LAST_6H
        PRECIPITATION_HEIGHT_LAST_12H = LARGE.PRECIPITATION_HEIGHT_LAST_12H
        PRECIPITATION_HEIGHT_LAST_24H = LARGE.PRECIPITATION_HEIGHT_LAST_24H
        PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = (
            LARGE.PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H
        )
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = LARGE.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = LARGE.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = LARGE.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = LARGE.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H
        PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = LARGE.PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H
        PRESSURE_AIR_SITE_REDUCED = LARGE.PRESSURE_AIR_SITE_REDUCED
        PROBABILITY_DRIZZLE_LAST_1H = LARGE.PROBABILITY_DRIZZLE_LAST_1H
        PROBABILITY_DRIZZLE_LAST_6H = LARGE.PROBABILITY_DRIZZLE_LAST_6H
        PROBABILITY_DRIZZLE_LAST_12H = LARGE.PROBABILITY_DRIZZLE_LAST_12H
        PROBABILITY_FOG_LAST_1H = LARGE.PROBABILITY_FOG_LAST_1H
        PROBABILITY_FOG_LAST_6H = LARGE.PROBABILITY_FOG_LAST_6H
        PROBABILITY_FOG_LAST_12H = LARGE.PROBABILITY_FOG_LAST_12H
        PROBABILITY_FOG_LAST_24H = LARGE.PROBABILITY_FOG_LAST_24H
        PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_FREEZING_LAST_1H
        PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_FREEZING_LAST_6H
        PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_FREEZING_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_24H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_6H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_24H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_24H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_2MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_2MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_3MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_3MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_10MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_10MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_15MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_15MM_LAST_1H
        PROBABILITY_PRECIPITATION_HEIGHT_GT_25MM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_HEIGHT_GT_25MM_LAST_1H
        PROBABILITY_PRECIPITATION_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_LAST_1H
        PROBABILITY_PRECIPITATION_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_LAST_6H
        PROBABILITY_PRECIPITATION_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_LAST_12H
        PROBABILITY_PRECIPITATION_LAST_24H = LARGE.PROBABILITY_PRECIPITATION_LAST_24H
        PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_LIQUID_LAST_1H
        PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_LIQUID_LAST_6H
        PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_LIQUID_LAST_12H
        PROBABILITY_PRECIPITATION_SOLID_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_SOLID_LAST_1H
        PROBABILITY_PRECIPITATION_SOLID_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_SOLID_LAST_6H
        PROBABILITY_PRECIPITATION_SOLID_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_SOLID_LAST_12H
        PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H
        PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H
        PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H
        PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = LARGE.PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H
        PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = LARGE.PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H
        PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = LARGE.PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H
        PROBABILITY_RADIATION_GLOBAL_LAST_1H = LARGE.PROBABILITY_RADIATION_GLOBAL_LAST_1H
        PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0PCT_LAST_24H = (
            LARGE.PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0PCT_LAST_24H
        )
        PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30PCT_LAST_24H = (
            LARGE.PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30PCT_LAST_24H
        )
        PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60PCT_LAST_24H = (
            LARGE.PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60PCT_LAST_24H
        )
        PROBABILITY_THUNDER_LAST_1H = LARGE.PROBABILITY_THUNDER_LAST_1H
        PROBABILITY_THUNDER_LAST_6H = LARGE.PROBABILITY_THUNDER_LAST_6H
        PROBABILITY_THUNDER_LAST_12H = LARGE.PROBABILITY_THUNDER_LAST_12H
        PROBABILITY_THUNDER_LAST_24H = LARGE.PROBABILITY_THUNDER_LAST_24H
        PROBABILITY_VISIBILITY_BELOW_1000M = LARGE.PROBABILITY_VISIBILITY_BELOW_1000M
        PROBABILITY_WIND_GUST_GE_25KN_LAST_6H = LARGE.PROBABILITY_WIND_GUST_GE_25KN_LAST_6H
        PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = LARGE.PROBABILITY_WIND_GUST_GE_25KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_40KN_LAST_6H = LARGE.PROBABILITY_WIND_GUST_GE_40KN_LAST_6H
        PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = LARGE.PROBABILITY_WIND_GUST_GE_40KN_LAST_12H
        PROBABILITY_WIND_GUST_GE_55KN_LAST_6H = LARGE.PROBABILITY_WIND_GUST_GE_55KN_LAST_6H
        PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = LARGE.PROBABILITY_WIND_GUST_GE_55KN_LAST_12H
        RADIATION_GLOBAL = LARGE.RADIATION_GLOBAL
        RADIATION_GLOBAL_LAST_3H = LARGE.RADIATION_GLOBAL_LAST_3H
        RADIATION_SKY_LONG_WAVE_LAST_3H = LARGE.RADIATION_SKY_LONG_WAVE_LAST_3H
        SUNSHINE_DURATION = LARGE.SUNSHINE_DURATION
        SUNSHINE_DURATION_LAST_3H = LARGE.SUNSHINE_DURATION_LAST_3H
        SUNSHINE_DURATION_RELATIVE_LAST_24H = LARGE.SUNSHINE_DURATION_RELATIVE_LAST_24H
        SUNSHINE_DURATION_YESTERDAY = LARGE.SUNSHINE_DURATION_YESTERDAY
        TEMPERATURE_AIR_MAX_2M = LARGE.TEMPERATURE_AIR_MAX_2M
        TEMPERATURE_AIR_MEAN_0_05M = LARGE.TEMPERATURE_AIR_MEAN_0_05M
        TEMPERATURE_AIR_MEAN_2M = LARGE.TEMPERATURE_AIR_MEAN_2M
        TEMPERATURE_AIR_MEAN_2M_LAST_24H = LARGE.TEMPERATURE_AIR_MEAN_2M_LAST_24H
        TEMPERATURE_AIR_MIN_0_05M_LAST_12H = LARGE.TEMPERATURE_AIR_MIN_0_05M_LAST_12H
        TEMPERATURE_AIR_MIN_2M = LARGE.TEMPERATURE_AIR_MIN_2M
        TEMPERATURE_DEW_POINT_MEAN_2M = LARGE.TEMPERATURE_DEW_POINT_MEAN_2M
        VISIBILITY_RANGE = LARGE.VISIBILITY_RANGE
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = LARGE.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H
        WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = LARGE.WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H
        WEATHER_LAST_6H = LARGE.WEATHER_LAST_6H
        WEATHER_SIGNIFICANT = LARGE.WEATHER_SIGNIFICANT
        WEATHER_SIGNIFICANT_LAST_3H = LARGE.WEATHER_SIGNIFICANT_LAST_3H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = LARGE.WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = LARGE.WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = LARGE.WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = LARGE.WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H
        WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = LARGE.WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H
        WIND_DIRECTION = LARGE.WIND_DIRECTION
        WIND_GUST_MAX_LAST_1H = LARGE.WIND_GUST_MAX_LAST_1H
        WIND_GUST_MAX_LAST_3H = LARGE.WIND_GUST_MAX_LAST_3H
        WIND_GUST_MAX_LAST_12H = LARGE.WIND_GUST_MAX_LAST_12H
        WIND_SPEED = LARGE.WIND_SPEED


class DwdMosmixUnit(DatasetTreeCore):
    class SMALL(DatasetTreeCore):
        class SMALL(UnitEnum):
            TEMPERATURE_AIR_MEAN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_DEW_POINT_MEAN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MAX_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MIN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            WIND_DIRECTION = (
                OriginUnit.WIND_DIRECTION.value,
                SIUnit.WIND_DIRECTION.value,
            )
            WIND_SPEED = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_1H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_3H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_12H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WEATHER_SIGNIFICANT = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_LAST_6H = OriginUnit.DIMENSIONLESS.value, SIUnit.DIMENSIONLESS.value
            CLOUD_COVER_TOTAL = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_EFFECTIVE = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_500FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_1000FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BETWEEN_2KM_TO_7KM = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            CLOUD_COVER_ABOVE_7KM = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PRESSURE_AIR_SITE_REDUCED = OriginUnit.PASCAL.value, SIUnit.PASCAL.value
            TEMPERATURE_AIR_MEAN_0_05M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            RADIATION_GLOBAL = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            VISIBILITY_RANGE = OriginUnit.METER.value, SIUnit.METER.value
            SUNSHINE_DURATION = OriginUnit.SECOND.value, SIUnit.SECOND.value
            PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_FOG_LAST_1H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_6H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_12H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )

    class LARGE(DatasetTreeCore):
        class LARGE(UnitEnum):
            # https://opendata.dwd.de/weather/lib/MetElementDefinition.xml
            TEMPERATURE_AIR_MEAN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_DEW_POINT_MEAN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MAX_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MIN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            WIND_DIRECTION = (
                OriginUnit.WIND_DIRECTION.value,
                SIUnit.WIND_DIRECTION.value,
            )
            WIND_SPEED = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_1H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_3H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            WIND_GUST_MAX_LAST_12H = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WEATHER_SIGNIFICANT = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_LAST_6H = OriginUnit.DIMENSIONLESS.value, SIUnit.DIMENSIONLESS.value
            CLOUD_COVER_TOTAL = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_EFFECTIVE = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_500FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BELOW_1000FT = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            CLOUD_COVER_BETWEEN_2KM_TO_7KM = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            CLOUD_COVER_ABOVE_7KM = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PRESSURE_AIR_SITE_REDUCED = OriginUnit.PASCAL.value, SIUnit.PASCAL.value
            TEMPERATURE_AIR_MEAN_0_05M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            RADIATION_GLOBAL_LAST_3H = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            RADIATION_GLOBAL = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            RADIATION_SKY_LONG_WAVE_LAST_3H = (
                OriginUnit.KILOJOULE_PER_SQUARE_METER.value,
                SIUnit.JOULE_PER_SQUARE_METER.value,
            )
            VISIBILITY_RANGE = OriginUnit.METER.value, SIUnit.METER.value
            SUNSHINE_DURATION = OriginUnit.SECOND.value, SIUnit.SECOND.value
            PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_FOG_LAST_1H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_6H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_12H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            TEMPERATURE_AIR_MIN_0_05M_LAST_12H = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            TEMPERATURE_AIR_MEAN_2M_LAST_24H = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            PRECIPITATION_DURATION = OriginUnit.SECOND.value, SIUnit.SECOND.value
            PROBABILITY_DRIZZLE_LAST_1H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_THUNDER_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_SOLID_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_VISIBILITY_BELOW_1000M = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            ERROR_ABSOLUTE_WIND_SPEED = (
                OriginUnit.METER_PER_SECOND.value,
                SIUnit.METER_PER_SECOND.value,
            )
            ERROR_ABSOLUTE_WIND_DIRECTION = (
                OriginUnit.WIND_DIRECTION.value,
                SIUnit.WIND_DIRECTION.value,
            )
            ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_2M = (
                OriginUnit.DEGREE_KELVIN.value,
                SIUnit.DEGREE_KELVIN.value,
            )
            PRECIPITATION_HEIGHT_LAST_6H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_2MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            SUNSHINE_DURATION_YESTERDAY = OriginUnit.SECOND.value, SIUnit.SECOND.value
            SUNSHINE_DURATION_RELATIVE_LAST_24H = (
                OriginUnit.DIMENSIONLESS.value,
                SIUnit.DIMENSIONLESS.value,
            )
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0PCT_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30PCT_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60PCT_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_RADIATION_GLOBAL_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_3MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_10MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_15MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_25MM_LAST_1H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_THUNDER_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_SOLID_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_DRIZZLE_LAST_6H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_FOG_LAST_24H = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_WIND_GUST_GE_25KN_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_40KN_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_WIND_GUST_GE_55KN_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_THUNDER_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_SOLID_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_DRIZZLE_LAST_12H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_6H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PRECIPITATION_HEIGHT_LAST_12H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            WEATHER_SIGNIFICANT_LAST_3H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            PRECIPITATION_HEIGHT_LAST_24H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = (
                OriginUnit.KILOGRAM_PER_SQUARE_METER.value,
                SIUnit.KILOGRAM_PER_SQUARE_METER.value,
            )
            CLOUD_COVER_BELOW_7KM = OriginUnit.PERCENT.value, SIUnit.PERCENT.value
            PROBABILITY_PRECIPITATION_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            CLOUD_BASE_CONVECTIVE = OriginUnit.METER.value, SIUnit.METER.value
            PROBABILITY_THUNDER_LAST_24H = (
                OriginUnit.PERCENT.value,
                SIUnit.PERCENT.value,
            )
            ERROR_ABSOLUTE_PRESSURE_AIR_SITE = (
                OriginUnit.PASCAL.value,
                SIUnit.PASCAL.value,
            )
            SUNSHINE_DURATION_LAST_3H = OriginUnit.SECOND.value, SIUnit.SECOND.value
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )
            WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = (
                OriginUnit.SIGNIFICANT_WEATHER.value,
                SIUnit.SIGNIFICANT_WEATHER.value,
            )


class DwdMosmixValues(TimeseriesValues):
    """
    Fetch weather mosmix data (KML/MOSMIX_S dataset).

    Parameters
    ----------
    station_id : List
        - If None, data for all stations_result is returned.
        - If not None, station_ids are a list of station ids for which data is desired.

    parameter: List
        - If None, data for all parameters is returned.
        - If not None, list of parameters, per MOSMIX definition, see
          https://www.dwd.de/DE/leistungen/opendata/help/schluessel_datenformate/kml/mosmix_elemente_pdf.pdf?__blob=publicationFile&v=2
    """  # noqa:B950,E501

    _tz = Timezone.GERMANY
    _data_tz = Timezone.UTC

    def _create_humanized_parameters_mapping(self) -> dict[str, str]:
        """
        Method for creation of parameter name mappings based on
        self._parameter_base

        :return:
        """
        return {
            parameter.value: parameter.name.lower()
            for parameter in self.sr.stations._parameter_base[self.sr.stations.mosmix_type.name][
                self.sr.stations.mosmix_type.name
            ]
        }

    def __init__(self, stations_result: StationsResult) -> None:
        """

        :param stations_result:
        """
        super().__init__(stations_result=stations_result)

        self.kml = KMLReader(
            station_ids=self.sr.station_id.to_list(),
            settings=self.sr.stations.settings,
        )

    @property
    def metadata(self) -> pl.DataFrame:
        """
        Wrapper for mosmix metadata

        :return:
        """
        return self.sr.df

    def _collect_station_parameter(self, station_id: str, parameter: str, dataset: str) -> pl.DataFrame:  # noqa: ARG002
        """
        Wrapper of read_mosmix to collect mosmix data (either latest or for
        defined dates)

        :return: pandas DataFrame with data corresponding to station id and parameter
        """
        df = self.read_mosmix(station_id, self.sr.stations.issue)
        if df.is_empty():
            return df
        df = df.unpivot(
            index=[
                Columns.DATE.value,
            ],
            variable_name=Columns.PARAMETER.value,
            value_name=Columns.VALUE.value,
        )
        return df.with_columns(
            pl.lit(station_id, dtype=pl.String).alias(Columns.STATION_ID.value),
            pl.lit(value=None, dtype=pl.Float64).alias(Columns.QUALITY.value),
        )

    def read_mosmix(self, station_id: str, date: dt.datetime | DwdForecastDate) -> pl.DataFrame:
        """
        Manage data acquisition for a given date that is used to filter the found files
        on the MOSMIX path of the DWD server.

        :param date: datetime or enumeration for latest MOSMIX mosmix
        :return: pandas DataFrame with gathered information
        """
        if self.sr.stations.mosmix_type == DwdMosmixType.SMALL:
            return self.read_mosmix_small(station_id, date)
        else:
            return self.read_mosmix_large(station_id, date)

    def read_mosmix_small(self, station_id, date: DwdForecastDate | dt.datetime) -> pl.DataFrame:
        """Reads single MOSMIX-S file for all stations."""
        url = urljoin("https://opendata.dwd.de", DWD_MOSMIX_S_PATH)
        file_url = self.get_url_for_date(url, date)
        self.kml.read(file_url)
        return self.kml.get_station_forecast(station_id)

    def read_mosmix_large(
        self,
        station_id: str,
        date: DwdForecastDate | dt.datetime,
    ) -> pl.DataFrame:
        """Reads either a large MOSMIX-L file with all stations or a small MOSMIX-L file for a single station."""
        if self.sr.stations.station_group == DwdMosmixStationGroup.ALL_STATIONS:
            url = urljoin("https://opendata.dwd.de", DWD_MOSMIX_L_PATH)
        else:
            url = urljoin("https://opendata.dwd.de", DWD_MOSMIX_L_SINGLE_PATH).format(station_id=station_id)
        file_url = self.get_url_for_date(url, date)
        self.kml.read(file_url)
        return self.kml.get_station_forecast(station_id)

    def get_url_for_date(self, url: str, date: dt.datetime | DwdForecastDate) -> str:
        """
        Method to get a file url based on the MOSMIX-S/MOSMIX-L url and the date that is
        used for filtering.

        :param url: MOSMIX-S/MOSMIX-L path on the dwd server
        :param date: date used for filtering of the available files
        :return: file url based on the filtering
        """
        urls = list_remote_files_fsspec(url, self.sr.stations.settings, CacheExpiry.NO_CACHE)

        if date == DwdForecastDate.LATEST:
            try:
                return list(filter(lambda url_: "LATEST" in url_.upper(), urls))[0]
            except IndexError as e:
                raise IndexError(f"Unable to find LATEST file within {url}") from e

        date = date.astimezone(dt.timezone.utc).replace(tzinfo=None)

        df = pl.DataFrame({"url": urls})

        df = df.with_columns(
            pl.col("url").str.split("/").list.last().str.split("_").list.gather(2).flatten().alias("date"),
        )

        df = df.filter(pl.col("date").ne("LATEST"))

        df = df.with_columns(
            pl.col("date")
            .map_elements(lambda d: f"{d}00", return_dtype=pl.String)
            .str.to_datetime(DatetimeFormat.YMDHM.value),
        )

        df = df.filter(pl.col("date").eq(date))

        if df.is_empty():
            raise IndexError(f"Unable to find {date} file within {url}")

        return df.get_column("url").item()


class DwdMosmixRequest(TimeseriesRequest):
    """Implementation of sites for MOSMIX mosmix sites"""

    _provider = Provider.DWD
    _kind = Kind.FORECAST
    _tz = Timezone.GERMANY
    _dataset_base = DwdMosmixDataset
    _parameter_base = DwdMosmixParameter
    _unit_base = DwdMosmixUnit
    _resolution_type = ResolutionType.FIXED
    _resolution_base = Resolution  # use general Resolution for fixed Resolution
    _period_type = PeriodType.FIXED
    _period_base = DwdMosmixPeriod
    _has_datasets = True
    _unique_dataset = True
    _data_range = DataRange.FIXED
    _values = DwdMosmixValues
    _url = "https://www.dwd.de/DE/leistungen/met_verfahren_mosmix/mosmix_stationskatalog.cfg?view=nasPublication"

    @property
    def _dataset_accessor(self) -> str:
        """
        Implementation for tidying mosmix data

        :return:
        """
        return self.mosmix_type.name

    @classmethod
    def _setup_resolution_filter(cls, resolution):
        """
        Use SMALL and LARGE instead of resolution, which is fixed for Mosmix

        :param resolution:
        :return:
        """
        return (
            [parse_enumeration_from_template(res, intermediate=cls._dataset_base) for res in to_list(resolution)]
            if resolution
            else [*cls._dataset_base]
        )

    _base_columns = [
        Columns.STATION_ID.value,
        Columns.ICAO_ID.value,
        Columns.START_DATE.value,
        Columns.END_DATE.value,
        Columns.LATITUDE.value,
        Columns.LONGITUDE.value,
        Columns.HEIGHT.value,
        Columns.NAME.value,
        Columns.STATE.value,
    ]

    @staticmethod
    def adjust_datetime(datetime_: dt.datetime) -> dt.datetime:
        """
        Adjust datetime to MOSMIX release frequency, which is required for MOSMIX-L
        that is only released very 6 hours (3, 9, 15, 21). Datetime is floored
        to closest release time e.g. if hour is 14, it will be rounded to 9

        :param datetime_: datetime that is adjusted
        :return: adjusted datetime with floored hour
        """
        regular_date = dt.datetime.fromordinal(datetime_.date().toordinal()).replace(hour=3)

        if regular_date > datetime_:
            regular_date -= dt.timedelta(hours=6)

        delta_hours = (datetime_.hour - regular_date.hour) % 6

        return datetime_ - dt.timedelta(hours=delta_hours)

    def __init__(
        self,
        parameter: str | DwdMosmixParameter | Parameter | Sequence[str | DwdMosmixParameter | Parameter],
        mosmix_type: str | DwdMosmixType,
        start_date: str | dt.datetime | None = None,
        end_date: str | dt.datetime | None = None,
        issue: str | dt.datetime | DwdForecastDate | None = DwdForecastDate.LATEST,
        station_group: DwdMosmixStationGroup | None = None,
        settings: Settings | None = None,
    ) -> None:
        """

        :param parameter: parameter(s) to be collected
        :param mosmix_type: mosmix type, either small or large
        :param start_date: start date for filtering returned dataframe
        :param end_date: end date
        :param issue: start of issue of mosmix which should be caught (Mosmix run at time XX:YY)
        """
        self.mosmix_type = parse_enumeration_from_template(mosmix_type, DwdMosmixType)

        if self.mosmix_type == DwdMosmixType.SMALL:
            self.station_group = DwdMosmixStationGroup.ALL_STATIONS
        else:
            self.station_group = (
                parse_enumeration_from_template(station_group, DwdMosmixStationGroup)
                or DwdMosmixStationGroup.SINGLE_STATIONS
            )

        super().__init__(
            parameter=parameter,
            start_date=start_date,
            end_date=end_date,
            resolution=Resolution.HOURLY,
            period=Period.FUTURE,
            settings=settings,
        )

        if not issue:
            issue = DwdForecastDate.LATEST

        try:
            issue = parse_enumeration_from_template(issue, DwdForecastDate)
        except InvalidEnumerationError:
            pass

        if issue is not DwdForecastDate.LATEST:
            if isinstance(issue, str):
                issue = dt.datetime.fromisoformat(issue)
            issue = dt.datetime(issue.year, issue.month, issue.day, issue.hour, tzinfo=issue.tzinfo)
            # Shift issue date to 3, 9, 15, 21 hour format
            if mosmix_type == DwdMosmixType.LARGE:
                issue = self.adjust_datetime(issue)

        # TODO: this should be replaced by the freq property in the main class
        if self.mosmix_type == DwdMosmixType.SMALL:
            self.resolution = Resolution.HOURLY
        else:
            self.resolution = Resolution.HOUR_6

        self.issue = issue

    def _all(self) -> pl.LazyFrame:
        """
        Create meta data DataFrame from available station list

        :return:
        """
        log.info(f"Downloading file {self._url}.")
        payload = download_file(self._url, self.settings, CacheExpiry.METAINDEX)
        text = StringIO(payload.read().decode(encoding="latin-1"))
        lines = text.readlines()
        header = lines.pop(0)
        df = pl.DataFrame({"column_0": lines[1:]})
        df.columns = [header]
        column_specs = ((0, 5), (6, 9), (11, 30), (32, 38), (39, 46), (48, 56))
        df = read_fwf_from_df(df, column_specs)
        df.columns = [
            Columns.STATION_ID.value,
            Columns.ICAO_ID.value,
            Columns.NAME.value,
            Columns.LATITUDE.value,
            Columns.LONGITUDE.value,
            Columns.HEIGHT.value,
        ]
        df = df.select(
            pl.col(Columns.STATION_ID.value),
            pl.col(Columns.ICAO_ID.value).replace("----", None),
            pl.lit(None, pl.Datetime(time_zone="UTC")).alias(Columns.START_DATE.value),
            pl.lit(None, pl.Datetime(time_zone="UTC")).alias(Columns.END_DATE.value),
            pl.col(Columns.LATITUDE.value).cast(float).map_batches(convert_dm_to_dd),
            pl.col(Columns.LONGITUDE.value).cast(float).map_batches(convert_dm_to_dd),
            pl.col(Columns.HEIGHT.value).cast(int),
            pl.col(Columns.NAME.value),
            pl.lit(None, pl.String).alias(Columns.STATE.value),
        )
        return df.lazy()
